"""Get a LabVIEW Window netlist from the exported Vivado project."""

# Copyright (c) 2025 National Instruments Corporation
#
# SPDX-License-Identifier: MIT
#


import os
import shutil

from . import common


def _get_window_netlist(config):
    """Gets the Window netlist from the Vivado Project as well as other HDL Files."""
    get_netlist_tcl_path = os.path.join(os.getcwd(), "TCL/GetWindowNetlist.tcl")
    current_dir = os.getcwd()

    # Extract project directory and name from the XPR path
    vivado_project_path = os.path.dirname(config.vivado_project_export_xpr)
    project_name = os.path.splitext(os.path.basename(config.vivado_project_export_xpr))[0]

    print(f"Vivado project path: {vivado_project_path}")
    print(f"Vivado project name: {project_name}")

    os.chdir(vivado_project_path)

    # Use the vivado_tools_path from the config instead of the XILINX environment variable
    vivado_path = config.vivado_tools_path

    if vivado_path:
        # Determine the Vivado executable based on the operating system
        if os.name == "nt":  # Windows
            vivado_executable = os.path.join(vivado_path, "bin", "vivado.bat")
        else:  # Linux or other OS
            vivado_executable = os.path.join(vivado_path, "bin", "vivado")

        print(f"Current working directory: {os.getcwd()}")
        common.run_command(
            f'"{vivado_executable}" {project_name}.xpr -mode batch -source {get_netlist_tcl_path}',
            cwd=os.getcwd(),
            capture_output=False,
        )

        # Copy TheWindow.edf to destination folder
        source_file = os.path.join(vivado_project_path, "TheWindow.edf")
        destination_folder = config.the_window_folder_output

        # Create destination directory if it doesn't exist
        os.makedirs(destination_folder, exist_ok=True)

        destination_file = os.path.join(destination_folder, "TheWindow.edf")

        try:
            if os.path.exists(source_file):
                shutil.copy(source_file, destination_file)
                print(f"Copied {source_file} to {destination_file}")
            else:
                print(f"Error: Source file {source_file} not found")
        except Exception as e:
            print(f"Error copying file: {str(e)}")

        os.chdir(current_dir)
    else:
        print("Vivado tools path not found in configuration.")


def _copy_lv_generated_files(config):
    """Copy files that are generated by LV into TheWindow folder for use in the HDL vivado flow."""
    # Extract the parent directory path from the XPR path
    # The NIProtectedFiles folder is typically at the same level as VivadoProject
    project_export_base = os.path.dirname(os.path.dirname(config.vivado_project_export_xpr))
    protected_files_folder = os.path.join(project_export_base, "NIProtectedFiles")

    # This list is hardcoded for now
    #
    # As we expand to future target support, we may need to make the list of LV FPGA generated
    # files to copy more dynamic based on the target
    files_to_copy = [
        "CodeGenerationResults.lvtxt",
        "PkgLvFpgaConst.vhd",
        "PkgCommIntConfiguration.vhd",
        "PkgDmaPortCommIfcRegs.vhd",
    ]

    for file_name in files_to_copy:
        source_file = os.path.join(protected_files_folder, file_name)
        destination_file = os.path.join(config.the_window_folder_output, file_name)

        try:
            if os.path.exists(source_file):
                shutil.copy(source_file, destination_file)
                print(f"Copied {source_file} to {destination_file}")
            else:
                print(f"Error: Source file {source_file} not found")
        except Exception as e:
            print(f"Error copying file: {str(e)}")


def _extract_lv_window_constraints(config):
    """Extract LabVIEW FPGA constraints from the main constraints.xdc file.

    This function finds the constraints section marked with BEGIN_LV_FPGA_CONSTRAINTS and
    END_LV_FPGA_CONSTRAINTS in the constraints.xdc file, and creates a separate file
    containing just those constraints in the Window folder, excluding the marker lines.

    Args:
        config (FileConfiguration): Configuration settings object
    """
    # Extract the parent directory path from the XPR path
    # The NIProtectedFiles folder is typically at the same level as VivadoProject
    project_export_base = os.path.dirname(os.path.dirname(config.vivado_project_export_xpr))
    protected_files_folder = os.path.join(project_export_base, "NIProtectedFiles")

    source_file = os.path.join(protected_files_folder, "constraints.xdc")
    destination_folder = config.the_window_folder_output
    destination_file = os.path.join(destination_folder, "TheWindowConstraints.xdc")

    # Make sure destination folder exists
    os.makedirs(destination_folder, exist_ok=True)

    # Extract constraints between markers
    try:
        if os.path.exists(source_file):
            with open(source_file, "r", encoding="utf-8") as f_in:
                lines = f_in.readlines()

            # Find the marker lines
            start_idx = None
            end_idx = None

            for i, line in enumerate(lines):
                if "# BEGIN_LV_FPGA_CONSTRAINTS" in line:
                    start_idx = i
                if "# END_LV_FPGA_CONSTRAINTS" in line:
                    end_idx = i
                    break

            # Check if markers were found
            if start_idx is None or end_idx is None:
                print("Warning: Could not find constraint markers in constraints.xdc")
                return

            # Extract the constraints EXCLUDING the marker lines
            lv_constraints = lines[start_idx + 1 : end_idx]

            # Write the constraints to the destination file
            with open(destination_file, "w", encoding="utf-8") as f_out:
                f_out.writelines(lv_constraints)

            print(f"Successfully extracted LV FPGA constraints to {destination_file}")
        else:
            print(f"Error: Source file {source_file} not found")
    except Exception as e:
        print(f"Error extracting constraints: {str(e)}")


def get_window():
    """Main entry point for the script."""
    config = common.load_config()

    _get_window_netlist(config)
    _copy_lv_generated_files(config)
    _extract_lv_window_constraints(config)

    print("\n" + "=" * 80)
    print("NOTICE: If you have already created a Vivado project, you must run")
    print('         "nihdl create-project --update" to pull in the latest Window netlist files.')
    print("=" * 80)


if __name__ == "__main__":
    get_window()
