import ast
import logging
from dataclasses import dataclass
from enum import Enum
from typing import Any, Optional

import z3


class PropertyType(Enum):
    """Types of properties to verify."""

    ASSERTION = "assertion"
    INVARIANT = "invariant"
    REACHABILITY = "reachability"
    LIVENESS = "liveness"
    SAFETY = "safety"
    DEADLOCK = "deadlock"


@dataclass
class Property:
    """Represents a property to verify."""

    type: PropertyType
    expression: str
    location: Optional[str] = None
    scope: Optional[str] = None
    description: Optional[str] = None
    severity: str = "error"


@dataclass
class Counterexample:
    """Represents a counterexample to a property."""

    property: Property
    variable_values: dict[str, Any]
    execution_trace: list[str]
    path_conditions: list[Any]
    location: Optional[str] = None


@dataclass
class VerificationResult:
    """Result of property verification."""

    property: Property
    verified: bool
    counterexample: Optional[Counterexample] = None
    verification_time: float = 0.0
    proof: Optional[Any] = None


class ModelCheckingStrategy(Enum):
    """Strategies for model checking."""

    BOUNDED = "bounded"
    INDUCTIVE = "inductive"
    COMPOSITIONAL = "compositional"
    ABSTRACTION_REFINEMENT = "abstraction_refinement"


@dataclass
class ModelCheckerConfig:
    """Configuration for model checking."""

    strategy: ModelCheckingStrategy = ModelCheckingStrategy.BOUNDED
    max_depth: int = 100
    timeout: Optional[int] = None
    memory_limit: Optional[int] = None
    use_abstractions: bool = True
    generate_proofs: bool = False
    parallel_checking: bool = False


class ModelCheckingError(Exception):
    """Base class for model checking errors."""

    pass


class ModelChecker:
    """Advanced model checker with property verification and counterexample generation."""

    def __init__(self, engine, config: Optional[ModelCheckerConfig] = None):
        self.engine = engine
        self.config = config or ModelCheckerConfig()
        self.properties: list[Property] = []
        self.abstractions: dict[str, Any] = {}
        self.verification_results: list[VerificationResult] = []
        self._setup_logging()

    def verify_property(self, property_: Property) -> VerificationResult:
        """
        Verify a property using the appropriate strategy.

        Args:
            property_: Property to verify

        Returns:
            Verification result with possible counterexample
        """
        self.properties.append(property_)

        try:
            if self.config.strategy == ModelCheckingStrategy.BOUNDED:
                result = self._bounded_model_checking(property_)
            elif self.config.strategy == ModelCheckingStrategy.INDUCTIVE:
                result = self._inductive_verification(property_)
            elif self.config.strategy == ModelCheckingStrategy.COMPOSITIONAL:
                result = self._compositional_verification(property_)
            else:
                result = self._abstraction_refinement(property_)

            self.verification_results.append(result)
            return result

        except Exception as e:
            self.logger.error(f"Verification error: {str(e)}")
            raise ModelCheckingError(str(e))

    def verify_assertions(self, code: str) -> list[VerificationResult]:
        """Verify all assertions in the code."""
        assertions = self._extract_assertions(code)
        results = []

        for assertion in assertions:
            property_ = Property(
                type=PropertyType.ASSERTION,
                expression=assertion["condition"],
                location=assertion["location"],
            )
            results.append(self.verify_property(property_))

        return results

    def verify_invariants(self, invariants: list[str]) -> list[VerificationResult]:
        """Verify multiple invariants."""
        results = []

        for invariant in invariants:
            property_ = Property(type=PropertyType.INVARIANT, expression=invariant)
            results.append(self.verify_property(property_))

        return results

    def check_reachability(self, target_state: dict[str, Any]) -> VerificationResult:
        """Check if a target state is reachable."""
        expression = self._state_to_expression(target_state)
        property_ = Property(type=PropertyType.REACHABILITY, expression=expression)
        return self.verify_property(property_)

    def verify_deadlock_freedom(self) -> VerificationResult:
        """Verify that the system is deadlock-free."""
        property_ = Property(type=PropertyType.DEADLOCK, expression="no_deadlock")
        return self.verify_property(property_)

    def generate_test_cases(self, property_: Property) -> list[dict[str, Any]]:
        """Generate test cases that exercise the property."""
        test_cases = []

        # Get symbolic execution paths
        paths = self.engine.execute(property_.expression)

        for path in paths:
            if self._is_interesting_path(path, property_):
                test_case = self._generate_test_case(path)
                test_cases.append(test_case)

        return test_cases

    def _bounded_model_checking(self, property_: Property) -> VerificationResult:
        """Perform bounded model checking."""
        self.engine.solver.push()

        try:
            # Convert property to constraints
            property_constraint = self._property_to_constraint(property_)

            # Add negation of property (look for counterexample)
            self.engine.solver.add_constraint(z3.Not(property_constraint))

            # Check satisfiability
            if self.engine.solver.check_sat():
                # Found counterexample
                model = self.engine.solver.get_model()
                counterexample = self._create_counterexample(property_, model)
                return VerificationResult(
                    property=property_, verified=False, counterexample=counterexample
                )
            else:
                # Property verified up to bound
                return VerificationResult(property=property_, verified=True)

        finally:
            self.engine.solver.pop()

    def _inductive_verification(self, property_: Property) -> VerificationResult:
        """Perform inductive verification."""
        # Base case
        base_result = self._verify_base_case(property_)
        if not base_result.verified:
            return base_result

        # Inductive step
        return self._verify_inductive_step(property_)

    def _compositional_verification(self, property_: Property) -> VerificationResult:
        """Perform compositional verification."""
        # Decompose system into components
        components = self._decompose_system()

        # Verify each component separately
        component_results = []
        for component in components:
            result = self._verify_component(component, property_)
            component_results.append(result)

        # Combine results
        return self._combine_component_results(component_results)

    def _abstraction_refinement(self, property_: Property) -> VerificationResult:
        """Perform abstraction refinement verification."""
        while True:
            # Create abstraction
            abstraction = self._create_abstraction()

            # Verify property on abstraction
            result = self._verify_abstraction(abstraction, property_)

            if result.verified:
                return result

            # Refine abstraction based on counterexample
            if not self._refine_abstraction(abstraction, result.counterexample):
                return result  # No further refinement possible

    def _extract_assertions(self, code: str) -> list[dict]:
        """Extract assertions from code."""
        assertions = []
        tree = ast.parse(code)

        for node in ast.walk(tree):
            if isinstance(node, ast.Assert):
                assertions.append(
                    {
                        "condition": ast.unparse(node.test),
                        "location": f"line {node.lineno}",
                    }
                )

        return assertions

    def _property_to_constraint(self, property_: Property) -> Any:
        """Convert a property to a Z3 constraint."""
        if property_.type == PropertyType.ASSERTION:
            return self._parse_expression(property_.expression)
        elif property_.type == PropertyType.INVARIANT:
            return self._create_invariant_constraint(property_.expression)
        elif property_.type == PropertyType.REACHABILITY:
            return self._create_reachability_constraint(property_.expression)
        elif property_.type == PropertyType.DEADLOCK:
            return self._create_deadlock_constraint()
        else:
            raise ModelCheckingError(f"Unsupported property type: {property_.type}")

    def _create_counterexample(
        self, property_: Property, model: dict[str, Any]
    ) -> Counterexample:
        """Create a counterexample from a model."""
        return Counterexample(
            property=property_,
            variable_values=model,
            execution_trace=self._extract_execution_trace(model),
            path_conditions=self._extract_path_conditions(model),
            location=property_.location,
        )

    def _verify_base_case(self, property_: Property) -> VerificationResult:
        """Verify the base case for inductive verification."""
        self.engine.solver.push()
        try:
            # Add initial state constraints
            initial_state = self._get_initial_state_constraint()
            self.engine.solver.add_constraint(initial_state)

            # Add property constraint
            property_constraint = self._property_to_constraint(property_)
            self.engine.solver.add_constraint(z3.Not(property_constraint))

            if self.engine.solver.check_sat():
                model = self.engine.solver.get_model()
                return VerificationResult(
                    property=property_,
                    verified=False,
                    counterexample=self._create_counterexample(property_, model),
                )
            return VerificationResult(property=property_, verified=True)
        finally:
            self.engine.solver.pop()

    def _verify_inductive_step(self, property_: Property) -> VerificationResult:
        """Verify the inductive step."""
        self.engine.solver.push()
        try:
            # Add property at state n
            property_n = self._property_to_constraint(property_)
            self.engine.solver.add_constraint(property_n)

            # Add transition relation
            transition = self._get_transition_relation()
            self.engine.solver.add_constraint(transition)

            # Check property at state n+1
            property_n1 = self._property_to_constraint_next(property_)
            self.engine.solver.add_constraint(z3.Not(property_n1))

            if self.engine.solver.check_sat():
                model = self.engine.solver.get_model()
                return VerificationResult(
                    property=property_,
                    verified=False,
                    counterexample=self._create_counterexample(property_, model),
                )
            return VerificationResult(property=property_, verified=True)
        finally:
            self.engine.solver.pop()

    def _setup_logging(self):
        """Setup logging configuration."""
        logging.basicConfig(
            level=logging.INFO,
            format="%(asctime)s - %(name)s - %(levelname)s - %(message)s",
        )
        self.logger = logging.getLogger("ModelChecker")


def create_model_checker(
    engine, config: Optional[ModelCheckerConfig] = None
) -> ModelChecker:
    """Create a new model checker instance."""
    return ModelChecker(engine, config)
