"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AstroSiteConstruct = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const aws_apigatewayv2_alpha_1 = require("@aws-cdk/aws-apigatewayv2-alpha");
const aws_apigatewayv2_integrations_alpha_1 = require("@aws-cdk/aws-apigatewayv2-integrations-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_certificatemanager_1 = require("aws-cdk-lib/aws-certificatemanager");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const aws_cloudfront_origins_1 = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_lambda_nodejs_1 = require("aws-cdk-lib/aws-lambda-nodejs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
/**
 * The base class for all constructs.
 */
class AstroSiteConstruct extends constructs_1.Construct {
    constructor(scope, id) {
        super(scope, id);
    }
    /**
     * New bucket.
     *
     * @param scope
     * @param whEnabled
     * @param props
     * @returns
     */
    newBucket(scope, whEnabled, props) {
        const publicAccess = whEnabled
            ? {
                publicReadAccess: true,
                blockPublicAccess: new aws_s3_1.BlockPublicAccess({
                    blockPublicAcls: false,
                    blockPublicPolicy: false,
                    ignorePublicAcls: false,
                    restrictPublicBuckets: false,
                }),
            }
            : {};
        return new aws_s3_1.Bucket(scope, "StaticAssets", {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            websiteIndexDocument: whEnabled
                ? props?.indexhtml ?? "index.html"
                : undefined,
            websiteErrorDocument: whEnabled
                ? props?.errorhtml ?? "error.html"
                : undefined,
            cors: props?.cors,
            ...publicAccess,
        });
    }
    /**
     * New nodejs function.
     *
     * @param scope
     * @param serverEntry
     * @param props
     * @returns
     */
    newFunction(scope, serverEntry, props) {
        const runtime = this.strToRuntime(props?.runtime);
        return new aws_lambda_nodejs_1.NodejsFunction(scope, "SSRFunction", {
            ...props,
            bundling: {
                ...props?.bundling,
                logLevel: props?.bundling?.logLevel ?? aws_lambda_nodejs_1.LogLevel.ERROR,
            },
            runtime,
            entry: serverEntry,
        });
    }
    /**
     * New HttpApi Gateway.
     *
     * @param scope
     * @param fn
     * @param props
     * @returns
     */
    newHttpApiGw(scope, fn, props) {
        return new aws_apigatewayv2_alpha_1.HttpApi(scope, "HttpApi", {
            corsPreflight: props?.cors,
            defaultAuthorizer: props?.authorizer,
            defaultAuthorizationScopes: props?.authorizationScopes,
            defaultIntegration: new aws_apigatewayv2_integrations_alpha_1.HttpLambdaIntegration("HttpLambdaIntegration", fn),
        });
    }
    /**
     * New S3 origin.
     * @param scope
     * @param bucket
     * @returns
     */
    newS3Origin(scope, bucket) {
        const originAccessIdentity = new aws_cloudfront_1.OriginAccessIdentity(scope, "OriginAccessIdentity", {
            comment: `Allows CloudFront to reach the bucket ${bucket.bucketName}`,
        });
        bucket.grantRead(originAccessIdentity);
        return new aws_cloudfront_origins_1.S3Origin(bucket, { originAccessIdentity });
    }
    /**
     * New HttpApi Gateway origin.
     *
     * @param httpApi
     * @returns
     */
    newHttpApiGatewayOrigin(httpApi) {
        return new aws_cloudfront_origins_1.HttpOrigin(aws_cdk_lib_1.Fn.select(1, aws_cdk_lib_1.Fn.split("://", httpApi.apiEndpoint ?? "")));
    }
    /**
     * New CloudFront distribution.
     *
     * @param scope
     * @param defaultBehavior
     * @param props
     */
    newDistribution(scope, defaultBehavior, props) {
        const edgeLambdas = [
            ...(defaultBehavior.edgeLambdas ?? []),
            ...(props?.edgeFunctions ?? []),
        ];
        const functionAssociations = [
            ...(defaultBehavior.functionAssociations ?? []),
            ...(props?.cfFunctions ?? []),
        ];
        return new aws_cloudfront_1.Distribution(scope, "Distribution", {
            domainNames: [props?.domain ?? ""],
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(this, `Certificate-${props?.domain ?? ""}`, props?.certificateArn ?? ""),
            geoRestriction: props?.geoRestriction,
            logBucket: props?.logBucket,
            logIncludesCookies: props?.logIncludesCookies,
            logFilePrefix: props?.logFilePrefix,
            priceClass: props?.priceClass ?? aws_cloudfront_1.PriceClass.PRICE_CLASS_200,
            webAclId: props?.webACLId,
            errorResponses: props?.errorResponses,
            httpVersion: aws_cloudfront_1.HttpVersion.HTTP2_AND_3,
            defaultBehavior: {
                ...defaultBehavior,
                edgeLambdas: edgeLambdas,
                functionAssociations: functionAssociations,
            },
        });
    }
    /**
     * Transform string to Runtime.
     *
     * @param str
     * @returns
     * @default Runtime.NODEJS_18_X
     */
    strToRuntime(str) {
        if (str == "nodejs16.x")
            return aws_lambda_1.Runtime.NODEJS_16_X;
        return aws_lambda_1.Runtime.NODEJS_18_X;
    }
    /**
     * Parse routes from directory.
     * if the item is directory will parse to {"/item/*": "/item/*"} or {"/item/{proxy+}": "/item/{proxy}"}
     * if the item is file will parse to {"/item": "/item"}
     *
     * @param dir
     * @param isCf CloudFront route or not, HttpApi Gateway route by defauly, default false
     * @returns
     */
    parseRoutesFromDir(dir, isCf) {
        const map = {};
        const items = fs.readdirSync(dir);
        for (const item of items) {
            if (fs.lstatSync(path.join(dir, item)).isDirectory()) {
                if (isCf ?? false) {
                    map[`/${item}/*`] = `/${item}/*`;
                }
                else {
                    map[`/${item}/{proxy+}`] = `/${item}/{proxy}`;
                }
            }
            else {
                map[`/${item}`] = `/${item}`;
            }
        }
        return map;
    }
}
exports.AstroSiteConstruct = AstroSiteConstruct;
_a = JSII_RTTI_SYMBOL_1;
AstroSiteConstruct[_a] = { fqn: "@astrojs-aws/construct.AstroSiteConstruct", version: "0.0.13" };
//# sourceMappingURL=data:application/json;base64,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