# -*- coding: utf8 -*-
# Copyright (c) 2017-2025 Tencent. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import json

from tencentcloud.common.exception.tencent_cloud_sdk_exception import TencentCloudSDKException
from tencentcloud.common.abstract_client import AbstractClient
from tencentcloud.iai.v20180301 import models


class IaiClient(AbstractClient):
    _apiVersion = '2018-03-01'
    _endpoint = 'iai.tencentcloudapi.com'
    _service = 'iai'


    def AnalyzeDenseLandmarks(self, request):
        r"""对请求图片进行五官定位（也称人脸关键点定位），获得人脸的精准信息，返回多达888点关键信息，对五官和脸部轮廓进行精确定位。

        :param request: Request instance for AnalyzeDenseLandmarks.
        :type request: :class:`tencentcloud.iai.v20180301.models.AnalyzeDenseLandmarksRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.AnalyzeDenseLandmarksResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("AnalyzeDenseLandmarks", params, headers=headers)
            response = json.loads(body)
            model = models.AnalyzeDenseLandmarksResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def AnalyzeFace(self, request):
        r"""对请求图片进行五官定位（也称人脸关键点定位），计算构成人脸轮廓的 90 个点，包括眉毛（左右各 8 点）、眼睛（左右各 8 点）、鼻子（13 点）、嘴巴（22 点）、脸型轮廓（21 点）、眼珠[或瞳孔]（2点）。

        >
        - 公共参数中的签名方式请使用V3版本，即配置SignatureMethod参数为TC3-HMAC-SHA256。

        :param request: Request instance for AnalyzeFace.
        :type request: :class:`tencentcloud.iai.v20180301.models.AnalyzeFaceRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.AnalyzeFaceResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("AnalyzeFace", params, headers=headers)
            response = json.loads(body)
            model = models.AnalyzeFaceResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def CompareFace(self, request):
        r"""对两张图片中的人脸进行相似度比对，返回人脸相似度分数。

        若您需要判断 “此人是否是某人”，即验证某张图片中的人是否是已知身份的某人，如常见的人脸登录场景，建议使用[人脸验证](https://cloud.tencent.com/document/product/867/32806)或[人员验证](https://cloud.tencent.com/document/product/867/38879)接口。

        >
        - 公共参数中的签名方式请使用V3版本，即配置SignatureMethod参数为TC3-HMAC-SHA256。

        :param request: Request instance for CompareFace.
        :type request: :class:`tencentcloud.iai.v20180301.models.CompareFaceRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.CompareFaceResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("CompareFace", params, headers=headers)
            response = json.loads(body)
            model = models.CompareFaceResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def CopyPerson(self, request):
        r"""将已存在于某人员库的人员复制到其他人员库，该人员的描述信息不会被复制。单个人员最多只能同时存在100个人员库中。
        >
        - 注：若该人员创建时算法模型版本为2.0，复制到非2.0算法模型版本的Group中时，复制操作将会失败。

        :param request: Request instance for CopyPerson.
        :type request: :class:`tencentcloud.iai.v20180301.models.CopyPersonRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.CopyPersonResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("CopyPerson", params, headers=headers)
            response = json.loads(body)
            model = models.CopyPersonResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def CreateFace(self, request):
        r"""将一组人脸图片添加到一个人员中。一个人员最多允许包含 5 张图片。若该人员存在多个人员库中，所有人员库中该人员图片均会增加。

        >
        - 公共参数中的签名方式请使用V3版本，即配置SignatureMethod参数为TC3-HMAC-SHA256。

        :param request: Request instance for CreateFace.
        :type request: :class:`tencentcloud.iai.v20180301.models.CreateFaceRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.CreateFaceResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("CreateFace", params, headers=headers)
            response = json.loads(body)
            model = models.CreateFaceResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def CreateGroup(self, request):
        r"""用于创建一个空的人员库，如果人员库已存在返回错误。
        可根据需要创建自定义描述字段，用于辅助描述该人员库下的人员信息。

        1个APPID下最多创建10万个人员库（Group）、最多包含5000万张人脸（Face）。

        不同算法模型版本（FaceModelVersion）的人员库（Group）最多可包含人脸（Face）数不同。算法模型版本为2.0的人员库最多包含100万张人脸，算法模型版本为3.0的人员库最多可包含300万张人脸。

        :param request: Request instance for CreateGroup.
        :type request: :class:`tencentcloud.iai.v20180301.models.CreateGroupRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.CreateGroupResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("CreateGroup", params, headers=headers)
            response = json.loads(body)
            model = models.CreateGroupResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def CreatePerson(self, request):
        r"""创建人员，添加人脸、姓名、性别及其他相关信息。

        >
        - 公共参数中的签名方式请使用V3版本，即配置SignatureMethod参数为TC3-HMAC-SHA256。

        :param request: Request instance for CreatePerson.
        :type request: :class:`tencentcloud.iai.v20180301.models.CreatePersonRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.CreatePersonResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("CreatePerson", params, headers=headers)
            response = json.loads(body)
            model = models.CreatePersonResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def DeleteFace(self, request):
        r"""删除一个人员下的人脸图片。如果该人员只有一张人脸图片，则返回错误。

        :param request: Request instance for DeleteFace.
        :type request: :class:`tencentcloud.iai.v20180301.models.DeleteFaceRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.DeleteFaceResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("DeleteFace", params, headers=headers)
            response = json.loads(body)
            model = models.DeleteFaceResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def DeleteGroup(self, request):
        r"""删除该人员库及包含的所有的人员。同时，人员对应的所有人脸信息将被删除。若某人员同时存在多个人员库中，该人员不会被删除，但属于该人员库中的自定义描述字段信息会被删除，属于其他人员库的自定义描述字段信息不受影响。

        :param request: Request instance for DeleteGroup.
        :type request: :class:`tencentcloud.iai.v20180301.models.DeleteGroupRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.DeleteGroupResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("DeleteGroup", params, headers=headers)
            response = json.loads(body)
            model = models.DeleteGroupResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def DeletePerson(self, request):
        r"""删除该人员信息，此操作会导致所有人员库均删除此人员。同时，该人员的所有人脸信息将被删除。

        :param request: Request instance for DeletePerson.
        :type request: :class:`tencentcloud.iai.v20180301.models.DeletePersonRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.DeletePersonResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("DeletePerson", params, headers=headers)
            response = json.loads(body)
            model = models.DeletePersonResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def DeletePersonFromGroup(self, request):
        r"""从某人员库中删除人员，此操作仅影响该人员库。若该人员仅存在于指定的人员库中，该人员将被删除，其所有的人脸信息也将被删除。

        :param request: Request instance for DeletePersonFromGroup.
        :type request: :class:`tencentcloud.iai.v20180301.models.DeletePersonFromGroupRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.DeletePersonFromGroupResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("DeletePersonFromGroup", params, headers=headers)
            response = json.loads(body)
            model = models.DeletePersonFromGroupResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def DetectFace(self, request):
        r"""检测给定图片中的人脸（Face）的位置、相应的面部属性和人脸质量信息，位置包括 (x，y，w，h)，面部属性包括性别（gender）、年龄（age）、表情（expression）、魅力（beauty）、眼镜（glass）、发型（hair）、口罩（mask）和姿态 (pitch，roll，yaw)，人脸质量信息包括整体质量分（score）、模糊分（sharpness）、光照分（brightness）和五官遮挡分（completeness）。


        其中，人脸质量信息主要用于评价输入的人脸图片的质量。在使用人脸识别服务时，建议您对输入的人脸图片进行质量检测，提升后续业务处理的效果。该功能的应用场景包括：

        1） 人员库[创建人员](https://cloud.tencent.com/document/product/867/32793)/[增加人脸](https://cloud.tencent.com/document/product/867/32795)：保证人员人脸信息的质量，便于后续的业务处理。

        2） [人脸搜索](https://cloud.tencent.com/document/product/867/32798)：保证输入的图片质量，快速准确匹配到对应的人员。

        3） [人脸验证](https://cloud.tencent.com/document/product/867/32806)：保证人脸信息的质量，避免明明是本人却认证不通过的情况。

        4） [人脸融合](https://cloud.tencent.com/product/facefusion)：保证上传的人脸质量，人脸融合的效果更好。

        >
        - 公共参数中的签名方式请使用V3版本，即配置SignatureMethod参数为TC3-HMAC-SHA256。

        :param request: Request instance for DetectFace.
        :type request: :class:`tencentcloud.iai.v20180301.models.DetectFaceRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.DetectFaceResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("DetectFace", params, headers=headers)
            response = json.loads(body)
            model = models.DetectFaceResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def DetectFaceAttributes(self, request):
        r"""检测给定图片中的人脸（Face）的位置、相应的面部属性和人脸质量信息，位置包括 (x，y，w，h)，面部属性包括性别（gender）、年龄（age）、表情（expression）、魅力（beauty）、眼镜（glass）、发型（hair）、口罩（mask）和姿态 (pitch，roll，yaw)，人脸质量信息包括整体质量分（score）、模糊分（sharpness）、光照分（brightness）和五官遮挡分（completeness）。


        其中，人脸质量信息主要用于评价输入的人脸图片的质量。在使用人脸识别服务时，建议您对输入的人脸图片进行质量检测，提升后续业务处理的效果。该功能的应用场景包括：

        1） 人员库[创建人员](https://cloud.tencent.com/document/product/867/32793)/[增加人脸](https://cloud.tencent.com/document/product/867/32795)：保证人员人脸信息的质量，便于后续的业务处理。

        2） [人脸搜索](https://cloud.tencent.com/document/product/867/32798)：保证输入的图片质量，快速准确匹配到对应的人员。

        3） [人脸验证](https://cloud.tencent.com/document/product/867/32806)：保证人脸信息的质量，避免明明是本人却认证不通过的情况。

        4） [人脸融合](https://cloud.tencent.com/product/facefusion)：保证上传的人脸质量，人脸融合的效果更好。

        >
        - 本接口是[人脸检测与分析](https://cloud.tencent.com/document/product/867/32800)的升级，具体在于：

        1.本接口可以指定需要计算返回的人脸属性，避免无效计算，降低耗时；

        2.本接口支持更多属性细项数，也会持续增加更多功能。

        请您使用本接口完成相应的人脸检测与属性分析需求。

        - 公共参数中的签名方式请使用V3版本，即配置SignatureMethod参数为TC3-HMAC-SHA256。

        :param request: Request instance for DetectFaceAttributes.
        :type request: :class:`tencentcloud.iai.v20180301.models.DetectFaceAttributesRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.DetectFaceAttributesResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("DetectFaceAttributes", params, headers=headers)
            response = json.loads(body)
            model = models.DetectFaceAttributesResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def GetGroupInfo(self, request):
        r"""获取人员库信息。

        :param request: Request instance for GetGroupInfo.
        :type request: :class:`tencentcloud.iai.v20180301.models.GetGroupInfoRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.GetGroupInfoResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("GetGroupInfo", params, headers=headers)
            response = json.loads(body)
            model = models.GetGroupInfoResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def GetGroupList(self, request):
        r"""获取人员库列表。

        :param request: Request instance for GetGroupList.
        :type request: :class:`tencentcloud.iai.v20180301.models.GetGroupListRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.GetGroupListResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("GetGroupList", params, headers=headers)
            response = json.loads(body)
            model = models.GetGroupListResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def GetPersonBaseInfo(self, request):
        r"""获取指定人员的信息，包括姓名、性别、人脸等。

        :param request: Request instance for GetPersonBaseInfo.
        :type request: :class:`tencentcloud.iai.v20180301.models.GetPersonBaseInfoRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.GetPersonBaseInfoResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("GetPersonBaseInfo", params, headers=headers)
            response = json.loads(body)
            model = models.GetPersonBaseInfoResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def GetPersonGroupInfo(self, request):
        r"""获取指定人员的信息，包括加入的人员库、描述内容等。

        :param request: Request instance for GetPersonGroupInfo.
        :type request: :class:`tencentcloud.iai.v20180301.models.GetPersonGroupInfoRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.GetPersonGroupInfoResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("GetPersonGroupInfo", params, headers=headers)
            response = json.loads(body)
            model = models.GetPersonGroupInfoResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def GetPersonList(self, request):
        r"""获取指定人员库中的人员列表。

        :param request: Request instance for GetPersonList.
        :type request: :class:`tencentcloud.iai.v20180301.models.GetPersonListRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.GetPersonListResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("GetPersonList", params, headers=headers)
            response = json.loads(body)
            model = models.GetPersonListResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def GetPersonListNum(self, request):
        r"""获取指定人员库中人员数量。

        :param request: Request instance for GetPersonListNum.
        :type request: :class:`tencentcloud.iai.v20180301.models.GetPersonListNumRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.GetPersonListNumResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("GetPersonListNum", params, headers=headers)
            response = json.loads(body)
            model = models.GetPersonListNumResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def GetUpgradeGroupFaceModelVersionJobList(self, request):
        r"""避免官网歧义

        获取人员库升级任务列表

        :param request: Request instance for GetUpgradeGroupFaceModelVersionJobList.
        :type request: :class:`tencentcloud.iai.v20180301.models.GetUpgradeGroupFaceModelVersionJobListRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.GetUpgradeGroupFaceModelVersionJobListResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("GetUpgradeGroupFaceModelVersionJobList", params, headers=headers)
            response = json.loads(body)
            model = models.GetUpgradeGroupFaceModelVersionJobListResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def GetUpgradeGroupFaceModelVersionResult(self, request):
        r"""避免官网歧义

        人员库升级结果查询

        :param request: Request instance for GetUpgradeGroupFaceModelVersionResult.
        :type request: :class:`tencentcloud.iai.v20180301.models.GetUpgradeGroupFaceModelVersionResultRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.GetUpgradeGroupFaceModelVersionResultResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("GetUpgradeGroupFaceModelVersionResult", params, headers=headers)
            response = json.loads(body)
            model = models.GetUpgradeGroupFaceModelVersionResultResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def ModifyGroup(self, request):
        r"""修改人员库名称、备注、自定义描述字段名称。

        :param request: Request instance for ModifyGroup.
        :type request: :class:`tencentcloud.iai.v20180301.models.ModifyGroupRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.ModifyGroupResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("ModifyGroup", params, headers=headers)
            response = json.loads(body)
            model = models.ModifyGroupResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def ModifyPersonBaseInfo(self, request):
        r"""修改人员信息，包括名称、性别等。人员名称和性别修改会同步到包含该人员的所有人员库。

        :param request: Request instance for ModifyPersonBaseInfo.
        :type request: :class:`tencentcloud.iai.v20180301.models.ModifyPersonBaseInfoRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.ModifyPersonBaseInfoResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("ModifyPersonBaseInfo", params, headers=headers)
            response = json.loads(body)
            model = models.ModifyPersonBaseInfoResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def ModifyPersonGroupInfo(self, request):
        r"""修改指定人员库人员描述内容。

        :param request: Request instance for ModifyPersonGroupInfo.
        :type request: :class:`tencentcloud.iai.v20180301.models.ModifyPersonGroupInfoRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.ModifyPersonGroupInfoResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("ModifyPersonGroupInfo", params, headers=headers)
            response = json.loads(body)
            model = models.ModifyPersonGroupInfoResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def RevertGroupFaceModelVersion(self, request):
        r"""此接口文档早已废弃，为了避免官网18版本存留歧义，需要下线

        本接口用于回滚人员库的人脸识别算法模型版本。单个人员库有且仅有一次回滚机会。

        回滚操作会在10s内生效，回滚操作中，您对人员库的操作可能会失效。

        :param request: Request instance for RevertGroupFaceModelVersion.
        :type request: :class:`tencentcloud.iai.v20180301.models.RevertGroupFaceModelVersionRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.RevertGroupFaceModelVersionResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("RevertGroupFaceModelVersion", params, headers=headers)
            response = json.loads(body)
            model = models.RevertGroupFaceModelVersionResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def SearchFaces(self, request):
        r"""用于对一张待识别的人脸图片，在一个或多个人员库中识别出最相似的 TopK 人员，识别结果按照相似度从大到小排序。

        支持一次性识别图片中的最多 10 张人脸，支持一次性跨 100 个人员库（Group）搜索。

        单次搜索的人员库人脸总数量和人员库的算法模型版本（FaceModelVersion）相关。算法模型版本为2.0的人员库，单次搜索人员库人脸总数量不得超过 100 万张；算法模型版本为3.0的人员库，单次搜索人员库人脸总数量不得超过 300 万张。

        与[人员搜索](https://cloud.tencent.com/document/product/867/38881)及[人员搜索按库返回](https://cloud.tencent.com/document/product/867/38880)接口不同的是，本接口将该人员（Person）下的每个人脸（Face）都作为单独个体进行验证，而人员搜索及人员搜索按库返回接口 会将该人员（Person）下的所有人脸（Face）进行融合特征处理，即若某个Person下有4张 Face，本接口会将4张 Face 的特征进行融合处理，生成对应这个 Person 的特征，使搜索更加准确。


        本接口需与[人员库管理相关接口](https://cloud.tencent.com/document/product/867/32794)结合使用。

        >
        - 公共参数中的签名方式请使用V3版本，即配置SignatureMethod参数为TC3-HMAC-SHA256。

        :param request: Request instance for SearchFaces.
        :type request: :class:`tencentcloud.iai.v20180301.models.SearchFacesRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.SearchFacesResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("SearchFaces", params, headers=headers)
            response = json.loads(body)
            model = models.SearchFacesResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def SearchFacesReturnsByGroup(self, request):
        r"""用于对一张待识别的人脸图片，在一个或多个人员库中识别出最相似的 TopK 人员，按照**人员库的维度**以人员相似度从大到小顺序排列。

        支持一次性识别图片中的最多 10 张人脸，支持跨人员库（Group）搜索。

        单次搜索的人员库人脸总数量和人员库的算法模型版本（FaceModelVersion）相关。算法模型版本为2.0的人员库，单次搜索人员库人脸总数量不得超过 100 万张；算法模型版本为3.0的人员库，单次搜索人员库人脸总数量不得超过 300 万张。

        与[人员搜索](https://cloud.tencent.com/document/product/867/38881)及[人员搜索按库返回](https://cloud.tencent.com/document/product/867/38880)接口不同的是，本接口将该人员（Person）下的每个人脸（Face）都作为单独个体进行验证，而[人员搜索](https://cloud.tencent.com/document/product/867/38881)及[人员搜索按库返回](https://cloud.tencent.com/document/product/867/38880)接口 会将该人员（Person）下的所有人脸（Face）进行融合特征处理，即若某个Person下有4张 Face，本接口会将4张 Face 的特征进行融合处理，生成对应这个 Person 的特征，使搜索更加准确。

        本接口需与[人员库管理相关接口](https://cloud.tencent.com/document/product/867/32794)结合使用。

        >
        - 公共参数中的签名方式请使用V3版本，即配置SignatureMethod参数为TC3-HMAC-SHA256。

        :param request: Request instance for SearchFacesReturnsByGroup.
        :type request: :class:`tencentcloud.iai.v20180301.models.SearchFacesReturnsByGroupRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.SearchFacesReturnsByGroupResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("SearchFacesReturnsByGroup", params, headers=headers)
            response = json.loads(body)
            model = models.SearchFacesReturnsByGroupResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def SearchPersons(self, request):
        r"""用于对一张待识别的人脸图片，在一个或多个人员库中识别出最相似的 TopK 人员，按照相似度从大到小排列。

        支持一次性识别图片中的最多 10 张人脸，支持一次性跨 100 个人员库（Group）搜索。

        单次搜索的人员库人脸总数量和人员库的算法模型版本（FaceModelVersion）相关。算法模型版本为2.0的人员库，单次搜索人员库人脸总数量不得超过 100 万张；算法模型版本为3.0的人员库，单次搜索人员库人脸总数量不得超过 300 万张。

        本接口会将该人员（Person）下的所有人脸（Face）进行融合特征处理，即若某个 Person 下有4张 Face ，本接口会将4张 Face 的特征进行融合处理，生成对应这个 Person 的特征，使人员搜索（确定待识别的人脸图片是某人）更加准确。而[人脸搜索](https://cloud.tencent.com/document/product/867/32798)及[人脸搜索按库返回接口](https://cloud.tencent.com/document/product/867/38882)将该人员（Person）下的每个人脸（Face）都作为单独个体进行搜索。

        >
        - 公共参数中的签名方式请使用V3版本，即配置SignatureMethod参数为TC3-HMAC-SHA256。
        - 仅支持算法模型版本（FaceModelVersion）为3.0的人员库。

        :param request: Request instance for SearchPersons.
        :type request: :class:`tencentcloud.iai.v20180301.models.SearchPersonsRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.SearchPersonsResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("SearchPersons", params, headers=headers)
            response = json.loads(body)
            model = models.SearchPersonsResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def SearchPersonsReturnsByGroup(self, request):
        r"""用于对一张待识别的人脸图片，在一个或多个人员库中识别出最相似的 TopK 人员，按照**人员库的维度**以人员相似度从大到小顺序排列。

        支持一次性识别图片中的最多 10 张人脸，支持跨人员库（Group）搜索。

        单次搜索的人员库人脸总数量和人员库的算法模型版本（FaceModelVersion）相关。算法模型版本为2.0的人员库，单次搜索人员库人脸总数量不得超过 100 万张；算法模型版本为3.0的人员库，单次搜索人员库人脸总数量不得超过 300 万张。

        本接口会将该人员（Person）下的所有人脸（Face）进行融合特征处理，即若某个 Person 下有4张 Face ，本接口会将4张 Face 的特征进行融合处理，生成对应这个 Person 的特征，使人员搜索（确定待识别的人脸图片是某人）更加准确。而[人脸搜索](https://cloud.tencent.com/document/product/867/32798)及[人脸搜索按库返回接口](https://cloud.tencent.com/document/product/867/38882)将该人员（Person）下的每个人脸（Face）都作为单独个体进行搜索。
        >
        - 公共参数中的签名方式请使用V3版本，即配置SignatureMethod参数为TC3-HMAC-SHA256。
        - 仅支持算法模型版本（FaceModelVersion）为3.0的人员库。

        :param request: Request instance for SearchPersonsReturnsByGroup.
        :type request: :class:`tencentcloud.iai.v20180301.models.SearchPersonsReturnsByGroupRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.SearchPersonsReturnsByGroupResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("SearchPersonsReturnsByGroup", params, headers=headers)
            response = json.loads(body)
            model = models.SearchPersonsReturnsByGroupResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def UpgradeGroupFaceModelVersion(self, request):
        r"""接口已经不使用了，需要下线掉，以免留在官网产生歧义

        升级人员库。升级过程中，人员库仍然为原算法版本，人员库相关操作仍然支持。升级完成后，人员库为新算法版本。
        单个人员库有且仅支持一次回滚操作。
        注：此处QPS限制为10。

        :param request: Request instance for UpgradeGroupFaceModelVersion.
        :type request: :class:`tencentcloud.iai.v20180301.models.UpgradeGroupFaceModelVersionRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.UpgradeGroupFaceModelVersionResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("UpgradeGroupFaceModelVersion", params, headers=headers)
            response = json.loads(body)
            model = models.UpgradeGroupFaceModelVersionResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def VerifyFace(self, request):
        r"""给定一张人脸图片和一个 PersonId，判断图片中的人和 PersonId 对应的人是否为同一人。PersonId 请参考[人员库管理相关接口](https://cloud.tencent.com/document/product/867/32794)。

        与[人脸比对](https://cloud.tencent.com/document/product/867/32802)接口不同的是，人脸验证用于判断 “此人是否是此人”，“此人”的信息已存于人员库中，“此人”可能存在多张人脸图片；而[人脸比对](https://cloud.tencent.com/document/product/867/32802)用于判断两张人脸的相似度。

        与[人员验证](https://cloud.tencent.com/document/product/867/38879)接口不同的是，人脸验证将该人员（Person）下的每个人脸（Face）都作为单独个体进行验证，而[人员验证](https://cloud.tencent.com/document/product/867/38879)会将该人员（Person）下的所有人脸（Face）进行融合特征处理，即若某个 Person下有4张 Face，人员验证接口会将4张 Face 的特征进行融合处理，生成对应这个 Person 的特征，使人员验证（确定待识别的人脸图片是某人员）更加准确。

        >
        - 公共参数中的签名方式请使用V3版本，即配置SignatureMethod参数为TC3-HMAC-SHA256。

        :param request: Request instance for VerifyFace.
        :type request: :class:`tencentcloud.iai.v20180301.models.VerifyFaceRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.VerifyFaceResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("VerifyFace", params, headers=headers)
            response = json.loads(body)
            model = models.VerifyFaceResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))


    def VerifyPerson(self, request):
        r"""给定一张人脸图片和一个 PersonId，判断图片中的人和 PersonId 对应的人是否为同一人。PersonId 请参考[人员库管理相关接口](https://cloud.tencent.com/document/product/867/32794)。
        本接口会将该人员（Person）下的所有人脸（Face）进行融合特征处理，即若某个Person下有4张 Face，本接口会将4张 Face 的特征进行融合处理，生成对应这个 Person 的特征，使人员验证（确定待识别的人脸图片是某人员）更加准确。

         和人脸比对相关接口不同的是，人脸验证相关接口用于判断 “此人是否是此人”，“此人”的信息已存于人员库中，“此人”可能存在多张人脸图片；而人脸比对相关接口用于判断两张人脸的相似度。


        >
        - 公共参数中的签名方式请使用V3版本，即配置SignatureMethod参数为TC3-HMAC-SHA256。
        - 仅支持算法模型版本（FaceModelVersion）为3.0的人员库。

        :param request: Request instance for VerifyPerson.
        :type request: :class:`tencentcloud.iai.v20180301.models.VerifyPersonRequest`
        :rtype: :class:`tencentcloud.iai.v20180301.models.VerifyPersonResponse`

        """
        try:
            params = request._serialize()
            headers = request.headers
            body = self.call("VerifyPerson", params, headers=headers)
            response = json.loads(body)
            model = models.VerifyPersonResponse()
            model._deserialize(response["Response"])
            return model
        except Exception as e:
            if isinstance(e, TencentCloudSDKException):
                raise
            else:
                raise TencentCloudSDKException(type(e).__name__, str(e))