﻿# SMART GUIDE PRINCIPLES KNOWLEDGE BASE
GUIDE_PRINCIPLES = {
    "governance": {
        "core_question": "Who is accountable when AI systems make decisions?",
        "smart_analysis": [
            "🧭 DECISION TRACEABILITY: Map every AI decision to human responsibility chains",
            "🏛️ OVERSIGHT ARCHITECTURE: Create multi-layered review boards with domain experts",
            "📊 ACCOUNTABILITY MATRICES: Define clear RACI charts for system outcomes",
            "🚨 FAILSAFE PROTOCOLS: Establish emergency shutdown and human override procedures",
            "🔍 AUDIT TRAILS: Maintain immutable logs of all significant system decisions"
        ],
        "risk_indicators": [
            "No clear owner for system failures",
            "Lack of escalation procedures",
            "Single point of decision authority",
            "No independent oversight"
        ],
        "success_metrics": [
            "Clear ownership of all system outcomes",
            "Regular ethics board reviews conducted",
            "Documented escalation procedures",
            "Comprehensive audit trails maintained"
        ]
    },
    
    "understanding": {
        "core_question": "Do stakeholders truly comprehend how AI decisions affect them?",
        "smart_analysis": [
            "🧠 COGNITIVE ACCESSIBILITY: Explain complex AI in terms users actually understand",
            "🎯 CONTEXTUAL EXPLANATIONS: Provide different detail levels for different audiences",
            "🔍 DECISION TRANSPARENCY: Show how specific inputs lead to specific outputs",
            "⚠️ CONFIDENCE CALIBRATION: Clearly communicate system certainty and limitations",
            "📚 PROGRESSIVE LEARNING: Help users build AI literacy over time"
        ],
        "risk_indicators": [
            "Users blindly trust AI recommendations",
            "No explanation for rejections or errors",
            "Technical jargon in user communications",
            "Hidden assumptions in system design"
        ],
        "success_metrics": [
            "Users can articulate system limitations",
            "Clear explanations for all significant decisions",
            "Multiple explanation methods available",
            "Stakeholder comprehension validated through testing"
        ]
    },
    
    "integrity": {
        "core_question": "Does this AI system act consistently with human values?",
        "smart_analysis": [
            "⚖️ VALUE ALIGNMENT: Map organizational ethics to actual system behavior",
            "🛡️ MORAL GUARDRAILS: Implement boundaries that cannot be crossed",
            "🔁 CONSISTENCY TESTING: Verify uniform treatment across all scenarios",
            "💭 ETHICAL DILEMMA RESOLUTION: Establish protocols for value conflicts",
            "📈 VALUES EVOLUTION: Create mechanisms to update ethical frameworks"
        ],
        "risk_indicators": [
            "System behavior contradicts stated values",
            "No process for ethical dilemma resolution",
            "Values not translated to operational rules",
            "Inconsistent treatment of similar cases"
        ],
        "success_metrics": [
            "System behavior aligns with organizational ethics",
            "Clear protocols for ethical dilemmas",
            "Consistent treatment across user groups",
            "Regular values alignment verification"
        ]
    },
    
    "disclosure": {
        "core_question": "Are we being completely transparent about AI capabilities and limits?",
        "smart_analysis": [
            "🔓 RADICAL TRANSPARENCY: Proactively share limitations, not just capabilities",
            "🎭 AI IDENTIFICATION: Clearly mark all AI interactions to users",
            "📉 ERROR TRANSPARENCY: Honestly communicate mistakes and uncertainties",
            "🔍 DATA PROVENANCE: Show data sources and processing history",
            "🔄 UPDATE COMMUNICATION: Notify users of system changes and improvements"
        ],
        "risk_indicators": [
            "Hiding system limitations from users",
            "AI interactions not clearly identified",
            "Errors concealed or minimized",
            "Data sources obscured or unclear"
        ],
        "success_metrics": [
            "Users understand system limitations",
            "Clear AI interaction identification",
            "Honest error communication",
            "Transparent data sourcing"
        ]
    },
    
    "equity": {
        "core_question": "Does this AI system treat all people with fairness and dignity?",
        "smart_analysis": [
            "🌍 INCLUSIVE DESIGN: Build systems that work for diverse populations",
            "⚖️ FAIRNESS VALIDATION: Test across demographic groups and edge cases",
            "🔄 BIAS MITIGATION: Actively identify and address discriminatory patterns",
            "🎯 ACCESSIBILITY FIRST: Ensure equal access regardless of ability or background",
            "📊 IMPACT MONITORING: Continuously assess effects on different communities"
        ],
        "risk_indicators": [
            "Performance varies significantly across groups",
            "Training data lacks diversity",
            "No ongoing bias monitoring",
            "Accessibility considerations ignored"
        ],
        "success_metrics": [
            "Consistent performance across user groups",
            "Diverse representation in development",
            "Regular bias testing conducted",
            "Accessibility standards met or exceeded"
        ]
    }
}

# CONTEXT-AWARE RECOMMENDATIONS
CONTEXTUAL_ADVICE = {
    "healthcare": {
        "governance": "Clinical oversight committee with physician veto power",
        "understanding": "Explain diagnoses in patient-friendly terms with confidence scores",
        "integrity": "Patient welfare must override all other considerations",
        "disclosure": "Clear communication of AI's advisory role vs. medical diagnosis",
        "equity": "Test across diverse patient populations and medical conditions"
    },
    "finance": {
        "governance": "Compliance officers with regulatory authority over AI decisions",
        "understanding": "Clear explanations for credit denials or investment recommendations",
        "integrity": "Fiduciary responsibility embedded in all decision algorithms",
        "disclosure": "Transparent fee structures and conflict of interest statements",
        "equity": "Fair lending compliance across all demographic groups"
    },
    "education": {
        "governance": "Educator oversight with ability to override AI recommendations",
        "understanding": "Explain student assessments in developmentally appropriate terms",
        "integrity": "Student growth prioritized over algorithmic efficiency",
        "disclosure": "Clear communication of AI's role in educational decisions",
        "equity": "Ensure equal educational opportunities across all student groups"
    }
}

def get_system_risk_level(system_name, system_description=""):
    """Determine risk level based on system name and description."""
    system_text = (system_name + " " + system_description).lower()
    
    high_risk_terms = ["health", "medical", "diagnosis", "treatment", "criminal", "justice", 
                      "legal", "sentencing", "financial", "loan", "credit", "hiring", 
                      "employment", "admissions", "benefits", "housing", "student", "performance"]
    
    medium_risk_terms = ["customer", "service", "recommendation", "marketing", "sales",
                        "resource", "allocation", "advice", "counseling", "predictor"]
    
    for term in high_risk_terms:
        if term in system_text:
            return "high"
    
    for term in medium_risk_terms:
        if term in system_text:
            return "medium"
    
    return "low"

def get_smart_guide_analysis(principle, context=""):
    """Get intelligent analysis for each GUIDE principle."""
    principle_data = GUIDE_PRINCIPLES.get(principle, {})
    
    analysis = {
        "core_question": principle_data.get("core_question", ""),
        "smart_analysis": principle_data.get("smart_analysis", []),
        "context_advice": CONTEXTUAL_ADVICE.get(context, {}).get(principle, "")
    }
    
    return analysis

def assess_guide_maturity(system_description, principle):
    """Assess maturity level for each GUIDE principle."""
    principle_data = GUIDE_PRINCIPLES.get(principle, {})
    
    # Simple maturity assessment
    risk_terms = principle_data.get("risk_indicators", [])
    success_terms = principle_data.get("success_metrics", [])
    
    risk_count = sum(1 for term in risk_terms if any(word in system_description.lower() for word in term.lower().split()))
    success_count = sum(1 for term in success_terms if any(word in system_description.lower() for word in term.lower().split()))
    
    if success_count > risk_count:
        return "MATURE", f"Strong {principle} practices detected"
    elif risk_count > success_count:
        return "DEVELOPING", f"Need improvement in {principle} areas"
    else:
        return "ESTABLISHED", f"Basic {principle} framework in place"

def generate_smart_recommendations(system_context, risk_level):
    """Generate SPECIFIC, TAILORED recommendations based on exact context."""
    
    # Extract key information from system name
    system_lower = system_context.lower()
    
    # Determine the exact AI type
    ai_type = "general"
    if any(term in system_lower for term in ["diagnosis", "detection", "medical", "health"]):
        ai_type = "medical_diagnosis"
    elif any(term in system_lower for term in ["loan", "credit", "financial", "bank"]):
        ai_type = "financial_decision" 
    elif any(term in system_lower for term in ["hiring", "applicant", "recruitment"]):
        ai_type = "hiring"
    elif any(term in system_lower for term in ["student", "education", "grading"]):
        ai_type = "education"
    elif any(term in system_lower for term in ["sentencing", "criminal", "legal"]):
        ai_type = "criminal_justice"
    
    # SPECIFIC recommendations for each AI type
    specific_recommendations = {
        "medical_diagnosis": [
            "REQUIRE radiologist confirmation before any cancer diagnosis is communicated to patients",
            "IMPLEMENT real-time confidence scoring showing 'Low/Medium/High' certainty for each finding",
            "ESTABLISH 24/7 physician hotline for AI system questions and overrides",
            "DOCUMENT every AI recommendation in patient's electronic health record with timestamps",
            "CONDUCT monthly accuracy audits comparing AI diagnoses with final pathology reports"
        ],
        "financial_decision": [
            "PROVIDE specific reasons for loan denials citing exact credit factors used",
            "IMPLEMENT 30-day appeal process with human loan officer review",
            "TEST algorithm monthly for disparate impact across racial and gender groups",
            "MAINTAIN 7-year audit trail of all credit decisions for regulatory compliance",
            "DISCLOSE to applicants when AI versus human makes the final decision"
        ],
        "hiring": [
            "REMOVE demographic information from resumes before AI screening",
            "REQUIRE human HR review for all AI-rejected candidates",
            "VALIDATE hiring criteria against actual job performance data quarterly",
            "PROVIDE candidates with specific feedback on which qualifications were weighted most heavily",
            "AUDIT promotion recommendations for gender and racial parity every 6 months"
        ],
        "education": [
            "ALLOW teachers to override all AI-generated grades with written justification",
            "PROVIDE students with specific examples of why they received particular scores",
            "TEST grading consistency across different student demographic groups each semester",
            "CREATE parent portal explaining AI's role in educational recommendations",
            "IMPLEMENT 'grade challenge' process with human teacher re-evaluation"
        ],
        "criminal_justice": [
            "REQUIRE judge signature on all AI-generated risk assessments before use in court",
            "PROVIDE defendants with complete explanation of risk score factors",
            "AUDIT predictions against actual recidivism rates quarterly",
            "TRAIN public defenders on how to challenge AI risk assessment methodologies",
            "PUBLISH annual transparency report on algorithm performance and fairness metrics"
        ],
        "general": [
            "ESTABLISH clear human oversight protocol for system overrides",
            "IMPLEMENT regular bias testing across user demographics",
            "CREATE transparent documentation of system limitations",
            "DEVELOP stakeholder education materials about AI capabilities",
            "MAINTAIN comprehensive audit trails of all significant decisions"
        ]
    }
    
    # Get the SPECIFIC recommendations for this AI type
    recommendations = specific_recommendations.get(ai_type, specific_recommendations["general"])
    
    # Add risk-level specific actions
    risk_actions = {
        "high": [
            "MANDATORY third-party ethical audit every 6 months",
            "REQUIRE regulatory approval before production deployment",
            "IMPLEMENT emergency shutdown procedures accessible to multiple team members",
            "MAINTAIN 24/7 incident response team for system failures",
            "CONDUCT weekly system performance and fairness reviews"
        ],
        "medium": [
            "QUARTERLY internal ethics committee reviews",
            "MONTHLY bias and performance testing",
            "STAKEHOLDER feedback collection after significant decisions",
            "REGULAR system documentation updates",
            "BI-ANNUAL team training on ethical AI use"
        ],
        "low": [
            "ANNUAL system review checklist completion",
            "BASIC user education about system capabilities",
            "STANDARD error logging and monitoring",
            "OCCASIONAL performance validation testing"
        ]
    }
    
    # Combine specific + risk-based recommendations
    final_recommendations = recommendations + risk_actions.get(risk_level, [])
    
    return final_recommendations[:8]  # Return top 8 most relevant
