import json
import os
import sys

# Add templates to path
sys.path.append(os.path.join(os.path.dirname(__file__), '..'))

class GUIDEAuditor:
    def __init__(self):
        self.responses = {}
        self.situation_type = None
        
    def select_situation(self):
        print("\n" + "="*60)
        print("🎯 GUIDE AI ETHICS ASSESSMENT")
        print("="*60)
        print("\nSelect your AI system type:")
        print("1. AI Hiring System")
        print("2. AI Medical Diagnosis") 
        print("3. Custom AI System")
        
        while True:
            choice = input("\nEnter choice (1-3): ")
            if choice == "1":
                self.situation_type = "hiring"
                break
            elif choice == "2":
                self.situation_type = "healthcare"
                break
            elif choice == "3":
                self.situation_type = "custom"
                break
            else:
                print("Invalid choice. Please enter 1, 2, or 3.")
    
    def conduct_interview(self):
        # Simple questions for each principle
        questions = {
            "G": [
                "Who is responsible for overseeing this AI system?",
                "What governance policies are in place?",
                "How often is the system audited?"
            ],
            "U": [
                "Can users understand the AI's decisions?",
                "Are explanations provided for AI outcomes?",
                "How transparent is the decision process?"
            ],
            "I": [
                "How do you ensure data quality?",
                "What prevents system manipulation?",
                "How consistent is the AI across scenarios?"
            ],
            "D": [
                "Are users informed about AI involvement?",
                "What system limitations are disclosed?",
                "How transparent are the decision criteria?"
            ],
            "E": [
                "How do you test for bias across user groups?",
                "What ensures fair treatment of all users?",
                "How do you monitor for disparate impacts?"
            ]
        }
        
        print(f"\n🔍 Assessing: {self.situation_type}")
        print("Please answer questions (1-5 scale):")
        print("1: Not implemented, 3: Partial, 5: Fully implemented")
        print("-" * 60)
        
        self.responses = {"situation": self.situation_type, "scores": {}}
        
        principle_names = {
            "G": "Governance", "U": "Understanding", "I": "Integrity",
            "D": "Disclosure", "E": "Equity"
        }
        
        for principle in ["G", "U", "I", "D", "E"]:
            print(f"\n📋 {principle_names[principle]}")
            print("-" * 40)
            
            principle_scores = []
            for question in questions[principle]:
                print(f"\n{question}")
                while True:
                    try:
                        score_input = input("Rate (1-5): ")
                        if score_input.isdigit():
                            score = int(score_input)
                            if 1 <= score <= 5:
                                notes = input("Additional notes (optional): ") or ""
                                principle_scores.append({
                                    "question": question,
                                    "score": score,
                                    "notes": notes
                                })
                                break
                        print("Please enter a number between 1-5.")
                    except KeyboardInterrupt:
                        print("\nAssessment cancelled.")
                        sys.exit(0)
                    except Exception as e:
                        print("Please enter a valid number.")
            
            self.responses["scores"][principle] = principle_scores
    
    def calculate_scores(self):
        scores = {}
        
        for principle, questions in self.responses["scores"].items():
            principle_scores = [q["score"] for q in questions]
            avg_score = sum(principle_scores) / len(principle_scores)
            
            scores[principle] = {
                "average_score": round(avg_score, 2),
                "questions": questions,
                "maturity_level": self.get_maturity_level(avg_score),
                "strengths": [q["question"] for q in questions if q["score"] >= 4],
                "weaknesses": [q["question"] for q in questions if q["score"] <= 2]
            }
        
        # Overall maturity
        overall_avg = sum([s["average_score"] for s in scores.values()]) / 5
        scores["overall"] = {
            "average_score": round(overall_avg, 2),
            "maturity_level": self.get_maturity_level(overall_avg),
            "recommendation_priority": self.get_priority_level(overall_avg)
        }
        
        return scores
    
    def get_maturity_level(self, score):
        if score >= 4.5:
            return "Advanced"
        elif score >= 3.5:
            return "Established" 
        elif score >= 2.5:
            return "Developing"
        elif score >= 1.5:
            return "Initial Stage"
        else:
            return "Not Implemented"
    
    def get_priority_level(self, score):
        if score >= 4.0:
            return "Optimization"
        elif score >= 3.0:
            return "Improvement"
        else:
            return "Critical Attention"
    
    def generate_recommendations(self, scores):
        recommendations = {
            "G": {
                "low": [
                    "Establish clear AI governance committee with defined responsibilities",
                    "Create formal AI ethics policy and review processes"
                ],
                "medium": [
                    "Expand governance to include diverse stakeholder representation",
                    "Develop incident response protocol for AI errors"
                ],
                "high": [
                    "Implement continuous governance monitoring",
                    "Establish AI ethics training program for all staff"
                ]
            },
            "U": {
                "low": [
                    "Implement basic explainability features in AI systems",
                    "Create user documentation explaining AI decision processes"
                ],
                "medium": [
                    "Develop comprehensive explanation interfaces",
                    "Create model cards documenting AI capabilities and limitations"
                ],
                "high": [
                    "Deploy advanced interpretability techniques",
                    "Create real-time explanation dashboards"
                ]
            },
            "I": {
                "low": [
                    "Implement basic data validation and quality checks",
                    "Establish system reliability monitoring"
                ],
                "medium": [
                    "Develop comprehensive testing protocols",
                    "Implement adversarial testing for system robustness"
                ],
                "high": [
                    "Deploy advanced anomaly detection systems",
                    "Implement continuous integrity validation"
                ]
            },
            "D": {
                "low": [
                    "Clearly mark all AI-generated content and interactions",
                    "Create basic disclosure statements for AI systems"
                ],
                "medium": [
                    "Develop comprehensive transparency frameworks",
                    "Create detailed system capability disclosures"
                ],
                "high": [
                    "Establish full transparency dashboards",
                    "Implement real-time system performance disclosures"
                ]
            },
            "E": {
                "low": [
                    "Implement basic bias testing across demographic groups",
                    "Establish fairness monitoring for key decisions"
                ],
                "medium": [
                    "Develop comprehensive fairness testing protocols",
                    "Implement regular bias audits (quarterly recommended)"
                ],
                "high": [
                    "Deploy continuous fairness monitoring systems",
                    "Establish advanced bias mitigation techniques"
                ]
            }
        }
        
        recs = {}
        for principle in ["G", "U", "I", "D", "E"]:
            if principle in scores:
                score = scores[principle]["average_score"]
                
                if score <= 2.5:
                    priority = "low"
                elif score <= 3.5:
                    priority = "medium"
                else:
                    priority = "high"
                
                recs[principle] = {
                    "priority": priority,
                    "actions": recommendations[principle][priority],
                    "current_score": score,
                    "target_score": min(5.0, score + 1.0)
                }
        
        return recs
    
    def generate_action_plan(self, recommendations):
        action_plan = {
            "immediate_actions": [],
            "short_term_goals": [], 
            "long_term_improvements": []
        }
        
        for principle, data in recommendations.items():
            if data["current_score"] < 2.5:
                action_plan["immediate_actions"].extend(data["actions"])
            elif data["current_score"] < 3.5:
                action_plan["short_term_goals"].extend(data["actions"])
            else:
                action_plan["long_term_improvements"].extend(data["actions"])
        
        return action_plan
    
    def run_complete_assessment(self):
        print("🚀 Starting GUIDE AI Ethics Assessment...")
        
        # Step 1: Situation selection and questionnaire
        self.select_situation()
        self.conduct_interview()
        
        # Step 2: Score assessment
        scores = self.calculate_scores()
        
        # Step 3: Generate recommendations
        recommendations = self.generate_recommendations(scores)
        
        # Step 4: Generate action plan
        action_plan = self.generate_action_plan(recommendations)
        
        # Step 5: Create final report
        report = {
            "assessment_summary": {
                "situation": self.situation_type,
                "overall_maturity": scores["overall"]["maturity_level"],
                "overall_score": scores["overall"]["average_score"],
                "priority_level": scores["overall"]["recommendation_priority"]
            },
            "principle_scores": {k: v for k, v in scores.items() if k != "overall"},
            "recommendations": recommendations,
            "action_plan": action_plan
        }
        
        return report