from pathlib import Path
from typing import List

import pandoc
from docutils import nodes
from docutils.core import publish_doctree
from docutils.parsers.rst import directives
from robot.running import TestCase
from robot.testdoc import JsonConverter, TestSuiteFactory
from sphinx.util.docutils import SphinxDirective
from sphinx.addnodes import desc, desc_content, desc_name, desc_signature


class SphinxJsonConverter(JsonConverter):
    def _convert_test(self, test: TestCase):
        ret_test = super()._convert_test(test)
        ret_test['lineno'] = test.lineno
        ret_test['source'] = test.source
        return ret_test


def option_to_bool(option) -> bool:
    """_summary_

    Args:
        option (_type_): _description_

    Returns:
        bool: _description_
    """
    value = directives.choice(option, ['yes', 'no', 'true', 'false'])
    match value:
        case "yes" | "true":
            return True
        case "no" | "false":
            return False
        case _:
            return None


class RfTestDoc(SphinxDirective):
    """A directive for RF TestDocs!"""
    required_arguments = 1
    optional_arguments = 2
    option_spec = {
        "recursive": option_to_bool,
        "skip_suite_title": option_to_bool,
    }

    def handle_test(self, test: dict) -> List[nodes.Node]:
        """_summary_

        Args:
            test (dict): _description_

        Returns:
            nodes.Node: _description_
        """
        desc_node = desc(classes=["robot", "tests"])

        test_id = nodes.make_id(test.get('fullName'))
        test_name = test.get('name')
        # _toc_name is marked as:
        # Private attributes for ToC generation. Will be modified or removed
        # without notice.
        # sphinx/directives/__init__.py Line: ~268
        # But is needed so it gets added to the ToC from:
        # sphinx/environment/collectors/toctree.py Line: ~136
        desc_sig_node = desc_signature(
            ids=[test_id], classes=['test_case_name'], _toc_name=test_name)
        desc_sig_node += desc_name(text=test_name)
        desc_node += desc_sig_node

        desc_content_node = desc_content()

        if test.get('lineno') and test.get('source'):
            source: Path = test['source']
            line_no = test['lineno']
            relative_src = source.relative_to(self.config.rf_test_doc_root)
            link_url = f"{self.config.rf_test_doc_root_url}/{relative_src}#L{line_no}"
            ln_link = nodes.reference(refuri=link_url, text="Test Case source")
            ln_link["target"] = "_blank"  # or ln_link["classes"] = ["external"]
            ln_para = nodes.paragraph()
            ln_para += ln_link
            desc_content_node += ln_para

        if test.get('tags'):
            tags_para = nodes.paragraph()
            tag_title = nodes.inline(classes=[
                "test_tag_title"])
            tag_title += nodes.Text("Tags:")
            tags_para += tag_title

            for i, tag in enumerate(test.get('tags', [])):
                if i > 0:
                    tags_para += nodes.Text("; ")
                else:
                    tags_para += nodes.Text(" ")
                highlight = any(
                    item in tag for item in self.config.rf_test_highligth_tags)
                if highlight:
                    highlighted_tag = nodes.inline(classes=[
                        "test_tag", "test_tag_highlight"])
                    highlighted_tag += nodes.Text(tag)
                    tags_para += highlighted_tag
                else:
                    normal_tag = nodes.inline(classes=[
                        "test_tag"])
                    normal_tag += nodes.Text(tag)
                    tags_para += normal_tag

            desc_content_node += tags_para

        if test.get('doc'):
            doc_para = nodes.paragraph()
            doc_pan = pandoc.read(source=test.get('doc'), format="html")
            doc_rst = pandoc.write(doc=doc_pan, format="rst")
            doc_tree = publish_doctree(doc_rst)
            doc_para.extend(doc_tree.children)
            desc_content_node += doc_para

        desc_node += desc_content_node

        return desc_node

    def handle_suite(self, suite: dict, recursive: bool=True,
                     skip_suite_title: bool=False) -> nodes.Node:
        """_summary_

        Args:
            suite (dict): _description_
            recursive (bool, optional): _description_. Defaults to True.
            skip_suite_title (bool, optional): _description_. Defaults to False.

        Returns:
            nodes.Node: _description_
        """
        section_id = nodes.make_id(suite.get('fullName'))
        suite_section = nodes.section(ids=[section_id])

        if not skip_suite_title:
            suite_section += nodes.title(text=suite.get('name'))

        if suite.get('doc'):
            # section_id = nodes.make_id(f"{suite.get('fullName')} suite doc")
            # doc_section = nodes.section(ids=[section_id])
            # doc_section += nodes.title(text="Suite Documentation")
            doc_pan = pandoc.read(source=suite.get('doc'), format="html")
            doc_rst = pandoc.write(doc=doc_pan, format="rst")
            doc_tree = publish_doctree(doc_rst)
            # doc_section.extend(doc_tree.children)
            suite_section.extend(doc_tree)

        if suite.get('tests'):
            suite_section += nodes.transition()
            section_id = nodes.make_id(f"{suite.get('fullName')} test cases")
            test_section = nodes.section(ids=[section_id])
            test_section += nodes.title(text="Test Cases")
            for child_test in suite.get('tests', []):
                test_section += self.handle_test(child_test)
            suite_section += test_section

        if recursive and suite.get('suites'):
            section_id = nodes.make_id(f"{suite.get('fullName')} Child Suites")
            child_suite_section = nodes.section(ids=[section_id])
            child_suite_section += nodes.paragraph()
            for child_suite in suite.get('suites', []):
                child_suite_section += self.handle_suite(child_suite)
            suite_section += child_suite_section

        return suite_section

    def run(self) -> List[nodes.Node]:
        suite_root = Path(self.config.rf_test_doc_root) / self.arguments[0]
        recursive = self.options.get('recursive', True)
        skip_suite_title = self.options.get('skip_suite_title', True)
        raw_suite = TestSuiteFactory([suite_root])
        suite = SphinxJsonConverter().convert(raw_suite)
        return [self.handle_suite(
            suite, recursive=recursive, skip_suite_title=skip_suite_title)]
