"""StackExchange tap class."""

from __future__ import annotations

import sys

from singer_sdk import Stream, Tap
from singer_sdk import typing as th

from tap_stackexchange import streams

if sys.version_info >= (3, 12):
    from typing import override
else:
    from typing_extensions import override

FILTER_ID = "!6VvPDzOeLJfUL"


class TapStackExchange(Tap):
    """Singer tap for the StackExchange API."""

    name = "tap-stackexchange"

    config_jsonschema = th.PropertiesList(
        th.Property(
            "key",
            th.StringType,
            required=False,
            description="Pass this to receive a higher request quota",
        ),
        th.Property(
            "filter",
            th.StringType,
            required=False,
            description="Custom API filter to apply to all requests",
            default=FILTER_ID,
        ),
        th.Property(
            "site",
            th.StringType,
            default="stackoverflow.com",
            description="StackExchange site",
        ),
        th.Property(
            "tags",
            th.ArrayType(th.StringType),  # ty: ignore[invalid-argument-type]
            default=[],
            description="Question tags",
        ),
        th.Property(
            "start_date",
            th.IntegerType,
            description="The earliest record date to sync",
        ),
    ).to_dict()

    @override
    def discover_streams(self) -> list[Stream]:
        return [
            streams.Questions(tap=self),
            streams.QuestionAnswers(tap=self),
            streams.QuestionComments(tap=self),
            streams.Tags(tap=self),
            streams.TagSynonyms(tap=self),
            streams.TopAnswerers(tap=self),
            streams.TopAskers(tap=self),
        ]
