import unittest
import sys
sys.path.append(".")

from hyperparse import parse_string

class TestParseString(unittest.TestCase):
    def test_parse_string_case1(self):
        s1 = "a=1,b,c=[1,2,3],d=4,e=3.2,f=itud"
        expected_dict1 = {"a": 1, "b": None, "c": [1, 2, 3], "d": 4, "e": 3.2, "f": "itud"}
        result_dict1 = parse_string(s1)
        self.assertDictEqual(result_dict1, expected_dict1)

    def test_parse_string_case2(self):
        s2 = "x=[1,2,3],y=hello,z=none"
        expected_dict2 = {"x": [1, 2, 3], "y": "hello", "z": None}
        result_dict2 = parse_string(s2)
        self.assertDictEqual(result_dict2, expected_dict2)

    def test_parse_string_case3(self):
        s3 = "a=1.23,b=False,c=[1,2,3],d=hello,e=None"
        expected_dict3 = {"a": 1.23, "b": False, "c": [1, 2, 3], "d": "hello", "e": None}
        result_dict3 = parse_string(s3)
        self.assertDictEqual(result_dict3, expected_dict3)

    def test_parse_string_case4(self):
        s4 = ""
        expected_dict4 = {}
        result_dict4 = parse_string(s4)
        self.assertDictEqual(result_dict4, expected_dict4)

    def test_parse_string_case5(self):
        s5 = "a,b,c"
        expected_dict5 = {"a": None, "b": None, "c": None}
        result_dict5 = parse_string(s5)
        self.assertDictEqual(result_dict5, expected_dict5)

    def test_parse_string_case6(self):
        s6 = "a='hello',b=123.45,c=True,d=None"
        expected_dict6 = {"a": "hello", "b": 123.45, "c": True, "d": None}
        result_dict6 = parse_string(s6)
        self.assertDictEqual(result_dict6, expected_dict6)

    def test_parse_string_case7(self):
        s7 = "a=[1,2,3],b=[4,5,6],c=[7,8,9]"
        expected_dict7 = {"a": [1, 2, 3], "b": [4, 5, 6], "c": [7, 8, 9]}
        result_dict7 = parse_string(s7)
        self.assertDictEqual(result_dict7, expected_dict7)

    def test_parse_string_case8(self):
        s8 = "a=1,b=2,c=3,d=4"
        expected_dict8 = {"a": 1, "b": 2, "c": 3, "d": 4}
        result_dict8 = parse_string(s8)
        self.assertDictEqual(result_dict8, expected_dict8)

    def test_parse_string_case9(self):
        s6 = "a=[1.2,'hello',3.4]"
        expected_dict6 = {"a": [1.2, "hello", 3.4]}
        result_dict6 = parse_string(s6)
        self.assertDictEqual(result_dict6, expected_dict6)

    def test_scientific_notation(self):
        s = "a=1e-5,b=2.5E10,c=-3.14e2"
        expected = {"a": 1e-5, "b": 2.5e10, "c": -314.0}
        self.assertDictEqual(parse_string(s), expected)

    def test_negative_numbers(self):
        s = "a=-1,b=-3.14,c=-100"
        expected = {"a": -1, "b": -3.14, "c": -100}
        self.assertDictEqual(parse_string(s), expected)

    def test_nested_list(self):
        s = "a=[[1,2],[3,4]],b=[1,[2,3]]"
        expected = {"a": [[1, 2], [3, 4]], "b": [1, [2, 3]]}
        self.assertDictEqual(parse_string(s), expected)

    def test_dict_value(self):
        s = "a={x:1,y:2},b={name:hello}"
        expected = {"a": {"x": 1, "y": 2}, "b": {"name": "hello"}}
        self.assertDictEqual(parse_string(s), expected)

    def test_nested_structure(self):
        s = "a={x:[1,2],y:{z:3}}"
        expected = {"a": {"x": [1, 2], "y": {"z": 3}}}
        self.assertDictEqual(parse_string(s), expected)

    def test_empty_list(self):
        s = "a=[]"
        expected = {"a": []}
        self.assertDictEqual(parse_string(s), expected)

    def test_empty_dict(self):
        s = "a={}"
        expected = {"a": {}}
        self.assertDictEqual(parse_string(s), expected)

    def test_attr_access(self):
        usermode = parse_string('a=1,b=hello,c=[1,2,3]')
        self.assertEqual(usermode.a, 1)
        self.assertEqual(usermode.b, 'hello')
        self.assertEqual(usermode.c, [1, 2, 3])

    def test_nested_attr_access(self):
        cfg = parse_string('model={name:bert,layers:12}')
        self.assertEqual(cfg.model.name, 'bert')
        self.assertEqual(cfg.model.layers, 12)

    def test_hasall(self):
        usermode = parse_string('a=1,b=2,c=3')
        self.assertTrue(usermode.hasall('a'))
        self.assertTrue(usermode.hasall('a', 'b'))
        self.assertTrue(usermode.hasall('a', 'b', 'c'))
        self.assertFalse(usermode.hasall('a', 'x'))
        self.assertFalse(usermode.hasall('x', 'y'))

    def test_hasany(self):
        usermode = parse_string('a=1,b=2')
        self.assertTrue(usermode.hasany('a'))
        self.assertTrue(usermode.hasany('a', 'x'))
        self.assertTrue(usermode.hasany('x', 'a'))
        self.assertFalse(usermode.hasany('x'))
        self.assertFalse(usermode.hasany('x', 'y'))

if __name__ == '__main__':
    unittest.main()
