import os
import inspect
import argparse
import ctypes
import sys

global var_list, str_list
var_list = {}
str_list = {}


class AttrDict(dict):
    """Dict that supports attribute access (d.key) in addition to d['key']"""
    def __getattr__(self, key):
        try:
            value = self[key]
            if isinstance(value, dict) and not isinstance(value, AttrDict):
                value = AttrDict(value)
                self[key] = value
            return value
        except KeyError:
            raise AttributeError(f"'{type(self).__name__}' has no attribute '{key}'")

    def hasany(self, *keys):
        """Check if any key exists: usermode.hasany('a', 'b')"""
        return any(key in self for key in keys)

    def hasall(self, *keys):
        """Check if all keys exist: usermode.hasall('a', 'b')"""
        return all(key in self for key in keys)

    def __setattr__(self, key, value):
        self[key] = value

    def __delattr__(self, key):
        try:
            del self[key]
        except KeyError:
            raise AttributeError(f"'{type(self).__name__}' has no attribute '{key}'")

def convert_float(s):
    try:
        return float(s)
    except:
        return None

def is_int(s):
    """Check if string represents an integer (including negative)"""
    if not s:
        return False
    if s[0] == '-':
        return s[1:].isdigit() if len(s) > 1 else False
    return s.isdigit()

def parse_value(value):
    value = value.strip()
    if is_int(value):
        return int(value)
    elif convert_float(value) is not None:
        return float(value)
    elif value.lower() == "none":
        return None
    elif value.lower() == "true":
        return True
    elif value.lower() == "false":
        return False
    elif value == "":
        return None
    else:
        return value.strip("'\"")


def find_matching_bracket(s, start, open_char, close_char):
    """Find the matching closing bracket, handling nesting"""
    depth = 1
    i = start + 1
    while i < len(s) and depth > 0:
        if s[i] == open_char:
            depth += 1
        elif s[i] == close_char:
            depth -= 1
        i += 1
    return i - 1 if depth == 0 else -1

def parse_list(s):
    """Parse a list string like [1,2,[3,4]] into a Python list"""
    s = s.strip()
    if not s.startswith('[') or not s.endswith(']'):
        return None
    s = s[1:-1]  # Remove outer brackets
    if not s.strip():
        return []

    result = []
    i = 0
    current = ""

    while i < len(s):
        c = s[i]
        if c == '[':
            end = find_matching_bracket(s, i, '[', ']')
            if end == -1:
                return None
            result.append(parse_list(s[i:end+1]))
            i = end + 1
            if i < len(s) and s[i] == ',':
                i += 1
        elif c == '{':
            end = find_matching_bracket(s, i, '{', '}')
            if end == -1:
                return None
            result.append(parse_dict_value(s[i:end+1]))
            i = end + 1
            if i < len(s) and s[i] == ',':
                i += 1
        elif c == ',':
            if current.strip():
                result.append(parse_value(current))
            current = ""
            i += 1
        else:
            current += c
            i += 1

    if current.strip():
        result.append(parse_value(current))

    return result

def parse_dict_value(s):
    """Parse a dict string like {x:1,y:2} into a Python dict"""
    s = s.strip()
    if not s.startswith('{') or not s.endswith('}'):
        return None
    s = s[1:-1]  # Remove outer braces
    if not s.strip():
        return AttrDict()

    result = AttrDict()
    i = 0

    while i < len(s):
        # Skip whitespace
        while i < len(s) and s[i] in ' \t':
            i += 1
        if i >= len(s):
            break

        # Parse key
        key_start = i
        while i < len(s) and s[i] not in ':,}':
            i += 1
        key = s[key_start:i].strip().strip("'\"")

        if i >= len(s) or s[i] != ':':
            result[key] = None
            if i < len(s) and s[i] == ',':
                i += 1
            continue
        i += 1  # Skip ':'

        # Skip whitespace
        while i < len(s) and s[i] in ' \t':
            i += 1

        # Parse value
        if i < len(s) and s[i] == '[':
            end = find_matching_bracket(s, i, '[', ']')
            if end == -1:
                return None
            result[key] = parse_list(s[i:end+1])
            i = end + 1
        elif i < len(s) and s[i] == '{':
            end = find_matching_bracket(s, i, '{', '}')
            if end == -1:
                return None
            result[key] = parse_dict_value(s[i:end+1])
            i = end + 1
        else:
            val_start = i
            while i < len(s) and s[i] not in ',}':
                i += 1
            result[key] = parse_value(s[val_start:i])

        # Skip comma
        if i < len(s) and s[i] == ',':
            i += 1

    return result

def parse_string(s):
    if type(s) is tuple:
        s = ",".join(s)
    if not s or not s.strip():
        return AttrDict()

    d = AttrDict()
    i = 0

    while i < len(s):
        # Skip whitespace
        while i < len(s) and s[i] in ' \t':
            i += 1
        if i >= len(s):
            break

        # Parse key
        key_start = i
        while i < len(s) and s[i] not in '=,':
            i += 1
        key = s[key_start:i].strip()

        if not key:
            i += 1
            continue

        # No value case (just key)
        if i >= len(s) or s[i] == ',':
            d[key] = None
            if i < len(s):
                i += 1
            continue

        i += 1  # Skip '='

        # Skip whitespace
        while i < len(s) and s[i] in ' \t':
            i += 1

        # Parse value
        if i < len(s) and s[i] == '[':
            end = find_matching_bracket(s, i, '[', ']')
            if end == -1:
                d[key] = None
                continue
            d[key] = parse_list(s[i:end+1])
            i = end + 1
        elif i < len(s) and s[i] == '{':
            end = find_matching_bracket(s, i, '{', '}')
            if end == -1:
                d[key] = None
                continue
            d[key] = parse_dict_value(s[i:end+1])
            i = end + 1
        else:
            val_start = i
            while i < len(s) and s[i] != ',':
                i += 1
            d[key] = parse_value(s[val_start:i])

        # Skip comma
        if i < len(s) and s[i] == ',':
            i += 1

    return d

#s = "a=1,b,c=[1,2,3],d=4,e=3.2,f=itud,g=False"
def parse_env(name):
    env_vars = os.environ
    value = env_vars[name]
    result = parse_string(value)
    var_list[name] = result
    str_list[name] = value

"""for key, value in env_vars.items():
    result = parse_string(value)
    if key not in globals():
        globals()[key] = result
        globals()[key + "_str"] = value"""


def parse(name = ""):
    mode = {}
    if name not in var_list:
        if type(name) is str and name in os.environ:
            parse_env(name)
        elif f"--{name}" in sys.argv:
            parse_arg(name)
        elif len(sys.argv) > 1:
            parse_arg(name)
        else:
            var_list[name] = {}
    mode = var_list[name]

    if name not in str_list:
        mode_str = ""
    else:
        mode_str = str_list[name]
    return mode, mode_str

def get_hyper_str(name):
    if name not in str_list:
        parse(name)
    if name not in str_list:
        return None
    else:
        return str_list[name]

def parse_arg(name):
    if type(name) is int:
        try:
            value = sys.argv[name]
        except IndexError:
            print(f"You do not have {name} arguments")
            value = ""
    elif name != "":
        parser = argparse.ArgumentParser()
        parser.add_argument(f"--{name}", type=str, default="")
        args, unknown_args = parser.parse_known_args()
        value = getattr(args, name)
        sys.argv = [sys.argv[0]] + unknown_args
    else:
        value = sys.argv[1]
    result = parse_string(value)
    var_list[name] = result
    str_list[name] = value
    

def reset_hyper(hyper, pargs=None):
    if type(hyper) is str:
        hyper = parse(hyper)[0]
    if pargs is None:
        frame = inspect.currentframe().f_back
        pargs = frame.f_locals
    for k in hyper:
        v = hyper[k]
        if type(pargs) is dict:
            pargs[k] = v
        elif hasattr(pargs, '__setitem__'):
            # Python 3.13+ returns FrameLocalsProxy which supports item assignment
            pargs[k] = v
        else:
            setattr(pargs, k, v)
    if 'frame' in locals() and sys.version_info < (3, 13):
        ctypes.pythonapi.PyFrame_LocalsToFast(ctypes.py_object(frame), ctypes.c_int(0))
    

