"""
This experiment processes the "aggregators_binary" dataset which is a large dataset of organic molecules
which are divided into the two classes "aggregators" and "non-aggregators"

CHANGELOG

0.1.0 - 23.02.2023 - initial version
"""
import os
import pathlib
import typing as t

# from pycomex.experiment import SubExperiment
# from pycomex.util import Skippable
from pycomex.functional.experiment import Experiment
from pycomex.utils import folder_path, file_namespace


PATH = pathlib.Path(__file__).parent.absolute()
ASSETS_PATH = os.path.join(PATH, 'assets')

# The vgd file share provider from which to download the CSV file to be used as the source for the VGD
# conversion.
FILE_SHARE_PROVIDER: str = 'main'
# This may be one of the following two things:
# 1. A valid absolute file path on the local system pointing to a CSV file to be used as the source for
#    the VGD conversion
# 2. A valid relative path to a CSV file stashed on the given vgd file share provider which will be
#    downloaded first and then processed.
CSV_FILE_NAME: str = os.path.join(ASSETS_PATH, 'aggregators_binary.csv')
# Optionally, this may define the string name of the CSV column which contains the integer index
# associated with each dataset element. If this is not given, then integer indices will be randomly
# generated for each element in the final VGD
INDEX_COLUMN_NAME: t.Optional[str] = None
# This has to be the string name of the CSV column which contains the SMILES string representation of
# the molecule.
SMILES_COLUMN_NAME: str = 'smiles'
# This has to be the string name of the CSV column which contains the target value
TARGET_TYPE: str = 'classification'
TARGET_COLUMN_NAMES: t.List[str] = ['aggregator', 'nonaggregator']

# == DATASET PARAMETERS ==
# These parameters control aspects of the visual graph dataset creation process

# The name given to the visual graph dataset folder which will be created.
DATASET_NAME: str = 'aggregators_binary'
# The width and height of the molecule visualization PNGs in pixels.
IMAGE_WIDTH: int = 1000
IMAGE_HEIGHT: int = 1000
# This dict will be converted into the .meta.yml file which will be added to the final visual graph dataset
# folder. This is an optional file, which can add additional meta information about the entire dataset
# itself. Such as documentation in the form of a description of the dataset etc.
DATASET_META: t.Optional[dict] = {
    'version': '0.1.0',
    # A list of strings where each element is a description about the changes introduced in a newer
    # version of the dataset.
    'changelog': [
        '0.1.0 - 29.01.2023 - initial version'
    ],
    # A general description about the dataset, which gives a general overview about where the data was
    # sampled from, what the input features look like, what the prediction target is etc...
    'description': (
        'large dataset consisting of organic compounds which are divided into two classes: aggregators '
        'and non-aggregators.'
    ),
    # A list of informative strings (best case containing URLS) which are used as references for the
    # dataset. This could for example be a reference to a paper where the dataset was first introduced
    # or a link to site where the raw data can be downloaded etc.
    'references': [
        'Library used for the processing and visualization of molecules. https://www.rdkit.org/',
    ],
    # A small description about how to interpret the visualizations which were created by this dataset.
    'visualization_description': (
        'Molecular graphs generated by RDKit based on the SMILES representation of the molecule.'
    ),
    # A dictionary, where the keys should be the integer indices of the target value vector for the dataset
    # and the values should be string descriptions of what the corresponding target value is about.
    'target_descriptions': {
        0: 'one-hot: aggregator class',
        1: 'one-hot: non-aggregator class'
    }
}
GRAPH_METADATA_CALLBACKS = {
    'name': lambda mol, data: data['name'],
    'label': lambda mol, data: data['label'],
    'smiles': lambda mol, data: data['smiles'],
}

# == EVALUATION PARAMETERS ==
EVAL_LOG_STEP = 1000
NUM_BINS = 10
PLOT_COLOR = 'gray'

# == EXPERIMENT PARAMETERS ==
EXPERIMENT_PATH = os.path.join(PATH, 'generate_molecule_dataset_from_csv.py')
NAMESPACE = 'results/generate_molecule_dataset_from_csv_aggregators_binary'
BASE_PATH = PATH

experiment = Experiment.extend(
    'generate_molecule_dataset_from_csv.py',
    base_path=folder_path(__file__),
    namespace=file_namespace(__file__),
    glob=globals(),
)


# ~ Adding filters to the dataset processing step
# By adding these specific filters to the pre-processing of the dataset we implement the same processing
# steps described in the original paper which introduces this dataset.

def max_graph_size(mol, data):
    """
    We want to filter the very big molecules because we will likely not be able to properly visualize
    those for the explanations anyways.
    """
    return len(mol.GetAtoms()) >= 100


def is_charged(mol, data):
    smiles = data['smiles']
    return '+' in smiles or '-' in smiles


def is_adjoined_mixture(mol, data):
    smiles = data['smiles']
    return '.' in smiles


# Here we add our own custom filter to the list of filters that are applied on the dataset
@experiment.hook('modify_filter_callbacks')
def add_filters(e: Experiment, filter_callbacks: t.List[t.Callable]):
    # filter_callbacks.append(is_charged)
    filter_callbacks.append(is_adjoined_mixture)
    return filter_callbacks


experiment.run_if_main()
