import pandas as pd
import numpy as np

class EthicalAIAuditor:

    def __init__(self):
        # Default configuration of GUIDE principles
        self.ai_identification = True
        self.require_human_review = True
        self.debias_threshold = 0.20   # max allowed disparity
        self.audit_log = []

    # ------------------------------
    # GOVERNANCE (G)
    # ------------------------------
    def governance_check(self):
        issues = []

        if not self.ai_identification:
            issues.append("AI system is not labeled or disclosed to users.")

        if not self.require_human_review:
            issues.append("Human oversight missing for critical decisions.")

        if len(issues) == 0:
            return {"status": "PASS", "issues": []}

        return {"status": "FAIL", "issues": issues}

    # ------------------------------
    # FAIRNESS ANALYSIS (E)
    # ------------------------------
    def run_fairness_analysis(self, X_test, y_true, sensitive_attributes):
        """
        Performs a simple fairness audit:
        - calculates selection rates per group
        - identifies disparities
        - recommends actions
        """

        report = {}
        recommendations = []

        for col in sensitive_attributes.columns:
            rates = X_test.join(y_true).groupby(col)['qualified'].mean()
            disparity = rates.max() - rates.min()

            report[col] = {
                "rates": rates.to_dict(),
                "disparity": float(disparity)
            }

            # Generate recommendation
            if disparity > self.debias_threshold:
                recommendations.append(
                    f"High disparity detected for '{col}'. Consider removing or reweighting '{col}' in your model."
                )

        report["recommendations"] = recommendations
        return report

    # ------------------------------
    # DIGNITY CHECK (D)
    # ------------------------------
    def dignity_check(self):
        """
        Ensures the model is not producing disrespectful or harmful outcomes.
        Example placeholder module (can be expanded later).
        """

        return {
            "status": "PASS",
            "notes": "No dignity violations detected in this simplified version."
        }

    # ------------------------------
    # UNIVERSAL DESIGN (U)
    # ------------------------------
    def universal_design_check(self, df):
        """
        Basic check to ensure dataset includes diverse representations.
        """

        issues = []

        for col in df.columns:
            if df[col].nunique() < 2:
                issues.append(f"Column '{col}' lacks diversity.")

        if issues:
            return {"status": "FAIL", "issues": issues}

        return {"status": "PASS", "issues": []}

    # ------------------------------
    # MAIN AUDIT RUNNER
    # ------------------------------
    def run_full_audit(self, X_test, y_true, sensitive_attributes):
        """
        Runs ALL GUIDE modules:
        - Governance
        - Universal Design
        - Identification
        - Dignity
        - Equity
        """

        return {
            "governance": self.governance_check(),
            "universal_design": self.universal_design_check(X_test),
            "dignity": self.dignity_check(),
            "equity": self.run_fairness_analysis(X_test, y_true, sensitive_attributes)
        }
