"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynchronousGlueJob = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const pre_bundled_layer_1 = require("../common/pre-bundled-layer");
const aws_glue_alpha_1 = require("@aws-cdk/aws-glue-alpha");
/**
 * SynchronousGlueJob Construct to start an AWS Glue Job execution and wait for completion during CDK deploy
 */
class SynchronousGlueJob extends constructs_1.Construct {
    /**
     * Constructs a new instance of the DataGenerator class
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {JobProps} props the SynchronousGlueJob properties
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        const stack = aws_cdk_lib_1.Stack.of(this);
        const glueJob = new aws_glue_alpha_1.Job(scope, 'SynchronousGlueJob', props);
        aws_cdk_lib_1.Tags.of(glueJob).add('for-use-with', 'synchronous-glue-job');
        //Lambda policy to allow starting a Glue job
        const lambdaCRPolicy = [new aws_iam_1.PolicyStatement({
                resources: [
                    stack.formatArn({
                        region: aws_cdk_lib_1.Aws.REGION,
                        account: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        service: 'glue',
                        resource: 'job',
                        resourceName: '*',
                    }),
                ],
                conditions: { StringEquals: { 'aws:ResourceTag/for-use-with': 'synchronous-glue-job' } },
                actions: [
                    'glue:StartJobRun',
                    'glue:GetJobRun',
                    'glue:BatchStopJobRun',
                ],
            })];
        // AWS Lambda function for the AWS CDK Custom Resource responsible to start the job
        const glueJobStartFn = new pre_bundled_function_1.PreBundledFunction(this, 'StartFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'synchronous-glue-job/resources/lambdas',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: lambdaCRPolicy,
            handler: 'lambda.on_event',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        // AWS Lambda function for the AWS CDK Custom Resource responsible to wait for crawler completion
        const glueJobWaitFn = new pre_bundled_function_1.PreBundledFunction(this, 'WaitFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'synchronous-glue-job/resources/lambdas',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: lambdaCRPolicy,
            handler: 'lambda.is_complete',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        // Create an AWS CDK Custom Resource Provider for starting the source crawler and waiting for completion
        const glueJobStartWaitCRP = new custom_resources_1.Provider(this, 'SynchronousGlueJobCRP', {
            onEventHandler: glueJobStartFn,
            isCompleteHandler: glueJobWaitFn,
            queryInterval: aws_cdk_lib_1.Duration.seconds(120),
            totalTimeout: props.timeout,
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        // Create an AWS CDK Custom Resource for starting the source crawler and waiting for completion
        const customResource = new aws_cdk_lib_1.CustomResource(this, 'SynchronousGlueJobCR', {
            serviceToken: glueJobStartWaitCRP.serviceToken,
            properties: {
                JobName: glueJob.jobName,
            },
        });
        this.glueJobLogStream = customResource.getAttString('LogGroupName');
        // Force the dependency because jobName could be known at synth time
        customResource.node.addDependency(glueJob);
    }
}
exports.SynchronousGlueJob = SynchronousGlueJob;
_a = JSII_RTTI_SYMBOL_1;
SynchronousGlueJob[_a] = { fqn: "aws-analytics-reference-architecture.SynchronousGlueJob", version: "2.8.7" };
//# sourceMappingURL=data:application/json;base64,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