"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynchronousAthenaQuery = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const pre_bundled_layer_1 = require("../common/pre-bundled-layer");
/**
 * Execute an Amazon Athena query synchronously during CDK deployment
 */
class SynchronousAthenaQuery extends constructs_1.Construct {
    /**
    * Constructs a new instance of the SynchronousAthenaQuery class
    * @param {Construct} scope the Scope of the CDK Construct
    * @param {string} id the ID of the CDK Construct
    * @param {SynchronousAthenaQueryProps} props the CrawlerStartWait [properties]{@link SynchronousAthenaQueryProps}
    */
    constructor(scope, id, props) {
        super(scope, id);
        const stack = aws_cdk_lib_1.Stack.of(this);
        let athenaQueryStartFnPolicy = [];
        // Add permissions from the Amazon IAM Policy Statements
        props.executionRoleStatements?.forEach((element) => {
            athenaQueryStartFnPolicy.push(element);
        });
        // Add permissions to the Function for starting the query
        athenaQueryStartFnPolicy.push(new aws_iam_1.PolicyStatement({
            resources: [
                stack.formatArn({
                    region: aws_cdk_lib_1.Aws.REGION,
                    account: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    service: 'athena',
                    resource: 'workgroup',
                    resourceName: 'primary',
                }),
            ],
            actions: [
                'athena:StartQueryExecution',
            ],
        }));
        // add permissions to the Function to store result in the result path
        athenaQueryStartFnPolicy.push(new aws_iam_1.PolicyStatement({
            resources: [
                stack.formatArn({
                    region: '',
                    account: '',
                    service: 's3',
                    resource: props.resultPath.bucketName,
                    resourceName: props.resultPath.objectKey,
                }),
                stack.formatArn({
                    region: '',
                    account: '',
                    service: 's3',
                    resource: props.resultPath.bucketName,
                    resourceName: props.resultPath.objectKey + '/*',
                }),
                stack.formatArn({
                    region: '',
                    account: '',
                    service: 's3',
                    resource: props.resultPath.bucketName,
                }),
            ],
            actions: [
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:ListMultipartUploadParts',
                's3:AbortMultipartUpload',
                's3:PutObject',
                's3:CreateBucket',
            ],
        }));
        // AWS Lambda function for the AWS CDK Custom Resource responsible to start query
        const athenaQueryStartFn = new pre_bundled_function_1.PreBundledFunction(this, 'StartFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'synchronous-athena-query/resources/lambdas',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: athenaQueryStartFnPolicy,
            handler: 'lambda.on_event',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        let athenaQueryWaitFnPolicy = [];
        // Add permissions to the Function
        athenaQueryWaitFnPolicy.push(new aws_iam_1.PolicyStatement({
            resources: [
                stack.formatArn({
                    region: aws_cdk_lib_1.Aws.REGION,
                    account: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    service: 'athena',
                    resource: 'workgroup',
                    resourceName: 'primary',
                }),
            ],
            actions: [
                'athena:GetQueryExecution',
                'athena:GetQueryResults',
            ],
        }));
        // AWS Lambda function for the AWS CDK Custom Resource responsible to wait for query completion
        const athenaQueryWaitFn = new pre_bundled_function_1.PreBundledFunction(this, 'WaitFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'synchronous-athena-query/resources/lambdas',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: athenaQueryWaitFnPolicy,
            handler: 'lambda.is_complete',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        /*const providerManagedPolicy = new ManagedPolicy(this, 'providerManagedPolicy', {
          statements: [new PolicyStatement({
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
            resources: [`arn:aws:logs:${Aws.REGION}:${Aws.ACCOUNT_ID}:*`],
            effect: Effect.ALLOW,
          })],
        });
    
        const providerRole = new Role(this, 'providerRole', {
          assumedBy: new ServicePrincipal('lambda.amazonaws.com'),
          managedPolicies: [providerManagedPolicy],
        });*/
        // Create an AWS CDK Custom Resource Provider for starting the source crawler and waiting for completion
        const synchronousAthenaQueryCRP = new custom_resources_1.Provider(this, 'customresourceprovider', {
            onEventHandler: athenaQueryStartFn,
            isCompleteHandler: athenaQueryWaitFn,
            queryInterval: aws_cdk_lib_1.Duration.seconds(10),
            totalTimeout: aws_cdk_lib_1.Duration.minutes(props.timeout || 1),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        synchronousAthenaQueryCRP.node.addDependency(athenaQueryStartFn);
        synchronousAthenaQueryCRP.node.addDependency(athenaQueryWaitFn);
        const resultPathBucket = aws_s3_1.Bucket.fromBucketName(this, 'ResultPathBucket', props.resultPath.bucketName);
        // Create an AWS CDK Custom Resource for starting the source crawler and waiting for completion
        const myCR = new aws_cdk_lib_1.CustomResource(this, 'SynchronousAthenaQueryCR', {
            serviceToken: synchronousAthenaQueryCRP.serviceToken,
            properties: {
                Statement: props.statement,
                ResultPath: resultPathBucket.s3UrlForObject(props.resultPath.objectKey),
            },
        });
        myCR.node.addDependency(synchronousAthenaQueryCRP);
    }
}
exports.SynchronousAthenaQuery = SynchronousAthenaQuery;
_a = JSII_RTTI_SYMBOL_1;
SynchronousAthenaQuery[_a] = { fqn: "aws-analytics-reference-architecture.SynchronousAthenaQuery", version: "2.8.7" };
//# sourceMappingURL=data:application/json;base64,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