import { Bucket } from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
/**
 * The props for S3CrossAccount construct
 */
export interface S3CrossAccountProps {
    /**
     * The S3 Bucket object to grant cross account access.
     * This needs to be a Bucket object and not an IBucket because the construct modifies the Bucket policy
     */
    readonly s3Bucket: Bucket;
    /**
     * The S3 object key to grant cross account access (S3 prefix without the bucket name)
     * @default - Grant cross account for the entire bucket
     */
    readonly s3ObjectKey?: string;
    /**
     * The account ID to grant on the S3 location
     */
    readonly accountId: string;
}
/**
 * This CDK construct grants cross account permissions on an Amazon S3 location.
 * It uses a bucket policy and an Amazon KMS Key policy if the bucket is encrypted with KMS.
 * The cross account permission is granted to the entire account and not to a specific principal in this account.
 * It's the responsibility of the target account to grant permissions to the relevant principals.
 *
 * Note that it uses a Bucket object and not an IBucket because CDK can only add policies to objects managed in the CDK stack.
 *
 * Usage example:
 * ```typescript
 * import * as cdk from 'aws-cdk-lib';
 * import { S3CrossAccount } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new cdk.App();
 * const stack = new cdk.Stack(exampleApp, 'S3CrossAccountStack');
 *
 * const myBucket = new Bucket(stack, 'MyBucket')
 *
 * new S3CrossAccount(stack, 'S3CrossAccountGrant', {
 *   bucket: myBucket,
 *   s3ObjectKey: 'my-data',
 *   accountId: '1234567891011',
 * });
 * ```
 */
export declare class S3CrossAccount extends Construct {
    constructor(scope: Construct, id: string, props: S3CrossAccountProps);
}
