"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.clusterAutoscalerSetup = exports.karpenterSetup = exports.karpenterManifestSetup = exports.setDefaultKarpenterProvisioners = exports.setDefaultManagedNodeGroups = exports.eksClusterSetup = void 0;
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_sqs_1 = require("aws-cdk-lib/aws-sqs");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const utils_1 = require("../utils");
const emr_eks_nodegroup_1 = require("./emr-eks-nodegroup");
const emr_eks_cluster_1 = require("./emr-eks-cluster");
const IamPolicyAlb = require("./resources/k8s/iam-policy-alb.json");
/**
 * @internal
 * Upload podTemplates to the Amazon S3 location used by the cluster.
 * @param {Cluster} cluster the unique ID of the CDK resource
 * @param {Construct} scope The local path of the yaml podTemplate files to upload
 * @param {string} eksAdminRoleArn The admin role of the EKS cluster
 */
function eksClusterSetup(cluster, scope, eksAdminRoleArn) {
    // Add the provided Amazon IAM Role as Amazon EKS Admin
    if (eksAdminRoleArn != undefined) {
        cluster.eksCluster.awsAuth.addMastersRole(aws_iam_1.Role.fromRoleArn(scope, 'AdminRole', eksAdminRoleArn), 'AdminRole');
    }
    // Deploy the Helm Chart for the Certificate Manager. Required for EMR Studio ALB.
    const certManager = cluster.eksCluster.addHelmChart('CertManager', {
        createNamespace: true,
        namespace: 'cert-manager',
        chart: 'cert-manager',
        repository: 'https://charts.jetstack.io',
        version: 'v1.10.1',
        timeout: aws_cdk_lib_1.Duration.minutes(14),
        values: {
            startupapicheck: {
                timeout: '5m'
            },
            installCRDs: true
        }
    });
    //Create service account for ALB and install ALB
    const albPolicyDocument = aws_iam_1.PolicyDocument.fromJson(IamPolicyAlb);
    const albIAMPolicy = new aws_iam_1.Policy(scope, 'AWSLoadBalancerControllerIAMPolicy', { document: albPolicyDocument });
    const albServiceAccount = cluster.eksCluster.addServiceAccount('ALB', {
        name: 'aws-load-balancer-controller',
        namespace: 'kube-system',
    });
    albIAMPolicy.attachToRole(albServiceAccount.role);
    const albService = cluster.eksCluster.addHelmChart('ALB', {
        chart: 'aws-load-balancer-controller',
        repository: 'https://aws.github.io/eks-charts',
        namespace: 'kube-system',
        version: '1.4.6',
        timeout: aws_cdk_lib_1.Duration.minutes(14),
        values: {
            clusterName: cluster.clusterName,
            serviceAccount: {
                name: 'aws-load-balancer-controller',
                create: false,
            },
        },
    });
    albService.node.addDependency(albServiceAccount);
    albService.node.addDependency(certManager);
    // Add the kubernetes dashboard from helm chart
    cluster.eksCluster.addHelmChart('KubernetesDashboard', {
        createNamespace: true,
        namespace: 'kubernetes-dashboard',
        chart: 'kubernetes-dashboard',
        repository: 'https://kubernetes.github.io/dashboard/',
        version: 'v6.0.0',
        timeout: aws_cdk_lib_1.Duration.minutes(2),
        values: {
            fullnameOverride: 'kubernetes-dashboard',
            resources: {
                limits: {
                    memory: '600Mi',
                },
            },
        },
    });
    // Add the kubernetes dashboard service account
    cluster.eksCluster.addManifest('kubedashboard', {
        apiVersion: 'v1',
        kind: 'ServiceAccount',
        metadata: {
            name: 'eks-admin',
            namespace: 'kube-system',
        },
    });
    // Add the kubernetes dashboard cluster role binding
    cluster.eksCluster.addManifest('kubedashboardrolebinding', {
        apiVersion: 'rbac.authorization.k8s.io/v1',
        kind: 'ClusterRoleBinding',
        metadata: {
            name: 'eks-admin',
        },
        roleRef: {
            apiGroup: 'rbac.authorization.k8s.io',
            kind: 'ClusterRole',
            name: 'cluster-admin',
        },
        subjects: [
            {
                kind: 'ServiceAccount',
                name: 'eks-admin',
                namespace: 'kube-system',
            },
        ],
    });
    // Nodegroup capacity needed for all the tooling components including Karpenter
    let EmrEksNodeGroupTooling = { ...emr_eks_nodegroup_1.EmrEksNodegroup.TOOLING_ALL };
    EmrEksNodeGroupTooling.nodeRole = cluster.ec2InstanceNodeGroupRole;
    // Create the Amazon EKS Nodegroup for tooling
    cluster.addNodegroupCapacity('Tooling', EmrEksNodeGroupTooling);
    //IAM role created for the aws-node pod following AWS best practice not to use the EC2 instance role
    const awsNodeRole = new aws_iam_1.Role(scope, 'awsNodeRole', {
        assumedBy: new aws_iam_1.FederatedPrincipal(cluster.eksCluster.openIdConnectProvider.openIdConnectProviderArn, { ...[] }, 'sts:AssumeRoleWithWebIdentity'),
        roleName: `awsNodeRole-${cluster.clusterName}`,
        managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy')],
    });
    // update the aws-node service account with IAM role created for it
    new aws_eks_1.KubernetesManifest(scope, 'awsNodeServiceAccountUpdateManifest', {
        cluster: cluster.eksCluster,
        manifest: [
            {
                apiVersion: 'v1',
                kind: 'ServiceAccount',
                metadata: {
                    name: 'aws-node',
                    namespace: 'kube-system',
                    annotations: {
                        'eks.amazonaws.com/role-arn': awsNodeRole.roleArn,
                    },
                },
            }
        ],
        overwrite: true,
    });
    // Provide the Kubernetes Dashboard URL in AWS CloudFormation output
    new aws_cdk_lib_1.CfnOutput(scope, 'kubernetesDashboardURL', {
        description: 'Access Kubernetes Dashboard via kubectl proxy and this URL',
        value: 'http://localhost:8001/api/v1/namespaces/kubernetes-dashboard/services/https:kubernetes-dashboard:https/proxy/#/login',
    });
}
exports.eksClusterSetup = eksClusterSetup;
/**
 * @internal
 * Method to add the default EKS Managed Nodegroups configured for Spark workloads
 */
function setDefaultManagedNodeGroups(cluster) {
    let EmrEksNodeGroupCritical = { ...emr_eks_nodegroup_1.EmrEksNodegroup.CRITICAL_ALL };
    EmrEksNodeGroupCritical.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('criticalAll', EmrEksNodeGroupCritical);
    let EmrEksNodeGroupsharedDriver = { ...emr_eks_nodegroup_1.EmrEksNodegroup.SHARED_DRIVER };
    EmrEksNodeGroupsharedDriver.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('sharedDriver', EmrEksNodeGroupsharedDriver);
    let EmrEksNodeGroupsharedExecutor = { ...emr_eks_nodegroup_1.EmrEksNodegroup.SHARED_EXECUTOR };
    EmrEksNodeGroupsharedExecutor.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('sharedExecutor', EmrEksNodeGroupsharedExecutor);
    let EmrEksNodeGroupnotebookDriver = { ...emr_eks_nodegroup_1.EmrEksNodegroup.NOTEBOOK_DRIVER };
    EmrEksNodeGroupnotebookDriver.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('notebookDriver', EmrEksNodeGroupnotebookDriver);
    let EmrEksNodeGroupnotebookExecutor = { ...emr_eks_nodegroup_1.EmrEksNodegroup.NOTEBOOK_EXECUTOR };
    EmrEksNodeGroupnotebookExecutor.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('notebookExecutor', EmrEksNodeGroupnotebookExecutor);
}
exports.setDefaultManagedNodeGroups = setDefaultManagedNodeGroups;
/**
 * @internal
 * Method to add the default Karpenter provisioners for Spark workloads
 */
function setDefaultKarpenterProvisioners(cluster) {
    const subnets = cluster.eksCluster.vpc.selectSubnets({
        onePerAz: true,
        subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
    }).subnets;
    subnets.forEach((subnet, index) => {
        let criticalManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/critical-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterCriticalManifest-${index}`, criticalManfifestYAML);
        let sharedDriverManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/shared-driver-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterSharedDriverManifest-${index}`, sharedDriverManfifestYAML);
        let sharedExecutorManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/shared-executor-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterSharedExecutorManifest-${index}`, sharedExecutorManfifestYAML);
        let notebookDriverManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/notebook-driver-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterNotebookDriverManifest-${index}`, notebookDriverManfifestYAML);
        let notebookExecutorManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/notebook-executor-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterNotebookExecutorManifest-${index}`, notebookExecutorManfifestYAML);
    });
}
exports.setDefaultKarpenterProvisioners = setDefaultKarpenterProvisioners;
/**
 * @internal
 * Method to generate the Karpenter manifests from templates and targeted to the specific EKS cluster
 */
function karpenterManifestSetup(clusterName, path, subnet) {
    let manifest = utils_1.Utils.readYamlDocument(path);
    manifest = manifest.replace('{{subnet-id}}', subnet.subnetId);
    manifest = manifest.replace(/(\{{az}})/g, subnet.availabilityZone);
    manifest = manifest.replace('{{cluster-name}}', clusterName);
    let manfifestYAML = manifest.split("---").map((e) => utils_1.Utils.loadYaml(e));
    return manfifestYAML;
}
exports.karpenterManifestSetup = karpenterManifestSetup;
/**
 * @internal
 * Install all the required configurations of Karpenter SQS and Event rules to handle spot and unhealthy instance termination
 * Create a security group to be used by nodes created with karpenter
 * Tags the subnets and VPC to be used by karpenter
 * create a tooling provisioner that will deploy in each of the AZs, one per AZ
 */
function karpenterSetup(cluster, eksClusterName, scope, karpenterVersion) {
    const karpenterInterruptionQueue = new aws_sqs_1.Queue(scope, 'karpenterInterruptionQueue', {
        queueName: eksClusterName,
        retentionPeriod: aws_cdk_lib_1.Duration.seconds(300)
    });
    karpenterInterruptionQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: ['sqs:SendMessage'],
        principals: [new aws_iam_1.ServicePrincipal('sqs.amazonaws.com'), new aws_iam_1.ServicePrincipal('events.amazonaws.com')]
    }));
    new aws_events_1.Rule(scope, 'scheduledChangeRule', {
        eventPattern: {
            source: ['aws.heatlh'],
            detail: ['AWS Health Event']
        },
        targets: [new aws_events_targets_1.SqsQueue(karpenterInterruptionQueue)]
    });
    new aws_events_1.Rule(scope, 'instanceStateChangeRule', {
        eventPattern: {
            source: ['aws.ec2'],
            detail: ['EC2 Instance State-change Notification']
        },
        targets: [new aws_events_targets_1.SqsQueue(karpenterInterruptionQueue)]
    });
    const karpenterNodeRole = new aws_iam_1.Role(cluster, 'karpenter-node-role', {
        assumedBy: new aws_iam_1.ServicePrincipal(`ec2.${cluster.stack.urlSuffix}`),
        managedPolicies: [
            aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'),
            aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'),
            aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'),
            aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
        ],
        roleName: `KarpenterNodeRole-${eksClusterName}`,
    });
    const karpenterControllerPolicyStatementSSM = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: ['ssm:GetParameter', 'pricing:GetProducts'],
        resources: ['*'],
    });
    const karpenterControllerPolicyStatementEC2 = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: [
            'ec2:CreateLaunchTemplate',
            'ec2:DeleteLaunchTemplate',
            'ec2:CreateFleet',
            'ec2:RunInstances',
            'ec2:CreateTags',
            'ec2:TerminateInstances',
            'ec2:DescribeLaunchTemplates',
            'ec2:DescribeInstances',
            'ec2:DescribeSecurityGroups',
            'ec2:DescribeSubnets',
            'ec2:DescribeInstanceTypes',
            'ec2:DescribeInstanceTypeOfferings',
            'ec2:DescribeAvailabilityZones',
        ],
        resources: ['*'],
    });
    const karpenterControllerPolicyStatementIAM = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: ['iam:PassRole'],
        resources: [`arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/KarpenterNodeRole-${eksClusterName}`],
    });
    const karpenterInstanceProfile = new aws_iam_1.CfnInstanceProfile(cluster, 'karpenter-instance-profile', {
        roles: [karpenterNodeRole.roleName],
        instanceProfileName: `karpenterNodeInstanceProfile-${eksClusterName}`,
        path: '/'
    });
    cluster.awsAuth.addRoleMapping(karpenterNodeRole, {
        username: 'system:node:{{EC2PrivateDNSName}}',
        groups: ['system:bootstrappers', 'system:nodes'],
    });
    const karpenterNS = cluster.addManifest('karpenterNS', {
        apiVersion: 'v1',
        kind: 'Namespace',
        metadata: { name: 'karpenter' },
    });
    const karpenterAccount = cluster.addServiceAccount('Karpenter', {
        name: 'karpenter',
        namespace: 'karpenter',
    });
    karpenterAccount.node.addDependency(karpenterNS);
    karpenterAccount.addToPrincipalPolicy(karpenterControllerPolicyStatementSSM);
    karpenterAccount.addToPrincipalPolicy(karpenterControllerPolicyStatementEC2);
    karpenterAccount.addToPrincipalPolicy(karpenterControllerPolicyStatementIAM);
    //Deploy Karpenter Chart
    const karpenterChart = cluster.addHelmChart('Karpenter', {
        chart: 'karpenter',
        release: 'karpenter',
        repository: 'oci://public.ecr.aws/karpenter/karpenter',
        namespace: 'karpenter',
        version: karpenterVersion || emr_eks_cluster_1.EmrEksCluster.DEFAULT_KARPENTER_VERSION,
        timeout: aws_cdk_lib_1.Duration.minutes(14),
        wait: true,
        values: {
            serviceAccount: {
                name: 'karpenter',
                create: false,
                annotations: {
                    'eks.amazonaws.com/role-arn': karpenterAccount.role.roleArn,
                },
            },
            settings: {
                aws: {
                    defaultInstanceProfile: karpenterInstanceProfile.instanceProfileName,
                    clusterName: eksClusterName,
                    clusterEndpoint: cluster.clusterEndpoint,
                    interruptionQueueName: karpenterInterruptionQueue.queueName
                },
            }
        },
    });
    karpenterChart.node.addDependency(karpenterAccount);
    const karpenterInstancesSg = new aws_ec2_1.SecurityGroup(scope, 'karpenterSg', {
        vpc: cluster.vpc,
        allowAllOutbound: true,
        description: 'security group for a karpenter instances',
        securityGroupName: 'karpenterSg',
        disableInlineRules: true,
    });
    aws_cdk_lib_1.Tags.of(karpenterInstancesSg).add('karpenter.sh/discovery', `${eksClusterName}`);
    cluster.clusterSecurityGroup.addIngressRule(karpenterInstancesSg, aws_ec2_1.Port.allTraffic());
    karpenterInstancesSg.addIngressRule(karpenterInstancesSg, aws_ec2_1.Port.allTraffic());
    karpenterInstancesSg.addIngressRule(cluster.clusterSecurityGroup, aws_ec2_1.Port.allTraffic());
    aws_cdk_lib_1.Tags.of(cluster.vpc).add('karpenter.sh/discovery', eksClusterName);
    cluster.vpc.privateSubnets.forEach((subnet) => {
        aws_cdk_lib_1.Tags.of(subnet).add('karpenter.sh/discovery', eksClusterName);
    });
    cluster.vpc.publicSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('karpenter.sh/discovery', eksClusterName));
    const privateSubnets = cluster.vpc.selectSubnets({
        onePerAz: true,
        subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
    }).subnets;
    let listPrivateSubnets = privateSubnets.map(subnet => subnet.subnetId);
    let manifest = utils_1.Utils.readYamlDocument(`${__dirname}/resources/k8s/karpenter-provisioner-config/tooling-provisioner.yml`);
    manifest = manifest.replace(/(\{{cluster-name}})/g, eksClusterName);
    manifest = manifest.replace(/(\{{subnet-list}})/g, listPrivateSubnets.join(','));
    let manfifestYAML = manifest.split("---").map((e) => utils_1.Utils.loadYaml(e));
    const manifestApply = cluster.addManifest('provisioner-tooling', ...manfifestYAML);
    manifestApply.node.addDependency(karpenterChart);
    return karpenterChart;
}
exports.karpenterSetup = karpenterSetup;
/**
 * @internal
 * Deploy the cluster autoscaler controller in the k8s cluster
 */
function clusterAutoscalerSetup(cluster, eksClusterName, scope, k8sVersion) {
    //Version of the autoscaler, controls the image tag
    const versionMap = new Map([
        [aws_eks_1.KubernetesVersion.V1_23, "9.21.0"],
        [aws_eks_1.KubernetesVersion.V1_22, "9.13.1"],
        [aws_eks_1.KubernetesVersion.V1_21, "9.13.1"]
    ]);
    // Create a Kubernetes Service Account for the Cluster Autoscaler with Amazon IAM Role
    const AutoscalerServiceAccount = cluster.addServiceAccount('Autoscaler', {
        name: 'cluster-autoscaler',
        namespace: 'kube-system',
    });
    //Iam policy attached to the Role used by k8s autoscaller
    let autoscalingPolicyDescribe = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: [
            'autoscaling:DescribeAutoScalingGroups',
            'autoscaling:DescribeAutoScalingInstances',
            'autoscaling:DescribeLaunchConfigurations',
            'autoscaling:DescribeTags',
            'ec2:DescribeLaunchTemplateVersions',
            'eks:DescribeNodegroup',
        ],
        resources: ['*'],
    });
    let autoscalingPolicyMutateGroup = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: [
            'autoscaling:SetDesiredCapacity',
            'autoscaling:TerminateInstanceInAutoScalingGroup',
        ],
        resources: ['*'],
        conditions: {
            StringEquals: {
                'aws:ResourceTag/eks:cluster-name': eksClusterName,
            },
        },
    });
    // Add the right Amazon IAM Policy to the Amazon IAM Role for the Cluster Autoscaler
    AutoscalerServiceAccount.addToPrincipalPolicy(autoscalingPolicyDescribe);
    AutoscalerServiceAccount.addToPrincipalPolicy(autoscalingPolicyMutateGroup);
    cluster.addHelmChart('AutoScaler', {
        chart: 'cluster-autoscaler',
        repository: 'https://kubernetes.github.io/autoscaler',
        version: versionMap.get(k8sVersion),
        namespace: 'kube-system',
        timeout: aws_cdk_lib_1.Duration.minutes(14),
        values: {
            cloudProvider: 'aws',
            awsRegion: aws_cdk_lib_1.Stack.of(scope).region,
            autoDiscovery: { clusterName: eksClusterName },
            rbac: {
                serviceAccount: {
                    name: 'cluster-autoscaler',
                    create: false,
                },
            },
            extraArgs: {
                'skip-nodes-with-local-storage': false,
                'scan-interval': '5s',
                'expander': 'least-waste',
                'balance-similar-node-groups': true,
                'skip-nodes-with-system-pods': false,
            },
        },
    });
}
exports.clusterAutoscalerSetup = clusterAutoscalerSetup;
//# sourceMappingURL=data:application/json;base64,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