"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlywayRunner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const path = require("path");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const s3deploy = require("aws-cdk-lib/aws-s3-deployment");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const constructs_1 = require("constructs");
/**
 * A CDK construct that runs flyway migration scripts against a redshift cluster.
 *
 * This construct is based on two main resource, an AWS Lambda hosting a flyway runner
 * and one custom resource invoking it when content of migrationScriptsFolderAbsolutePath changes.
 *
 * Usage example:
 *
 * *This example assume that migration SQL files are located in `resources/sql` of the cdk project.*
 * ```typescript
 * import * as path from 'path';
 * import * as ec2 from 'aws-cdk-lib/aws-ec2';
 * import * as redshift from '@aws-cdk/aws-redshift-alpha';
 * import * as cdk from 'aws-cdk-lib';
 *
 * import { FlywayRunner } from 'aws-analytics-reference-architecture';
 *
 * const integTestApp = new cdk.App();
 * const stack = new cdk.Stack(integTestApp, 'fywayRunnerTest');
 *
 * const vpc = new ec2.Vpc(stack, 'Vpc');

 * const dbName = 'testdb';
 * const cluster = new redshift.Cluster(stack, 'Redshift', {
 *   removalPolicy: cdk.RemovalPolicy.DESTROY,
 *   masterUser: {
 *     masterUsername: 'admin',
 *   },
 *   vpc,
 *   defaultDatabaseName: dbName,
 * });

 * new FlywayRunner(stack, 'testMigration', {
 *   migrationScriptsFolderAbsolutePath: path.join(__dirname, './resources/sql'),
 *   cluster: cluster,
 *   vpc: vpc,
 *   databaseName: dbName,
 * });
 * ```
 */
class FlywayRunner extends constructs_1.Construct {
    /**
     * Constructs a new instance of the FlywayRunner construct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {FlywayRunnerProps} props the FlywayRunner [properties]{@link FlywayRunnerProps}
     */
    constructor(scope, id, props) {
        super(scope, id);
        const sqlFilesAsset = s3deploy.Source.asset(props.migrationScriptsFolderAbsolutePath);
        const migrationFilesBucket = new s3.Bucket(this, 'migrationFilesBucket', {
            versioned: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
        });
        const migrationFilesDeployment = new s3deploy.BucketDeployment(this, 'DeploySQLMigrationFiles', {
            sources: [sqlFilesAsset],
            destinationBucket: migrationFilesBucket,
        });
        let flywayLambdaPolicy = [
            new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                resources: [
                    '*',
                ],
                actions: [
                    'cloudformation:DescribeStacks',
                    'cloudformation:DescribeStackResource',
                ],
            }),
            new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                resources: [
                    '*',
                ],
                actions: [
                    'ec2:CreateNetworkInterface',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:DeleteNetworkInterface',
                ],
            }),
        ];
        const flywayLambda = new pre_bundled_function_1.PreBundledFunction(this, 'FlywayLambda', {
            codePath: path.join(__dirname.split('/').slice(-1)[0], './resources/flyway-lambda/flyway-all.jar'),
            lambdaPolicyStatements: flywayLambdaPolicy,
            handler: 'com.geekoosh.flyway.FlywayCustomResourceHandler::handleRequest',
            runtime: lambda.Runtime.JAVA_11,
            logRetention: props.logRetention ?? logs.RetentionDays.ONE_WEEK,
            memorySize: 2048,
            timeout: cdk.Duration.seconds(900),
            vpc: props.vpc,
            securityGroups: props.cluster.connections.securityGroups,
            environment: {
                S3_BUCKET: migrationFilesBucket.bucketName,
                DB_CONNECTION_STRING: `jdbc:redshift://${props.cluster.clusterEndpoint.socketAddress}/${props.databaseName}`,
                DB_SECRET: props.cluster.secret.secretFullArn,
            },
        });
        // Allowing connection to the cluster
        props.cluster.connections.allowDefaultPortInternally();
        props.cluster.secret?.grantRead(flywayLambda);
        migrationFilesBucket.grantRead(flywayLambda);
        const flywayCustomResourceProvider = new custom_resources_1.Provider(this, 'FlywayCustomResourceProvider', {
            onEventHandler: flywayLambda,
            logRetention: props.logRetention ?? logs.RetentionDays.ONE_WEEK,
            securityGroups: props.cluster.connections.securityGroups,
            vpc: props.vpc,
        });
        this.runner = new aws_cdk_lib_1.CustomResource(this, 'trigger', {
            serviceToken: flywayCustomResourceProvider.serviceToken,
            properties: {
                flywayMethod: 'migrate',
                placeholders: props.replaceDictionary,
                assetHash: migrationFilesDeployment.node.findChild('Asset1').assetHash,
            },
        });
        for (const subnet of props.vpc.privateSubnets) {
            flywayLambda.node.addDependency(subnet);
        }
        flywayCustomResourceProvider.node.addDependency(migrationFilesDeployment);
    }
}
exports.FlywayRunner = FlywayRunner;
_a = JSII_RTTI_SYMBOL_1;
FlywayRunner[_a] = { fqn: "aws-analytics-reference-architecture.FlywayRunner", version: "2.8.7" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmx5d2F5LXJ1bm5lci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9kYi1zY2hlbWEtbWFuYWdlci9mbHl3YXktcnVubmVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEscUVBQXFFO0FBQ3JFLGlDQUFpQztBQUVqQyw2QkFBNkI7QUFFN0IsMkNBQTJDO0FBRTNDLGlEQUFpRDtBQUNqRCw2Q0FBNkM7QUFFN0MseUNBQXlDO0FBRXpDLDBEQUEwRDtBQUMxRCxtQ0FBbUM7QUFDbkMsNkNBQTREO0FBQzVELG1FQUF3RDtBQUN4RCx5RUFBb0U7QUFDcEUsMkNBQXVDO0FBeUR2Qzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBdUNHO0FBQ0gsTUFBYSxZQUFhLFNBQVEsc0JBQVM7SUFHekM7Ozs7O09BS0c7SUFDSCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXdCO1FBQ2hFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxhQUFhLEdBQUcsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFFdEYsTUFBTSxvQkFBb0IsR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLHNCQUFzQixFQUFFO1lBQ3ZFLFNBQVMsRUFBRSxJQUFJO1lBQ2YsYUFBYSxFQUFFLDJCQUFhLENBQUMsT0FBTztZQUNwQyxpQkFBaUIsRUFBRSxJQUFJO1NBQ3hCLENBQUMsQ0FBQztRQUNILE1BQU0sd0JBQXdCLEdBQUcsSUFBSSxRQUFRLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLHlCQUF5QixFQUFFO1lBQzlGLE9BQU8sRUFBRSxDQUFDLGFBQWEsQ0FBQztZQUN4QixpQkFBaUIsRUFBRSxvQkFBb0I7U0FDeEMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxrQkFBa0IsR0FBdUI7WUFDM0MsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO2dCQUN4QixTQUFTLEVBQUU7b0JBQ1QsR0FBRztpQkFDSjtnQkFDRCxPQUFPLEVBQUU7b0JBQ1AsK0JBQStCO29CQUMvQixzQ0FBc0M7aUJBQ3ZDO2FBQ0YsQ0FBQztZQUNGLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztnQkFDeEIsU0FBUyxFQUFFO29CQUNULEdBQUc7aUJBQ0o7Z0JBQ0QsT0FBTyxFQUFFO29CQUNQLDRCQUE0QjtvQkFDNUIsK0JBQStCO29CQUMvQiw0QkFBNEI7aUJBQzdCO2FBQ0YsQ0FBQztTQUNILENBQUM7UUFFRixNQUFNLFlBQVksR0FBRyxJQUFJLHlDQUFrQixDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7WUFDaEUsUUFBUSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSwwQ0FBMEMsQ0FBQztZQUNsRyxzQkFBc0IsRUFBRSxrQkFBa0I7WUFDMUMsT0FBTyxFQUFFLGdFQUFnRTtZQUN6RSxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxPQUFPO1lBQy9CLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWSxJQUFJLElBQUksQ0FBQyxhQUFhLENBQUMsUUFBUTtZQUMvRCxVQUFVLEVBQUUsSUFBSTtZQUNoQixPQUFPLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDO1lBQ2xDLEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRztZQUNkLGNBQWMsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxjQUFjO1lBQ3hELFdBQVcsRUFBRTtnQkFDWCxTQUFTLEVBQUUsb0JBQW9CLENBQUMsVUFBVTtnQkFDMUMsb0JBQW9CLEVBQUUsbUJBQW1CLEtBQUssQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLGFBQWEsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO2dCQUM1RyxTQUFTLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFPLENBQUMsYUFBYzthQUNoRDtTQUNGLENBQUMsQ0FBQztRQUVILHFDQUFxQztRQUNyQyxLQUFLLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQywwQkFBMEIsRUFBRSxDQUFDO1FBRXZELEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLFNBQVMsQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUM5QyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFN0MsTUFBTSw0QkFBNEIsR0FBRyxJQUFJLDJCQUFRLENBQUMsSUFBSSxFQUFFLDhCQUE4QixFQUFFO1lBQ3RGLGNBQWMsRUFBRSxZQUFZO1lBQzVCLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWSxJQUFJLElBQUksQ0FBQyxhQUFhLENBQUMsUUFBUTtZQUMvRCxjQUFjLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsY0FBYztZQUN4RCxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7U0FDZixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsTUFBTSxHQUFHLElBQUksNEJBQWMsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ2hELFlBQVksRUFBRSw0QkFBNEIsQ0FBQyxZQUFZO1lBQ3ZELFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUUsU0FBUztnQkFDdkIsWUFBWSxFQUFFLEtBQUssQ0FBQyxpQkFBaUI7Z0JBQ3JDLFNBQVMsRUFBRyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBVyxDQUFDLFNBQVM7YUFDbEY7U0FDRixDQUFDLENBQUM7UUFDSCxLQUFLLE1BQU0sTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLENBQUMsY0FBYyxFQUFFO1lBQzdDLFlBQVksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ3pDO1FBQ0QsNEJBQTRCLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzVFLENBQUM7O0FBMUZILG9DQTJGQyIsInNvdXJjZXNDb250ZW50IjpbIi8vIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuLy8gU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IE1JVC0wXG5cbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBlYzIgZnJvbSAnYXdzLWNkay1saWIvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxvZ3MnO1xuaW1wb3J0ICogYXMgcmVkc2hpZnQgZnJvbSAnQGF3cy1jZGsvYXdzLXJlZHNoaWZ0LWFscGhhJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zMyc7XG5pbXBvcnQgeyBBc3NldCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zMy1hc3NldHMnO1xuaW1wb3J0ICogYXMgczNkZXBsb3kgZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzLWRlcGxveW1lbnQnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IEN1c3RvbVJlc291cmNlLCBSZW1vdmFsUG9saWN5IH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgUHJvdmlkZXIgfSBmcm9tICdhd3MtY2RrLWxpYi9jdXN0b20tcmVzb3VyY2VzJztcbmltcG9ydCB7IFByZUJ1bmRsZWRGdW5jdGlvbiB9IGZyb20gJy4uL2NvbW1vbi9wcmUtYnVuZGxlZC1mdW5jdGlvbic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbi8vaW1wb3J0IHsgU2NvcGVkSWFtUHJvdmlkZXIgfSBmcm9tICcuLi9jb21tb24vc2NvcGVkLWlhbS1jdXN0b21lci1yZXNvdXJjZSc7XG5cbi8qKlxuICogVGhlIHByb3BlcnRpZXMgb2YgdGhlIEZseXdheVJ1bm5lciBjb25zdHJ1Y3QsIG5lZWRlZCB0byBydW4gZmx5d2F5IG1pZ3JhdGlvbiBzY3JpcHRzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEZseXdheVJ1bm5lclByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBhYnNvbHV0ZSBwYXRoIHRvIHRoZSBmbHl3YXkgbWlncmF0aW9uIHNjcmlwdHMuXG4gICAqIFRob3NlIHNjcmlwdHMgbmVlZHMgdG8gZm9sbG93IGV4cGVjdGVkIGZseXdheSBuYW1pbmcgY29udmVudGlvbi5cbiAgICogQHNlZSBodHRwczovL2ZseXdheWRiLm9yZy9kb2N1bWVudGF0aW9uL2NvbmNlcHRzL21pZ3JhdGlvbnMuaHRtbCNzcWwtYmFzZWQtbWlncmF0aW9ucyBmb3IgbW9yZSBkZXRhaWxzLlxuICAgKi9cbiAgcmVhZG9ubHkgbWlncmF0aW9uU2NyaXB0c0ZvbGRlckFic29sdXRlUGF0aDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgY2x1c3RlciB0byBydW4gbWlncmF0aW9uIHNjcmlwdHMgYWdhaW5zdC5cbiAgICovXG4gIHJlYWRvbmx5IGNsdXN0ZXI6IHJlZHNoaWZ0LkNsdXN0ZXI7XG5cbiAgLyoqXG4gICAqIFRoZSB2cGMgaG9zdGluZyB0aGUgY2x1c3Rlci5cbiAgICovXG4gIHJlYWRvbmx5IHZwYzogZWMyLlZwYztcblxuICAvKipcbiAgICogVGhlIGRhdGFiYXNlIG5hbWUgdG8gcnVuIG1pZ3JhdGlvbiBzY3JpcHRzIGFnYWluc3QuXG4gICAqL1xuICByZWFkb25seSBkYXRhYmFzZU5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogUGVyaW9kIHRvIGtlZXAgdGhlIGxvZ3MgYXJvdW5kLlxuICAgKiBAZGVmYXVsdCBsb2dzLlJldGVudGlvbkRheXMuT05FX1dFRUtcbiAgICovXG4gIHJlYWRvbmx5IGxvZ1JldGVudGlvbj86IGxvZ3MuUmV0ZW50aW9uRGF5cztcblxuICAvKipcbiAgICogQSBrZXktdmFsdWUgbWFwIG9mIHN0cmluZyAoZW5jYXBzdWxhdGVkIGJldHdlZW4gYCR7YCBhbmQgYH1gKSB0byByZXBsYWNlIGluIHRoZSBTUUwgZmlsZXMgZ2l2ZW4uXG4gICAqXG4gICAqIEV4YW1wbGU6XG4gICAqXG4gICAqICogVGhlIFNRTCBmaWxlOlxuICAgKlxuICAgKiAgIGBgYHNxbFxuICAgKiAgIFNFTEVDVCAqIEZST00gJHtUQUJMRV9OQU1FfTtcbiAgICogICBgYGBcbiAgICogKiBUaGUgcmVwbGFjZW1lbnQgbWFwOlxuICAgKlxuICAgKiAgIGBgYHR5cGVzY3JpcHRcbiAgICogICByZXBsYWNlRGljdGlvbmFyeSA9IHtcbiAgICogICAgIFRBQkxFX05BTUU6ICdteV90YWJsZSdcbiAgICogICB9XG4gICAqICAgYGBgXG4gICAqIEBkZWZhdWx0IC0gTm8gcmVwbGFjZW1lbnQgaXMgZG9uZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVwbGFjZURpY3Rpb25hcnk/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xufVxuXG4vKipcbiAqIEEgQ0RLIGNvbnN0cnVjdCB0aGF0IHJ1bnMgZmx5d2F5IG1pZ3JhdGlvbiBzY3JpcHRzIGFnYWluc3QgYSByZWRzaGlmdCBjbHVzdGVyLlxuICpcbiAqIFRoaXMgY29uc3RydWN0IGlzIGJhc2VkIG9uIHR3byBtYWluIHJlc291cmNlLCBhbiBBV1MgTGFtYmRhIGhvc3RpbmcgYSBmbHl3YXkgcnVubmVyXG4gKiBhbmQgb25lIGN1c3RvbSByZXNvdXJjZSBpbnZva2luZyBpdCB3aGVuIGNvbnRlbnQgb2YgbWlncmF0aW9uU2NyaXB0c0ZvbGRlckFic29sdXRlUGF0aCBjaGFuZ2VzLlxuICpcbiAqIFVzYWdlIGV4YW1wbGU6XG4gKlxuICogKlRoaXMgZXhhbXBsZSBhc3N1bWUgdGhhdCBtaWdyYXRpb24gU1FMIGZpbGVzIGFyZSBsb2NhdGVkIGluIGByZXNvdXJjZXMvc3FsYCBvZiB0aGUgY2RrIHByb2plY3QuKlxuICogYGBgdHlwZXNjcmlwdFxuICogaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbiAqIGltcG9ydCAqIGFzIGVjMiBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWMyJztcbiAqIGltcG9ydCAqIGFzIHJlZHNoaWZ0IGZyb20gJ0Bhd3MtY2RrL2F3cy1yZWRzaGlmdC1hbHBoYSc7XG4gKiBpbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWInO1xuICpcbiAqIGltcG9ydCB7IEZseXdheVJ1bm5lciB9IGZyb20gJ2F3cy1hbmFseXRpY3MtcmVmZXJlbmNlLWFyY2hpdGVjdHVyZSc7XG4gKlxuICogY29uc3QgaW50ZWdUZXN0QXBwID0gbmV3IGNkay5BcHAoKTtcbiAqIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayhpbnRlZ1Rlc3RBcHAsICdmeXdheVJ1bm5lclRlc3QnKTtcbiAqXG4gKiBjb25zdCB2cGMgPSBuZXcgZWMyLlZwYyhzdGFjaywgJ1ZwYycpO1xuXG4gKiBjb25zdCBkYk5hbWUgPSAndGVzdGRiJztcbiAqIGNvbnN0IGNsdXN0ZXIgPSBuZXcgcmVkc2hpZnQuQ2x1c3RlcihzdGFjaywgJ1JlZHNoaWZ0Jywge1xuICogICByZW1vdmFsUG9saWN5OiBjZGsuUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICogICBtYXN0ZXJVc2VyOiB7XG4gKiAgICAgbWFzdGVyVXNlcm5hbWU6ICdhZG1pbicsXG4gKiAgIH0sXG4gKiAgIHZwYyxcbiAqICAgZGVmYXVsdERhdGFiYXNlTmFtZTogZGJOYW1lLFxuICogfSk7XG5cbiAqIG5ldyBGbHl3YXlSdW5uZXIoc3RhY2ssICd0ZXN0TWlncmF0aW9uJywge1xuICogICBtaWdyYXRpb25TY3JpcHRzRm9sZGVyQWJzb2x1dGVQYXRoOiBwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi9yZXNvdXJjZXMvc3FsJyksXG4gKiAgIGNsdXN0ZXI6IGNsdXN0ZXIsXG4gKiAgIHZwYzogdnBjLFxuICogICBkYXRhYmFzZU5hbWU6IGRiTmFtZSxcbiAqIH0pO1xuICogYGBgXG4gKi9cbmV4cG9ydCBjbGFzcyBGbHl3YXlSdW5uZXIgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICBwdWJsaWMgcmVhZG9ubHkgcnVubmVyOiBDdXN0b21SZXNvdXJjZTtcblxuICAvKipcbiAgICogQ29uc3RydWN0cyBhIG5ldyBpbnN0YW5jZSBvZiB0aGUgRmx5d2F5UnVubmVyIGNvbnN0cnVjdFxuICAgKiBAcGFyYW0ge0NvbnN0cnVjdH0gc2NvcGUgdGhlIFNjb3BlIG9mIHRoZSBDREsgQ29uc3RydWN0XG4gICAqIEBwYXJhbSB7c3RyaW5nfSBpZCB0aGUgSUQgb2YgdGhlIENESyBDb25zdHJ1Y3RcbiAgICogQHBhcmFtIHtGbHl3YXlSdW5uZXJQcm9wc30gcHJvcHMgdGhlIEZseXdheVJ1bm5lciBbcHJvcGVydGllc117QGxpbmsgRmx5d2F5UnVubmVyUHJvcHN9XG4gICAqL1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRmx5d2F5UnVubmVyUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3Qgc3FsRmlsZXNBc3NldCA9IHMzZGVwbG95LlNvdXJjZS5hc3NldChwcm9wcy5taWdyYXRpb25TY3JpcHRzRm9sZGVyQWJzb2x1dGVQYXRoKTtcblxuICAgIGNvbnN0IG1pZ3JhdGlvbkZpbGVzQnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0aGlzLCAnbWlncmF0aW9uRmlsZXNCdWNrZXQnLCB7XG4gICAgICB2ZXJzaW9uZWQ6IHRydWUsXG4gICAgICByZW1vdmFsUG9saWN5OiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgICBhdXRvRGVsZXRlT2JqZWN0czogdHJ1ZSxcbiAgICB9KTtcbiAgICBjb25zdCBtaWdyYXRpb25GaWxlc0RlcGxveW1lbnQgPSBuZXcgczNkZXBsb3kuQnVja2V0RGVwbG95bWVudCh0aGlzLCAnRGVwbG95U1FMTWlncmF0aW9uRmlsZXMnLCB7XG4gICAgICBzb3VyY2VzOiBbc3FsRmlsZXNBc3NldF0sXG4gICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogbWlncmF0aW9uRmlsZXNCdWNrZXQsXG4gICAgfSk7XG5cbiAgICBsZXQgZmx5d2F5TGFtYmRhUG9saWN5OiBQb2xpY3lTdGF0ZW1lbnQgW10gPSBbXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgJyonLFxuICAgICAgICBdLFxuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkRlc2NyaWJlU3RhY2tzJyxcbiAgICAgICAgICAnY2xvdWRmb3JtYXRpb246RGVzY3JpYmVTdGFja1Jlc291cmNlJyxcbiAgICAgICAgXSxcbiAgICAgIH0pLFxuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICcqJyxcbiAgICAgICAgXSxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdlYzI6Q3JlYXRlTmV0d29ya0ludGVyZmFjZScsXG4gICAgICAgICAgJ2VjMjpEZXNjcmliZU5ldHdvcmtJbnRlcmZhY2VzJyxcbiAgICAgICAgICAnZWMyOkRlbGV0ZU5ldHdvcmtJbnRlcmZhY2UnLFxuICAgICAgICBdLFxuICAgICAgfSksXG4gICAgXTtcblxuICAgIGNvbnN0IGZseXdheUxhbWJkYSA9IG5ldyBQcmVCdW5kbGVkRnVuY3Rpb24odGhpcywgJ0ZseXdheUxhbWJkYScsIHtcbiAgICAgIGNvZGVQYXRoOiBwYXRoLmpvaW4oX19kaXJuYW1lLnNwbGl0KCcvJykuc2xpY2UoLTEpWzBdLCAnLi9yZXNvdXJjZXMvZmx5d2F5LWxhbWJkYS9mbHl3YXktYWxsLmphcicpLFxuICAgICAgbGFtYmRhUG9saWN5U3RhdGVtZW50czogZmx5d2F5TGFtYmRhUG9saWN5LFxuICAgICAgaGFuZGxlcjogJ2NvbS5nZWVrb29zaC5mbHl3YXkuRmx5d2F5Q3VzdG9tUmVzb3VyY2VIYW5kbGVyOjpoYW5kbGVSZXF1ZXN0JyxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLkpBVkFfMTEsXG4gICAgICBsb2dSZXRlbnRpb246IHByb3BzLmxvZ1JldGVudGlvbiA/PyBsb2dzLlJldGVudGlvbkRheXMuT05FX1dFRUssXG4gICAgICBtZW1vcnlTaXplOiAyMDQ4LFxuICAgICAgdGltZW91dDogY2RrLkR1cmF0aW9uLnNlY29uZHMoOTAwKSxcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgc2VjdXJpdHlHcm91cHM6IHByb3BzLmNsdXN0ZXIuY29ubmVjdGlvbnMuc2VjdXJpdHlHcm91cHMsXG4gICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICBTM19CVUNLRVQ6IG1pZ3JhdGlvbkZpbGVzQnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgIERCX0NPTk5FQ1RJT05fU1RSSU5HOiBgamRiYzpyZWRzaGlmdDovLyR7cHJvcHMuY2x1c3Rlci5jbHVzdGVyRW5kcG9pbnQuc29ja2V0QWRkcmVzc30vJHtwcm9wcy5kYXRhYmFzZU5hbWV9YCxcbiAgICAgICAgREJfU0VDUkVUOiBwcm9wcy5jbHVzdGVyLnNlY3JldCEuc2VjcmV0RnVsbEFybiEsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gQWxsb3dpbmcgY29ubmVjdGlvbiB0byB0aGUgY2x1c3RlclxuICAgIHByb3BzLmNsdXN0ZXIuY29ubmVjdGlvbnMuYWxsb3dEZWZhdWx0UG9ydEludGVybmFsbHkoKTtcblxuICAgIHByb3BzLmNsdXN0ZXIuc2VjcmV0Py5ncmFudFJlYWQoZmx5d2F5TGFtYmRhKTtcbiAgICBtaWdyYXRpb25GaWxlc0J1Y2tldC5ncmFudFJlYWQoZmx5d2F5TGFtYmRhKTtcblxuICAgIGNvbnN0IGZseXdheUN1c3RvbVJlc291cmNlUHJvdmlkZXIgPSBuZXcgUHJvdmlkZXIodGhpcywgJ0ZseXdheUN1c3RvbVJlc291cmNlUHJvdmlkZXInLCB7XG4gICAgICBvbkV2ZW50SGFuZGxlcjogZmx5d2F5TGFtYmRhLFxuICAgICAgbG9nUmV0ZW50aW9uOiBwcm9wcy5sb2dSZXRlbnRpb24gPz8gbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9XRUVLLFxuICAgICAgc2VjdXJpdHlHcm91cHM6IHByb3BzLmNsdXN0ZXIuY29ubmVjdGlvbnMuc2VjdXJpdHlHcm91cHMsXG4gICAgICB2cGM6IHByb3BzLnZwYyxcbiAgICB9KTtcblxuICAgIHRoaXMucnVubmVyID0gbmV3IEN1c3RvbVJlc291cmNlKHRoaXMsICd0cmlnZ2VyJywge1xuICAgICAgc2VydmljZVRva2VuOiBmbHl3YXlDdXN0b21SZXNvdXJjZVByb3ZpZGVyLnNlcnZpY2VUb2tlbixcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgZmx5d2F5TWV0aG9kOiAnbWlncmF0ZScsXG4gICAgICAgIHBsYWNlaG9sZGVyczogcHJvcHMucmVwbGFjZURpY3Rpb25hcnksXG4gICAgICAgIGFzc2V0SGFzaDogKG1pZ3JhdGlvbkZpbGVzRGVwbG95bWVudC5ub2RlLmZpbmRDaGlsZCgnQXNzZXQxJykgYXMgQXNzZXQpLmFzc2V0SGFzaCxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgZm9yIChjb25zdCBzdWJuZXQgb2YgcHJvcHMudnBjLnByaXZhdGVTdWJuZXRzKSB7XG4gICAgICBmbHl3YXlMYW1iZGEubm9kZS5hZGREZXBlbmRlbmN5KHN1Ym5ldCk7XG4gICAgfVxuICAgIGZseXdheUN1c3RvbVJlc291cmNlUHJvdmlkZXIubm9kZS5hZGREZXBlbmRlbmN5KG1pZ3JhdGlvbkZpbGVzRGVwbG95bWVudCk7XG4gIH1cbn1cbiJdfQ==