"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataLakeStorage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ara_bucket_1 = require("./ara-bucket");
const context_options_1 = require("./common/context-options");
const tracked_construct_1 = require("./common/tracked-construct");
/**
 * A CDK Construct that creates the storage layers of a data lake composed of Amazon S3 Buckets.
 *
 * This construct is based on 3 Amazon S3 buckets configured with AWS best practices:
 *  * S3 buckets for Raw/Cleaned/Transformed data,
 *  * data lifecycle optimization/transitioning to different Amazon S3 storage classes
 *  * server side buckets encryption managed by KMS customer key
 *  * Default single KMS key
 *  * SSL communication enforcement
 *  * access logged to an S3 bucket
 *  * All public access blocked
 *
 * By default the transitioning rules to Amazon S3 storage classes are configured as following:
 *  * Raw data is moved to Infrequent Access after 30 days and archived to Glacier after 90 days
 *  * Clean and Transformed data is moved to Infrequent Access after 90 days and is not archived
 *
 * Objects and buckets are automatically deleted when the CDK application is detroyed.
 *
 * For custom requirements, consider using {@link AraBucket}.
 *
 * Usage example:
 * ```typescript
 * import * as cdk from 'aws-cdk-lib';
 * import { DataLakeStorage } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new cdk.App();
 * const stack = new cdk.Stack(exampleApp, 'DataLakeStorageStack');
 *
 * new DataLakeStorage(stack, 'MyDataLakeStorage', {
 *  rawInfrequentAccessDelay: 90,
 *  rawArchiveDelay: 180,
 *  cleanInfrequentAccessDelay: 180,
 *  cleanArchiveDelay: 360,
 *  transformInfrequentAccessDelay: 180,
 *  transformArchiveDelay: 360,
 * });
 * ```
 */
class DataLakeStorage extends tracked_construct_1.TrackedConstruct {
    /**
     * Construct a new instance of DataLakeStorage based on Amazon S3 buckets with best practices configuration
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataLakeStorageProps} props the DataLakeStorageProps properties
     * @access public
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingCode: context_options_1.ContextOptions.DATA_LAKE_ID,
        };
        super(scope, id, trackedConstructProps);
        var rawInfrequentAccessDelay = 30;
        var rawArchiveDelay = 90;
        var cleanInfrequentAccessDelay = 90;
        var cleanArchiveDelay = undefined;
        var transformInfrequentAccessDelay = 90;
        var transformArchiveDelay = undefined;
        if (props) {
            if (props.rawInfrequentAccessDelay) {
                if (props.rawInfrequentAccessDelay < 30) {
                    throw new Error('Transitioning to infrequent access storage class cannot be done before 30 days');
                }
                else {
                    rawInfrequentAccessDelay = props.rawInfrequentAccessDelay;
                }
            }
            if (props.rawArchiveDelay) {
                if (props.rawArchiveDelay < 90) {
                    throw new Error('Archiving to glacier storage class cannot be done before 90 days');
                }
                else {
                    rawArchiveDelay = props.rawArchiveDelay;
                }
            }
            if (props.cleanInfrequentAccessDelay) {
                if (props.cleanInfrequentAccessDelay < 30) {
                    throw new Error('Transitioning to infrequent access storage class cannot be done before 30 days');
                }
                else {
                    cleanInfrequentAccessDelay = props.cleanInfrequentAccessDelay;
                }
            }
            if (props.cleanArchiveDelay) {
                if (props.cleanArchiveDelay < 90) {
                    throw new Error('Archiving to glacier storage class cannot be done before 90 days');
                }
                else {
                    cleanArchiveDelay = props.cleanArchiveDelay;
                }
            }
            if (props.transformInfrequentAccessDelay) {
                if (props.transformInfrequentAccessDelay < 30) {
                    throw new Error('Transitioning to infrequent access storage class cannot be done before 30 days');
                }
                else {
                    transformInfrequentAccessDelay = props.transformInfrequentAccessDelay;
                }
            }
            if (props.transformArchiveDelay) {
                if (props.transformArchiveDelay < 90) {
                    throw new Error('Archiving to glacier storage class cannot be done before 90 days');
                }
                else {
                    transformArchiveDelay = props.transformArchiveDelay;
                }
            }
        }
        // Prepare Amazon S3 Lifecycle Rules for raw data
        const rawTransitions = [
            {
                storageClass: aws_s3_1.StorageClass.INFREQUENT_ACCESS,
                transitionAfter: aws_cdk_lib_1.Duration.days(rawInfrequentAccessDelay),
            },
            {
                storageClass: aws_s3_1.StorageClass.GLACIER,
                transitionAfter: aws_cdk_lib_1.Duration.days(rawArchiveDelay),
            },
        ];
        // Create the raw data bucket with the raw transitions
        this.rawBucket = ara_bucket_1.AraBucket.getOrCreate(this, {
            encryption: aws_s3_1.BucketEncryption.KMS,
            bucketName: 'raw',
            lifecycleRules: [
                {
                    transitions: rawTransitions,
                    abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(1),
                },
            ],
            serverAccessLogsPrefix: 'raw-bucket',
        });
        // Prepare Amazon S3 Lifecycle Rules for clean data
        const cleanTransitions = [
            {
                storageClass: aws_s3_1.StorageClass.INFREQUENT_ACCESS,
                transitionAfter: aws_cdk_lib_1.Duration.days(cleanInfrequentAccessDelay),
            },
        ];
        if (cleanArchiveDelay) {
            cleanTransitions.push({
                storageClass: aws_s3_1.StorageClass.GLACIER,
                transitionAfter: aws_cdk_lib_1.Duration.days(cleanArchiveDelay),
            });
        }
        // Create the clean data bucket
        this.cleanBucket = ara_bucket_1.AraBucket.getOrCreate(this, {
            encryption: aws_s3_1.BucketEncryption.KMS,
            bucketName: 'clean',
            lifecycleRules: [
                {
                    transitions: cleanTransitions,
                    abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(1),
                },
            ],
            serverAccessLogsPrefix: 'clean-bucket',
        });
        // Prepare Amazon S3 Lifecycle Rules for clean data
        const transformTransitions = [
            {
                storageClass: aws_s3_1.StorageClass.INFREQUENT_ACCESS,
                transitionAfter: aws_cdk_lib_1.Duration.days(transformInfrequentAccessDelay),
            },
        ];
        if (transformArchiveDelay) {
            transformTransitions.push({
                storageClass: aws_s3_1.StorageClass.GLACIER,
                transitionAfter: aws_cdk_lib_1.Duration.days(transformArchiveDelay),
            });
        }
        // Create the transform data bucket
        this.transformBucket = ara_bucket_1.AraBucket.getOrCreate(this, {
            encryption: aws_s3_1.BucketEncryption.KMS,
            bucketName: 'transform',
            lifecycleRules: [
                {
                    transitions: transformTransitions,
                    abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(1),
                },
            ],
            serverAccessLogsPrefix: 'transform-bucket',
        });
    }
}
exports.DataLakeStorage = DataLakeStorage;
_a = JSII_RTTI_SYMBOL_1;
DataLakeStorage[_a] = { fqn: "aws-analytics-reference-architecture.DataLakeStorage", version: "2.8.7" };
//# sourceMappingURL=data:application/json;base64,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