"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.PreBundledFunction = void 0;
const path = require("path");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * A Lambda function with prebundled dependencies.
 *
 * It changes of the code path by based on the environment that `cdk synth` is running on.
 *
 * This class is used together with a Projen custom task "copy-resources", and "pip-install".
 * The tasks will ensure that all Python and libraries files are available in "lib" folder,
 * with the same relative path.
 *
 * When this construct is being run in JSII, this file will be in `node_modules` folder
 * (as it's installed as a 3rd party library.) So we need to change reference based on __dirname.
 *
 * Additionally, the Lambda function is using a common Lambda layer with common dependencies when the runtime is Python.
 * Refer to [PreBundledLayer]{@link PreBundledLayer} construct for packaged Python dependencies.
 *
 *  * Usage example:
 * ```typescript
 * import { PreBundledFunction } from 'aws-analytics-reference-architecture';
 *
 * new PreBundledFunction(this, 'PreBundledFunction', {
 *   codePath: 'construct-dir/resources/lambdas/lambda_dir',
 *   lambdaPolicyStatements: findFilePathsFnPolicy,
 *   // you can use any property available in Function CDK Construct including
 *   memorySize: 1024,
 *   runtime: Runtime.PYTHON_3_8,
 *   handler: 'lambda-file-name.handler',
 *   logRetention: RetentionDays.ONE_WEEK,
 *   timeout: Duration.minutes(15),
 * });
 * ```
 */
class PreBundledFunction extends aws_lambda_1.Function {
    /**
     * Constructs a new instance of the PreBundledFunction construct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {PreBundledFunctionProps} props the PreBundledFunction [properties]{@link PreBundledFunctionProps}
     */
    constructor(scope, id, props) {
        if (props.code) {
            throw new Error('Pass "codePath" prop instead of "code" . See CONTRIB_FAQ.md on how to create prebundled Lambda function.');
        }
        let functionProps = { ...props };
        // __dirname is where this file is. In JSII, it is <jsii_tmp_path>/lib/common.
        // When running unit tests, it is ./src/common). In both case, we need to go up one level.
        let assetPath = path.join(__dirname, `../${props.codePath}`);
        functionProps.code = aws_lambda_1.Code.fromAsset(assetPath);
        functionProps.functionName = `${scope.node.id}${id}`;
        let lambdaPolicyStatement = [];
        lambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:CreateLogGroup'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/${functionProps.functionName}`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        lambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:CreateLogStream'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/${functionProps.functionName}*`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        lambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:PutLogEvents'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/${functionProps.functionName}:log-stream:*`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        functionProps.lambdaPolicyStatements?.forEach((element) => {
            lambdaPolicyStatement.push(element);
        });
        //Policy to allow lambda access to cloudwatch logs
        const lambdaExecutionRolePolicy = new aws_iam_1.ManagedPolicy(scope, 'LambdaExecutionRolePolicy' + functionProps.functionName, {
            statements: lambdaPolicyStatement,
            description: 'Policy similar to lambda execution role but scoped down',
        });
        //Use role from props or create an execution role for the lambda and attach to it a policy formed from user input
        const lambdaExecutionRole = props.role ? props.role : new aws_iam_1.Role(scope, 'LambdaExecutionRole' + functionProps.functionName, {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            description: 'Role used by lambda in ARA',
        });
        lambdaExecutionRole.addManagedPolicy(lambdaExecutionRolePolicy);
        let logRetentionLambdaPolicyStatement = [];
        logRetentionLambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:PutRetentionPolicy', 'logs:DeleteRetentionPolicy'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/${functionProps.functionName}`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        logRetentionLambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        //Policy to allow lambda access to cloudwatch logs
        const logRetentionLambdaExecutionRolePolicy = new aws_iam_1.ManagedPolicy(scope, 'LogRetentionLambdaExecutionRolePolicy' + functionProps.functionName, {
            statements: logRetentionLambdaPolicyStatement,
            description: 'Policy used to allow CR for log retention',
        });
        //Create an execution role for the lambda and attach to it a policy formed from user input
        const logRetentionLambdaExecutionRole = new aws_iam_1.Role(scope, 'LogRetentionLambdaExecutionRole' + functionProps.functionName, {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            description: 'Role used by lambda to modify log retention',
            managedPolicies: [logRetentionLambdaExecutionRolePolicy],
        });
        functionProps.role = lambdaExecutionRole;
        functionProps.logRetentionRole = logRetentionLambdaExecutionRole;
        //delete props that were added to force user input
        delete functionProps.codePath;
        delete functionProps.name;
        delete functionProps.lambdaPolicyStatements;
        super(scope, id, { ...functionProps });
    }
}
exports.PreBundledFunction = PreBundledFunction;
//# sourceMappingURL=data:application/json;base64,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