"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.reportBuild = exports.startBuild = void 0;
const defaultDestroyBuildSpec = `
version: 0.2
env:
  variables:
    CFN_RESPONSE_URL: CFN_RESPONSE_URL_NOT_SET
    CFN_STACK_ID: CFN_STACK_ID_NOT_SET
    CFN_REQUEST_ID: CFN_REQUEST_ID_NOT_SET
    CFN_LOGICAL_RESOURCE_ID: CFN_LOGICAL_RESOURCE_ID_NOT_SET
phases:
  pre_build:
    on-failure: ABORT
    commands:
      - echo "Default destroy buildspec"
      - cd $CODEBUILD_SRC_DIR/$CDK_APP_LOCATION
      - npm install -g aws-cdk && sudo apt-get install python3 && python -m
        ensurepip --upgrade && python -m pip install --upgrade pip && python -m
        pip install -r requirements.txt
      - \"export AWS_ACCOUNT_ID=$(echo $CODEBUILD_BUILD_ARN | cut -d: -f5)\"
      - 'echo \"AWS_ACCOUNT_ID: $AWS_ACCOUNT_ID\"'
      - cdk bootstrap aws://$AWS_ACCOUNT_ID/$AWS_REGION
  build:
    on-failure: ABORT
    commands:
      - \"export AWS_ACCOUNT_ID=$(echo $CODEBUILD_BUILD_ARN | cut -d: -f5)\"
      - 'echo \"AWS_ACCOUNT_ID: $AWS_ACCOUNT_ID\"'
      - cdk destroy --force --all --require-approval never
`;
const defaultDeployBuildSpec = `
version: 0.2
env:
  variables:
    CFN_RESPONSE_URL: CFN_RESPONSE_URL_NOT_SET
    CFN_STACK_ID: CFN_STACK_ID_NOT_SET
    CFN_REQUEST_ID: CFN_REQUEST_ID_NOT_SET
    CFN_LOGICAL_RESOURCE_ID: CFN_LOGICAL_RESOURCE_ID_NOT_SET
    PARAMETERS: PARAMETERS_NOT_SET
    STACKNAME: STACKNAME_NOT_SET
phases:
  pre_build:
    on-failure: ABORT
    commands:
      - echo "Default deploy buildspec"
      - cd $CODEBUILD_SRC_DIR/$CDK_APP_LOCATION
      - npm install -g aws-cdk && sudo apt-get install python3 && python -m
        ensurepip --upgrade && python -m pip install --upgrade pip && python -m
        pip install -r requirements.txt
      - \"export AWS_ACCOUNT_ID=$(echo $CODEBUILD_BUILD_ARN | cut -d: -f5)\"
      - 'echo \"AWS_ACCOUNT_ID: $AWS_ACCOUNT_ID\"'
      - cdk bootstrap aws://$AWS_ACCOUNT_ID/$AWS_REGION
  build:
    on-failure: ABORT
    commands:
      - \"export AWS_ACCOUNT_ID=$(echo $CODEBUILD_BUILD_ARN | cut -d: -f5)\"
      - 'echo \"AWS_ACCOUNT_ID: $AWS_ACCOUNT_ID\"'
      - cdk deploy $STACKNAME $PARAMETERS --require-approval=never
`;
// workaround to get a Lambda function with inline code and packaged into the ARA library
// We need inline code to ensure it's deployable via a CloudFormation template
// TODO modify the PreBundledFunction to allow for inline Lambda in addtion to asset based Lambda
exports.startBuild = (deployBuildSpec, destroyBuildSpec) => {
    return `
const respond = async function(event, context, responseStatus, responseData, physicalResourceId, noEcho) {
  return new Promise((resolve, reject) => {
    var responseBody = JSON.stringify({
      Status: responseStatus,
      Reason: \"See the details in CloudWatch Log Stream: \" + context.logGroupName + \" \" + context.logStreamName,
      PhysicalResourceId: physicalResourceId || context.logStreamName,
      StackId: event.StackId,
      RequestId: event.RequestId,
      LogicalResourceId: event.LogicalResourceId,
      NoEcho: noEcho || false,
      Data: responseData
    });
    
    console.log(\"Response body:\", responseBody);
    
    var https = require(\"https\");
    var url = require(\"url\");
    
    var parsedUrl = url.parse(event.ResponseURL);
    var options = {
      hostname: parsedUrl.hostname,
      port: 443,
      path: parsedUrl.path,
      method: \"PUT\",
      headers: {
        \"content-type\": \"\",
        \"content-length\": responseBody.length
      }
    };
    
    var request = https.request(options, function(response) {
      console.log(\"Status code: \" + response.statusCode);
      console.log(\"Status message: \" + response.statusMessage);
      resolve();
    });
    
    request.on(\"error\", function(error) {
      console.log(\"respond(..) failed executing https.request(..): \" + error);
      resolve();
    });
    
    request.write(responseBody);
    request.end();
  });
};

const AWS = require('aws-sdk');

exports.handler = async function (event, context) {
  console.log(JSON.stringify(event, null, 4));
  try {
    const projectName = event.ResourceProperties.ProjectName;
    const codebuild = new AWS.CodeBuild();
    
    console.log(\`Starting new build of project \${projectName}\`);
    
    const { build } = await codebuild.startBuild({
      projectName,
      // Pass CFN related parameters through the build for extraction by the
      // completion handler.
      buildspecOverride: event.RequestType === 'Delete' ? \`${destroyBuildSpec ? `${destroyBuildSpec.toBuildSpec()}` : defaultDestroyBuildSpec}\` : \`${deployBuildSpec ? `${deployBuildSpec.toBuildSpec()}` : defaultDeployBuildSpec}\`,
      environmentVariablesOverride: [
        {
          name: 'CFN_RESPONSE_URL',
          value: event.ResponseURL
        },
        {
          name: 'CFN_STACK_ID',
          value: event.StackId
        },
        {
          name: 'CFN_REQUEST_ID',
          value: event.RequestId
        },
        {
          name: 'CFN_LOGICAL_RESOURCE_ID',
          value: event.LogicalResourceId
        },
        {
          name: 'BUILD_ROLE_ARN',
          value: event.ResourceProperties.BuildRoleArn
        }
      ]
    }).promise();
    console.log(\`Build id \${build.id} started - resource completion handled by EventBridge\`);
  } catch(error) {
    console.error(error);
    await respond(event, context, 'FAILED', { Error: error });
  }
};
`;
};
exports.reportBuild = `
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0

const respond = async function(event, context, responseStatus, responseData, physicalResourceId, noEcho) {
  return new Promise((resolve, reject) => {
    var responseBody = JSON.stringify({
      Status: responseStatus,
      Reason: "See the details in CloudWatch Log Stream: " + context.logGroupName + " " + context.logStreamName,
      PhysicalResourceId: physicalResourceId || context.logStreamName,
      StackId: event.StackId,
      RequestId: event.RequestId,
      LogicalResourceId: event.LogicalResourceId,
      NoEcho: noEcho || false,
      Data: responseData
    });
    
    console.log("Response body:\
    ", responseBody);
    
    var https = require("https");
    var url = require("url");
    
    var parsedUrl = url.parse(event.ResponseURL);
    var options = {
      hostname: parsedUrl.hostname,
      port: 443,
      path: parsedUrl.path,
      method: "PUT",
      headers: {
        "content-type": "",
        "content-length": responseBody.length
      }
    };
    
    var request = https.request(options, function(response) {
      console.log("Status code: " + response.statusCode);
      console.log("Status message: " + response.statusMessage);
      resolve();
    });
    
    request.on("error", function(error) {
      console.log("respond(..) failed executing https.request(..): " + error);
      resolve();
    });
    
    request.write(responseBody);
    request.end();
  });
};

const AWS = require('aws-sdk');

exports.handler = async function (event, context) {
  console.log(JSON.stringify(event, null, 4));
  
  const projectName = event['detail']['project-name'];
  
  const codebuild = new AWS.CodeBuild();
  
  const buildId = event['detail']['build-id'];
  const { builds } = await codebuild.batchGetBuilds({
    ids: [ buildId ]
  }).promise();
  
  console.log(JSON.stringify(builds, null, 4));
  
  const build = builds[0];
  // Fetch the CFN resource and response parameters from the build environment.
  const environment = {};
  build.environment.environmentVariables.forEach(e => environment[e.name] = e.value);
  
  const response = {
    ResponseURL: environment.CFN_RESPONSE_URL,
    StackId: environment.CFN_STACK_ID,
    LogicalResourceId: environment.CFN_LOGICAL_RESOURCE_ID,
    RequestId: environment.CFN_REQUEST_ID
  };
  
  if (event['detail']['build-status'] === 'SUCCEEDED') {
    await respond(response, context, 'SUCCESS', { BuildStatus: 'SUCCESS'}, 'build');
  } else {
    await respond(response, context, 'FAILED', { Error: 'Build failed' });
  }
};
`;
//# sourceMappingURL=data:application/json;base64,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