"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynchronousGlueJob = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const pre_bundled_layer_1 = require("../common/pre-bundled-layer");
const aws_glue_alpha_1 = require("@aws-cdk/aws-glue-alpha");
/**
 * SynchronousGlueJob Construct to start an AWS Glue Job execution and wait for completion during CDK deploy
 */
class SynchronousGlueJob extends constructs_1.Construct {
    /**
     * Constructs a new instance of the DataGenerator class
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {JobProps} props the SynchronousGlueJob properties
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        const stack = aws_cdk_lib_1.Stack.of(this);
        const glueJob = new aws_glue_alpha_1.Job(scope, 'SynchronousGlueJob', props);
        aws_cdk_lib_1.Tags.of(glueJob).add('for-use-with', 'synchronous-glue-job');
        //Lambda policy to allow starting a Glue job
        const lambdaCRPolicy = [new aws_iam_1.PolicyStatement({
                resources: [
                    stack.formatArn({
                        region: aws_cdk_lib_1.Aws.REGION,
                        account: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        service: 'glue',
                        resource: 'job',
                        resourceName: '*',
                    }),
                ],
                conditions: { StringEquals: { 'aws:ResourceTag/for-use-with': 'synchronous-glue-job' } },
                actions: [
                    'glue:StartJobRun',
                    'glue:GetJobRun',
                    'glue:BatchStopJobRun',
                ],
            })];
        // AWS Lambda function for the AWS CDK Custom Resource responsible to start the job
        const glueJobStartFn = new pre_bundled_function_1.PreBundledFunction(this, 'StartFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'synchronous-glue-job/resources/lambdas',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: lambdaCRPolicy,
            handler: 'lambda.on_event',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        // AWS Lambda function for the AWS CDK Custom Resource responsible to wait for crawler completion
        const glueJobWaitFn = new pre_bundled_function_1.PreBundledFunction(this, 'WaitFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'synchronous-glue-job/resources/lambdas',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: lambdaCRPolicy,
            handler: 'lambda.is_complete',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        // Create an AWS CDK Custom Resource Provider for starting the source crawler and waiting for completion
        const glueJobStartWaitCRP = new custom_resources_1.Provider(this, 'SynchronousGlueJobCRP', {
            onEventHandler: glueJobStartFn,
            isCompleteHandler: glueJobWaitFn,
            queryInterval: aws_cdk_lib_1.Duration.seconds(120),
            totalTimeout: props.timeout,
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        // Create an AWS CDK Custom Resource for starting the source crawler and waiting for completion
        const customResource = new aws_cdk_lib_1.CustomResource(this, 'SynchronousGlueJobCR', {
            serviceToken: glueJobStartWaitCRP.serviceToken,
            properties: {
                JobName: glueJob.jobName,
            },
        });
        this.glueJobLogStream = customResource.getAttString('LogGroupName');
        // Force the dependency because jobName could be known at synth time
        customResource.node.addDependency(glueJob);
    }
}
_a = JSII_RTTI_SYMBOL_1;
SynchronousGlueJob[_a] = { fqn: "aws-analytics-reference-architecture.SynchronousGlueJob", version: "2.12.9" };
exports.SynchronousGlueJob = SynchronousGlueJob;
//# sourceMappingURL=data:application/json;base64,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