"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3CrossAccount = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
/**
 * This CDK construct grants cross account permissions on an Amazon S3 location.
 * It uses a bucket policy and an Amazon KMS Key policy if the bucket is encrypted with KMS.
 * The cross account permission is granted to the entire account and not to a specific principal in this account.
 * It's the responsibility of the target account to grant permissions to the relevant principals.
 *
 * Note that it uses a Bucket object and not an IBucket because CDK can only add policies to objects managed in the CDK stack.
 *
 * Usage example:
 * ```typescript
 * import * as cdk from 'aws-cdk-lib';
 * import { S3CrossAccount } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new cdk.App();
 * const stack = new cdk.Stack(exampleApp, 'S3CrossAccountStack');
 *
 * const myBucket = new Bucket(stack, 'MyBucket')
 *
 * new S3CrossAccount(stack, 'S3CrossAccountGrant', {
 *   bucket: myBucket,
 *   s3ObjectKey: 'my-data',
 *   accountId: '1234567891011',
 * });
 * ```
 */
class S3CrossAccount extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Get the target account as a Principal
        const targetAccount = new aws_iam_1.AccountPrincipal(props.accountId);
        // // Get the bucket from the S3 location to attache a bucket policy
        // props.bucket.addToResourcePolicy(
        //   new PolicyStatement({
        //     principals: [
        //       targetAccount,
        //     ],
        //     actions: [
        //       's3:GetObject',
        //       's3:PutObject',
        //       's3:DeleteObject',
        //       's3:ListBucketMultipartUploads',
        //       's3:ListMultipartUploadParts',
        //       's3:AbortMultipartUpload',
        //       's3:ListBucket',
        //     ],
        //     resources: [
        //       props.bucket.arnForObjects(props.objectKey || '') + '/*',
        //       props.bucket.bucketArn,
        //     ],
        //   }),
        // );
        // // If the bucket is encrypted with a custom KMS key, attach a policy to the key to grant encrypt and decrypt
        // if (props.bucket.encryptionKey)  props.bucket.encryptionKey.grantEncryptDecrypt(targetAccount);
        if (props.s3ObjectKey === undefined)
            props.s3Bucket.grantReadWrite(targetAccount);
        else
            props.s3Bucket.grantReadWrite(targetAccount, props.s3ObjectKey);
        props.s3Bucket.grantReadWrite(targetAccount, props.s3ObjectKey + '/*');
        // Grant target account access to encryption configuration of an S3 bucket
        props.s3Bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            principals: [targetAccount],
            actions: [
                's3:GetEncryptionConfiguration'
            ],
            resources: [
                props.s3Bucket.bucketArn,
            ]
        }));
    }
}
_a = JSII_RTTI_SYMBOL_1;
S3CrossAccount[_a] = { fqn: "aws-analytics-reference-architecture.S3CrossAccount", version: "2.12.9" };
exports.S3CrossAccount = S3CrossAccount;
//# sourceMappingURL=data:application/json;base64,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