"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LakeFormationS3Location = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const lakeformation = require("aws-cdk-lib/aws-lakeformation");
const constructs_1 = require("constructs");
/**
 * This CDK construct aims to register an S3 Location for Lakeformation with Read and Write access.
 * If the location is in a different account, cross account access should be granted via the [S3CrossAccount]{@link S3CrossAccount} construct.
 * If the S3 location is encrypted with KMS, the key must be explicitly passed to the construct because CDK cannot retrieve bucket encryption key from imported buckets.
 * Imported buckets are generally used in cross account setup like data mesh.
 *
 * This construct instantiate 2 objects:
 * * An IAM role with read/write permissions to the S3 location and encrypt/decrypt access to the KMS key used to encypt the bucket
 * * A CfnResource is based on an IAM role with 2 policy statement folowing the least privilege AWS best practices:
 *   * Statement 1 for S3 permissions
 *   * Statement 2 for KMS permissions if the bucket is encrypted
 *
 * The CDK construct instantiate the CfnResource in order to register the S3 location with Lakeformation using the IAM role defined above.
 *
 * Usage example:
 * ```typescript
 * import * as cdk from 'aws-cdk-lib';
 * import { LakeformationS3Location } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new cdk.App();
 * const stack = new cdk.Stack(exampleApp, 'LakeformationS3LocationStack');
 *
 * const myKey = new Key(stack, 'MyKey')
 * const myBucket = new Bucket(stack, 'MyBucket', {
 *   encryptionKey: myKey,
 * })
 *
 * new LakeFormationS3Location(stack, 'MyLakeformationS3Location', {
 *   s3Location: {
 *     bucketName: myBucket.bucketName,
 *     objectKey: 'my-prefix',
 *   },
 *   kmsKeyId: myBucket.encryptionKey.keyId,
 * });
 * ```
 */
class LakeFormationS3Location extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Create an Amazon IAM Role used by Lakeformation to register S3 location
        this.dataAccessRole = new aws_iam_1.Role(this, 'LFS3AccessRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lakeformation.amazonaws.com'),
        });
        // add policy to access S3 for Read and Write
        this.dataAccessRole.addToPolicy(new aws_iam_1.PolicyStatement({
            resources: [
                aws_cdk_lib_1.Arn.format({
                    account: '',
                    region: '',
                    resource: props.s3Location.bucketName,
                    service: 's3',
                    resourceName: props.s3Location.objectKey ? props.s3Location.objectKey + '/*' : '*',
                }, aws_cdk_lib_1.Stack.of(this)),
                aws_cdk_lib_1.Arn.format({
                    account: '',
                    region: '',
                    resource: props.s3Location.bucketName,
                    service: 's3',
                }, aws_cdk_lib_1.Stack.of(this)),
            ],
            actions: [
                "s3:GetObject*",
                "s3:GetBucket*",
                "s3:List*",
                "s3:DeleteObject*",
                "s3:PutObject",
                "s3:PutObjectLegalHold",
                "s3:PutObjectRetention",
                "s3:PutObjectTagging",
                "s3:PutObjectVersionTagging",
                "s3:Abort*",
            ],
        }));
        // add policy to access KMS key used for the bucket encryption
        if (props.kmsKeyId) {
            this.dataAccessRole.addToPolicy(new aws_iam_1.PolicyStatement({
                resources: [
                    aws_cdk_lib_1.Arn.format({
                        region: aws_cdk_lib_1.Aws.REGION,
                        resource: 'key',
                        service: 'kms',
                        account: props.accountId ? props.accountId : aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        resourceName: props.kmsKeyId,
                    }, aws_cdk_lib_1.Stack.of(this))
                ],
                actions: [
                    'kms:Encrypt*',
                    'kms:Decrypt*',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:Describe*',
                ],
            }));
        }
        new lakeformation.CfnResource(this, 'RegisterLocation', {
            resourceArn: aws_cdk_lib_1.Arn.format({
                account: '',
                region: '',
                resource: props.s3Location.bucketName,
                service: 's3',
                resourceName: props.s3Location.objectKey ? props.s3Location.objectKey + '/*' : '*',
            }, aws_cdk_lib_1.Stack.of(this)),
            useServiceLinkedRole: false,
            roleArn: this.dataAccessRole.roleArn,
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
LakeFormationS3Location[_a] = { fqn: "aws-analytics-reference-architecture.LakeFormationS3Location", version: "2.12.9" };
exports.LakeFormationS3Location = LakeFormationS3Location;
//# sourceMappingURL=data:application/json;base64,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