"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.vpcBootstrap = void 0;
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * @internal
 * Upload podTemplates to the Amazon S3 location used by the cluster.
 * @param {Construct} scope The local path of the yaml podTemplate files to upload
 * @param {string} vpcCidr The cidr for vpc
 * @param {string} eksClusterName The name used to tag the subnet and vpc
 */
function vpcBootstrap(scope, vpcCidr, eksClusterName) {
    const vpcMask = parseInt(vpcCidr.split('/')[1]);
    // Calculate subnet masks based on VPC's mask
    const publicSubnetMask = vpcMask + 4;
    const privateSubnetMask = publicSubnetMask + 2; // twice as large as public subnet
    const vpc = new aws_ec2_1.Vpc(scope, 'MyVPC', {
        ipAddresses: aws_ec2_1.IpAddresses.cidr(vpcCidr),
        maxAzs: 3,
        natGateways: 3,
        subnetConfiguration: [
            {
                cidrMask: publicSubnetMask,
                name: 'Public',
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            },
            {
                cidrMask: privateSubnetMask,
                name: 'Private',
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
            },
        ],
    });
    // Create a gateway endpoint for S3
    vpc.addGatewayEndpoint('S3Endpoint', {
        service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
    });
    // Add tags to subnets
    for (let subnet of [...vpc.publicSubnets, ...vpc.privateSubnets]) {
        aws_cdk_lib_1.Tags.of(subnet).add('karpenter.sh/discovery', eksClusterName);
    }
    // Add tags to vpc
    aws_cdk_lib_1.Tags.of(vpc).add('karpenter.sh/discovery', eksClusterName);
    return vpc;
}
exports.vpcBootstrap = vpcBootstrap;
//# sourceMappingURL=data:application/json;base64,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