"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrEksCluster = exports.EmrVersion = exports.Autoscaler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const aws_emrcontainers_1 = require("aws-cdk-lib/aws-emrcontainers");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const SimpleBase = require("simple-base");
const ara_bucket_1 = require("../ara-bucket");
const context_options_1 = require("../common/context-options");
const tracked_construct_1 = require("../common/tracked-construct");
const singleton_kms_key_1 = require("../singleton-kms-key");
const emr_managed_endpoint_1 = require("./emr-managed-endpoint");
const CriticalDefaultConfig = require("./resources/k8s/emr-eks-config/critical.json");
const NotebookDefaultConfig = require("./resources/k8s/emr-eks-config/notebook-pod-template-ready.json");
const SharedDefaultConfig = require("./resources/k8s/emr-eks-config/shared.json");
const K8sRoleBinding = require("./resources/k8s/rbac/emr-containers-role-binding.json");
const K8sRole = require("./resources/k8s/rbac/emr-containers-role.json");
const emr_eks_cluster_helpers_1 = require("./emr-eks-cluster-helpers");
const singleton_launch_template_1 = require("../singleton-launch-template");
const emr_eks_nodegroup_asg_tag_1 = require("./emr-eks-nodegroup-asg-tag");
const emr_eks_job_template_1 = require("./emr-eks-job-template");
const vpc_helper_1 = require("./vpc-helper");
/**
 * The different autoscaler available with EmrEksCluster
 */
var Autoscaler;
(function (Autoscaler) {
    Autoscaler["KARPENTER"] = "KARPENTER";
    Autoscaler["CLUSTER_AUTOSCALER"] = "CLUSTER_AUTOSCALER";
})(Autoscaler = exports.Autoscaler || (exports.Autoscaler = {}));
/**
 * The different EMR versions available on EKS
 */
var EmrVersion;
(function (EmrVersion) {
    EmrVersion["V6_14"] = "emr-6.14.0-latest";
    EmrVersion["V6_13"] = "emr-6.13.0-latest";
    EmrVersion["V6_12"] = "emr-6.12.0-latest";
    EmrVersion["V6_11"] = "emr-6.11.0-latest";
    EmrVersion["V6_10"] = "emr-6.10.0-latest";
    EmrVersion["V6_9"] = "emr-6.9.0-latest";
    EmrVersion["V6_8"] = "emr-6.8.0-latest";
    EmrVersion["V6_7"] = "emr-6.7.0-latest";
    EmrVersion["V6_6"] = "emr-6.6.0-latest";
    EmrVersion["V6_5"] = "emr-6.5.0-latest";
    EmrVersion["V6_4"] = "emr-6.4.0-latest";
    EmrVersion["V6_3"] = "emr-6.3.0-latest";
    EmrVersion["V6_2"] = "emr-6.2.0-latest";
    EmrVersion["V5_33"] = "emr-5.33.0-latest";
    EmrVersion["V5_32"] = "emr-5.32.0-latest";
})(EmrVersion = exports.EmrVersion || (exports.EmrVersion = {}));
/**
 * EmrEksCluster Construct packaging all the resources and configuration required to run Amazon EMR on EKS.
 * It deploys:
 * * An EKS cluster (VPC configuration can be customized)
 * * A tooling nodegroup to run tools including the Kubedashboard and the Cluster Autoscaler
 * * Optionally multiple nodegroups (one per AZ) for critical/shared/notebook EMR workloads
 * * Additional nodegroups can be configured
 *
 * The construct will upload on S3 the Pod templates required to run EMR jobs on the default nodegroups.
 * It will also parse and store the configuration of EMR on EKS jobs for each default nodegroup in object parameters
 *
 * Methods are available to add EMR Virtual Clusters to the EKS cluster and to create execution roles for the virtual clusters.
 *
 * Usage example:
 *
 * ```typescript
 * const emrEks: EmrEksCluster = EmrEksCluster.getOrCreate(stack, {
 *   eksAdminRoleArn: <ROLE_ARN>,
 *   eksClusterName: <CLUSTER_NAME>,
 * });
 *
 * const virtualCluster = emrEks.addEmrVirtualCluster(stack, {
 *   name: <Virtual_Cluster_Name>,
 *   createNamespace: <TRUE OR FALSE>,
 *   eksNamespace: <K8S_namespace>,
 * });
 *
 * const role = emrEks.createExecutionRole(stack, 'ExecRole',{
 *   policy: <POLICY>,
 * })
 *
 * // EMR on EKS virtual cluster ID
 * cdk.CfnOutput(self, 'VirtualClusterId',value = virtualCluster.attr_id)
 * // Job config for each nodegroup
 * cdk.CfnOutput(self, "CriticalConfig", value = emrEks.criticalDefaultConfig)
 * cdk.CfnOutput(self, "SharedConfig", value = emrEks.sharedDefaultConfig)
 * // Execution role arn
 * cdk.CfnOutput(self,'ExecRoleArn', value = role.roleArn)
 * ```
 *
 */
class EmrEksCluster extends tracked_construct_1.TrackedConstruct {
    /**
     * Get an existing EmrEksCluster based on the cluster name property or create a new one
     * only one EKS cluster can exist per stack
     * @param {Construct} scope the CDK scope used to search or create the cluster
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps} if created
     */
    static getOrCreate(scope, props) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = props.eksClusterName || EmrEksCluster.DEFAULT_CLUSTER_NAME;
        let emrEksCluster;
        if (stack.node.tryFindChild(id) == undefined) {
            emrEksCluster = new EmrEksCluster(stack, id, props);
        }
        return stack.node.tryFindChild(id) || emrEksCluster;
    }
    /**
     * Constructs a new instance of the EmrEksCluster construct.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps}
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingCode: context_options_1.ContextOptions.EMR_EKS_TRACKING_ID,
        };
        super(scope, id, trackedConstructProps);
        this.clusterName = props.eksClusterName ?? EmrEksCluster.DEFAULT_CLUSTER_NAME;
        //Set the flag for creating the EMR on EKS Service Linked Role
        this.createEmrOnEksServiceLinkedRole = props.createEmrOnEksServiceLinkedRole ?? true;
        //Define EKS cluster logging
        const eksClusterLogging = [
            aws_eks_1.ClusterLoggingTypes.API,
            aws_eks_1.ClusterLoggingTypes.AUTHENTICATOR,
            aws_eks_1.ClusterLoggingTypes.SCHEDULER,
            aws_eks_1.ClusterLoggingTypes.CONTROLLER_MANAGER,
            aws_eks_1.ClusterLoggingTypes.AUDIT,
        ];
        //Set the autoscaler mechanism flag
        this.isKarpenter = props.autoscaling == Autoscaler.KARPENTER ? true : false;
        this.defaultNodes = props.defaultNodes == undefined ? true : props.defaultNodes;
        // Create a role to be used as instance profile for nodegroups
        this.ec2InstanceNodeGroupRole = new aws_iam_1.Role(this, 'ec2InstanceNodeGroupRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
        });
        //attach policies to the role to be used by the nodegroups
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        // Create the custom resource provider for tagging the EC2 Auto Scaling groups
        this.nodegroupAsgTagsProviderServiceToken = new emr_eks_nodegroup_asg_tag_1.EmrEksNodegroupAsgTagProvider(this, 'AsgTagProvider', {
            eksClusterName: this.clusterName,
        }).provider.serviceToken;
        // Create the custom resource provider for tagging the EC2 Auto Scaling groups
        this.jobTemplateProviderToken = new emr_eks_job_template_1.EmrEksJobTemplateProvider(this, 'jobTemplateProvider').provider.serviceToken;
        // create an Amazon EKS CLuster with default parameters if not provided in the properties
        if (props.eksCluster == undefined) {
            let eksVpc = props.vpcCidr ? (0, vpc_helper_1.vpcBootstrap)(scope, props.vpcCidr, this.clusterName) : props.eksVpc;
            this.eksCluster = new aws_eks_1.Cluster(scope, `${this.clusterName}Cluster`, {
                defaultCapacity: 0,
                clusterName: this.clusterName,
                version: props.kubernetesVersion || EmrEksCluster.DEFAULT_EKS_VERSION,
                clusterLogging: eksClusterLogging,
                kubectlLayer: props.kubectlLambdaLayer ?? undefined,
                vpc: eksVpc,
                albController: {
                    version: aws_eks_1.AlbControllerVersion.V2_5_1,
                }
            });
            //Create VPC flow log for the EKS VPC
            let eksVpcFlowLogLogGroup = new aws_logs_1.LogGroup(this, 'eksVpcFlowLogLogGroup', {
                logGroupName: `/aws/emr-eks-vpc-flow/${this.clusterName}`,
                encryptionKey: singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey'),
                retention: aws_logs_1.RetentionDays.ONE_WEEK,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            });
            //Allow vpc flowlog to access KMS key to encrypt logs
            singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey').addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                principals: [new aws_iam_1.ServicePrincipal(`logs.${aws_cdk_lib_1.Stack.of(this).region}.amazonaws.com`)],
                actions: [
                    'kms:Encrypt*',
                    'kms:Decrypt*',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:Describe*',
                ],
                conditions: {
                    ArnLike: {
                        'kms:EncryptionContext:aws:logs:arn': `arn:aws:logs:${aws_cdk_lib_1.Stack.of(this).region}:${aws_cdk_lib_1.Stack.of(this).account}:*`,
                    },
                },
                resources: ['*'],
            }));
            //Setup the VPC flow logs
            const iamRoleforFlowLog = new aws_iam_1.Role(this, 'iamRoleforFlowLog', {
                assumedBy: new aws_iam_1.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
            this.eksCluster.vpc.addFlowLog('eksVpcFlowLog', {
                destination: aws_ec2_1.FlowLogDestination.toCloudWatchLogs(eksVpcFlowLogLogGroup, iamRoleforFlowLog),
            });
            //Setting up the cluster with the required controller
            (0, emr_eks_cluster_helpers_1.eksClusterSetup)(this, this, props.eksAdminRoleArn);
            //Deploy the right autoscaler using the flag set earlier
            if (this.isKarpenter) {
                this.karpenterChart = (0, emr_eks_cluster_helpers_1.karpenterSetup)(this.eksCluster, this.clusterName, this, props.karpenterVersion || EmrEksCluster.DEFAULT_KARPENTER_VERSION);
            }
            else {
                const kubernetesVersion = props.kubernetesVersion ?? EmrEksCluster.DEFAULT_EKS_VERSION;
                (0, emr_eks_cluster_helpers_1.clusterAutoscalerSetup)(this.eksCluster, this.clusterName, this, kubernetesVersion);
            }
        }
        else {
            //Initialize with the provided EKS Cluster
            this.eksCluster = props.eksCluster;
        }
        //Check if the user want to use the default nodegroup and
        //Add the default nodegroup configured and optimized to run Spark workloads
        if (this.defaultNodes && props.autoscaling == Autoscaler.CLUSTER_AUTOSCALER) {
            (0, emr_eks_cluster_helpers_1.setDefaultManagedNodeGroups)(this);
        }
        //Check if there user want to use the default Karpenter provisioners and
        //Add the defaults pre-configured and optimized to run Spark workloads
        if (this.defaultNodes && props.autoscaling == Autoscaler.KARPENTER) {
            (0, emr_eks_cluster_helpers_1.setDefaultKarpenterProvisioners)(this);
        }
        ara_bucket_1.AraBucket.getOrCreate(this, { bucketName: 's3-access-logs', objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_PREFERRED });
        // Tags the Amazon VPC and Subnets of the Amazon EKS Cluster
        aws_cdk_lib_1.Tags.of(this.eksCluster.vpc).add('for-use-with-amazon-emr-managed-policies', 'true');
        this.eksCluster.vpc.privateSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('for-use-with-amazon-emr-managed-policies', 'true'));
        this.eksCluster.vpc.publicSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('for-use-with-amazon-emr-managed-policies', 'true'));
        // Create Amazon IAM ServiceLinkedRole for Amazon EMR and add to kubernetes configmap
        // required to add a dependency on the Amazon EMR virtual cluster
        if (this.createEmrOnEksServiceLinkedRole) {
            this.emrServiceRole = new aws_iam_1.CfnServiceLinkedRole(this, 'EmrServiceRole', {
                awsServiceName: 'emr-containers.amazonaws.com',
            });
        }
        this.eksCluster.awsAuth.addRoleMapping(aws_iam_1.Role.fromRoleArn(this, 'ServiceRoleForAmazonEMRContainers', `arn:aws:iam::${aws_cdk_lib_1.Stack.of(this).account}:role/AWSServiceRoleForAmazonEMRContainers`), {
            username: 'emr-containers',
            groups: ['']
        });
        // Create an Amazon S3 Bucket for default podTemplate assets
        this.assetBucket = ara_bucket_1.AraBucket.getOrCreate(this, { bucketName: `${this.clusterName.toLowerCase()}-emr-eks-assets`, encryption: aws_s3_1.BucketEncryption.KMS_MANAGED });
        // Configure the podTemplate location
        this.podTemplateLocation = {
            bucketName: this.assetBucket.bucketName,
            objectKey: `${this.clusterName}/pod-template`,
        };
        aws_cdk_lib_1.Tags.of(this.assetBucket).add('for-use-with', 'cdk-analytics-reference-architecture');
        let s3DeploymentLambdaPolicyStatement = [];
        s3DeploymentLambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        //Policy to allow lambda access to cloudwatch logs
        const lambdaExecutionRolePolicy = new aws_iam_1.ManagedPolicy(this, 's3BucketDeploymentPolicy', {
            statements: s3DeploymentLambdaPolicyStatement,
            description: 'Policy used by S3 deployment cdk construct',
        });
        //Create an execution role for the lambda and attach to it a policy formed from user input
        this.assetUploadBucketRole = new aws_iam_1.Role(this, 's3BucketDeploymentRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            description: 'Role used by S3 deployment cdk construct',
            managedPolicies: [lambdaExecutionRolePolicy],
        });
        // Upload the default podTemplate to the Amazon S3 asset bucket
        this.uploadPodTemplate('defaultPodTemplates', (0, path_1.join)(__dirname, 'resources/k8s/pod-template'));
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the notebook default config
        NotebookDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-driver.yaml`);
        NotebookDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-executor.yaml`);
        this.notebookDefaultConfig = JSON.parse(JSON.stringify(NotebookDefaultConfig));
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the critical default config
        CriticalDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-driver.yaml`);
        CriticalDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-executor.yaml`);
        this.criticalDefaultConfig = JSON.stringify(CriticalDefaultConfig);
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the shared default config
        SharedDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-driver.yaml`);
        SharedDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-executor.yaml`);
        this.sharedDefaultConfig = JSON.stringify(SharedDefaultConfig);
        // Set the custom resource provider service token here to avoid circular dependencies
        this.managedEndpointProviderServiceToken = new emr_managed_endpoint_1.EmrManagedEndpointProvider(this, 'ManagedEndpointProvider', {
            assetBucket: this.assetBucket,
        }).provider.serviceToken;
        // Provide the podTemplate location on Amazon S3
        new aws_cdk_lib_1.CfnOutput(this, 'podTemplateLocation', {
            description: 'Use podTemplates in Amazon EMR jobs from this Amazon S3 Location',
            value: this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}`),
        });
    }
    /**
     * Add a new Amazon EMR Virtual Cluster linked to Amazon EKS Cluster.
     * @param {Construct} scope of the stack where virtual cluster is deployed
     * @param {EmrVirtualClusterOptions} options the EmrVirtualClusterProps [properties]{@link EmrVirtualClusterProps}
     */
    addEmrVirtualCluster(scope, options) {
        const eksNamespace = options.eksNamespace ?? 'default';
        const regex = /^[a-z0-9]+$/g;
        if (!eksNamespace.match(regex)) {
            throw new Error(`Namespace provided violates the constraints of Namespace naming ${eksNamespace}`);
        }
        const ns = options.createNamespace
            ? this.eksCluster.addManifest(`${options.name}Namespace`, {
                apiVersion: 'v1',
                kind: 'Namespace',
                metadata: { name: eksNamespace },
            })
            : null;
        // deep clone the Role template object and replace the namespace
        const k8sRole = JSON.parse(JSON.stringify(K8sRole));
        k8sRole.metadata.namespace = eksNamespace;
        const role = this.eksCluster.addManifest(`${options.name}Role`, k8sRole);
        if (this.createEmrOnEksServiceLinkedRole)
            role.node.addDependency(this.emrServiceRole);
        if (ns)
            role.node.addDependency(ns);
        // deep clone the Role Binding template object and replace the namespace
        const k8sRoleBinding = JSON.parse(JSON.stringify(K8sRoleBinding));
        k8sRoleBinding.metadata.namespace = eksNamespace;
        const roleBinding = this.eksCluster.addManifest(`${options.name}RoleBinding`, k8sRoleBinding);
        roleBinding.node.addDependency(role);
        const virtCluster = new aws_emrcontainers_1.CfnVirtualCluster(scope, `${options.name}VirtualCluster`, {
            name: options.name,
            containerProvider: {
                id: this.clusterName,
                type: 'EKS',
                info: { eksInfo: { namespace: options.eksNamespace || 'default' } },
            },
            tags: [{
                    key: 'for-use-with',
                    value: 'cdk-analytics-reference-architecture',
                }],
        });
        virtCluster.node.addDependency(roleBinding);
        if (this.createEmrOnEksServiceLinkedRole)
            virtCluster.node.addDependency(this.emrServiceRole);
        if (ns)
            virtCluster.node.addDependency(ns);
        aws_cdk_lib_1.Tags.of(virtCluster).add('for-use-with', 'cdk-analytics-reference-architecture');
        return virtCluster;
    }
    /**
     * Creates a new Amazon EMR managed endpoint to be used with Amazon EMR Virtual Cluster .
     * CfnOutput can be customized.
     * @param {Construct} scope the scope of the stack where managed endpoint is deployed
     * @param {string} id the CDK id for endpoint
     * @param {EmrManagedEndpointOptions} options the EmrManagedEndpointOptions to configure the Amazon EMR managed endpoint
     */
    addManagedEndpoint(scope, id, options) {
        if (options.managedEndpointName.length > 64) {
            throw new Error(`error managed endpoint name length is greater than 64 ${id}`);
        }
        if (this.notebookDefaultConfig == undefined) {
            throw new Error('error empty configuration override is not supported on non-default nodegroups');
        }
        let jsonConfigurationOverrides;
        // TODO this need to be broadended to all possible emr configuration
        // try {
        //   //Check if the configOverride provided by user is valid
        //   let isConfigOverrideValid: boolean = validateSchema(JSON.stringify(configOverrideSchema), options.configurationOverrides);
        //   jsonConfigurationOverrides = isConfigOverrideValid ? options.configurationOverrides : this.notebookDefaultConfig;
        // } catch (error) {
        //   throw new Error(`The configuration override is not valid JSON : ${options.configurationOverrides}`);
        // }
        jsonConfigurationOverrides = options.configurationOverrides ? options.configurationOverrides : this.notebookDefaultConfig;
        // Create custom resource with async waiter until the Amazon EMR Managed Endpoint is created
        const cr = new aws_cdk_lib_1.CustomResource(scope, id, {
            serviceToken: this.managedEndpointProviderServiceToken,
            properties: {
                clusterId: options.virtualClusterId,
                executionRoleArn: options.executionRole.roleArn,
                endpointName: options.managedEndpointName,
                releaseLabel: options.emrOnEksVersion || EmrEksCluster.DEFAULT_EMR_VERSION,
                configurationOverrides: jsonConfigurationOverrides,
            },
        });
        cr.node.addDependency(this.eksCluster);
        return cr;
    }
    /**
     * Add new nodegroups to the cluster for Amazon EMR on EKS. This method overrides Amazon EKS nodegroup options then create the nodegroup.
     * If no subnet is provided, it creates one nodegroup per private subnet in the Amazon EKS Cluster.
     * If NVME local storage is used, the user_data is modified.
     * @param {string} id the CDK ID of the resource
     * @param {EmrEksNodegroupOptions} props the EmrEksNodegroupOptions [properties]{@link EmrEksNodegroupOptions}
     */
    addEmrEksNodegroup(id, props) {
        if (this.isKarpenter) {
            throw new Error(`You cannot use this method when the autoscaler is set to ${Autoscaler.KARPENTER}`);
        }
        // Get the subnet from Properties or one private subnet for each AZ
        const subnetList = props.subnet ? [props.subnet] : this.eksCluster.vpc.selectSubnets({
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
        }).subnets;
        // Add Amazon SSM agent to the user data
        var userData = [
            'yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm',
            'systemctl enable amazon-ssm-agent',
            'systemctl start amazon-ssm-agent',
        ];
        var launchTemplateName = `EmrEksLaunch-${this.clusterName}`;
        // If the Nodegroup uses NVMe, add user data to configure them
        if (props.mountNvme) {
            userData = userData.concat([
                'INSTANCE_TYPE=$(ec2-metadata -t)',
                'if [[ $INSTANCE_TYPE == *"2xlarge"* ]]; then',
                'DEVICE="/dev/nvme1n1"',
                'mkfs.ext4 $DEVICE',
                'else',
                'yum install -y mdadm',
                'SSD_NVME_DEVICE_LIST=("/dev/nvme1n1" "/dev/nvme2n1")',
                'SSD_NVME_DEVICE_COUNT=${#SSD_NVME_DEVICE_LIST[@]}',
                'RAID_DEVICE=${RAID_DEVICE:-/dev/md0}',
                'RAID_CHUNK_SIZE=${RAID_CHUNK_SIZE:-512}  # Kilo Bytes',
                'FILESYSTEM_BLOCK_SIZE=${FILESYSTEM_BLOCK_SIZE:-4096}  # Bytes',
                'STRIDE=$((RAID_CHUNK_SIZE * 1024 / FILESYSTEM_BLOCK_SIZE))',
                'STRIPE_WIDTH=$((SSD_NVME_DEVICE_COUNT * STRIDE))',
                'mdadm --create --verbose "$RAID_DEVICE" --level=0 -c "${RAID_CHUNK_SIZE}" --raid-devices=${#SSD_NVME_DEVICE_LIST[@]} "${SSD_NVME_DEVICE_LIST[@]}"',
                'while [ -n "$(mdadm --detail "$RAID_DEVICE" | grep -ioE \'State :.*resyncing\')" ]; do',
                'echo "Raid is resyncing.."',
                'sleep 1',
                'done',
                'echo "Raid0 device $RAID_DEVICE has been created with disks ${SSD_NVME_DEVICE_LIST[*]}"',
                'mkfs.ext4 -m 0 -b "$FILESYSTEM_BLOCK_SIZE" -E "stride=$STRIDE,stripe-width=$STRIPE_WIDTH" "$RAID_DEVICE"',
                'DEVICE=$RAID_DEVICE',
                'fi',
                'systemctl stop docker',
                'mkdir -p /var/lib/kubelet/pods',
                'mount $DEVICE /var/lib/kubelet/pods',
                'chmod 750 /var/lib/docker',
                'systemctl start docker',
            ]);
            launchTemplateName = `EmrEksNvmeLaunch-${this.clusterName}`;
        }
        // Add headers and footers to user data
        const userDataMime = aws_cdk_lib_1.Fn.base64(`MIME-Version: 1.0
Content-Type: multipart/mixed; boundary="==MYBOUNDARY=="

--==MYBOUNDARY==
Content-Type: text/x-shellscript; charset="us-ascii"

#!/bin/bash
${userData.join('\r\n')}

--==MYBOUNDARY==--\\
`);
        // Create a new LaunchTemplate or reuse existing one
        const lt = singleton_launch_template_1.SingletonCfnLaunchTemplate.getOrCreate(this, launchTemplateName, userDataMime);
        // Create one Amazon EKS Nodegroup per subnet
        subnetList.forEach((subnet, index) => {
            // Make the ID unique across AZ using the index of subnet in the subnet list
            const resourceId = `${id}-${index}`;
            const nodegroupName = props.nodegroupName ? `${props.nodegroupName}-${index}` : resourceId;
            // Add the user data to the NodegroupOptions
            const nodeGroupParameters = {
                ...props,
                ...{
                    launchTemplateSpec: {
                        id: lt.ref,
                        version: lt.attrLatestVersionNumber,
                    },
                    subnets: {
                        subnets: [subnet],
                    },
                    nodegroupName: nodegroupName,
                },
            };
            // Create the Amazon EKS Nodegroup
            this.addNodegroupCapacity(resourceId, nodeGroupParameters);
        });
    }
    /**
     * Add a new Amazon EKS Nodegroup to the cluster.
     * This method is used to add a nodegroup to the Amazon EKS cluster and automatically set tags based on labels and taints
     *  so it can be used for the cluster autoscaler.
     * @param {string} nodegroupId the ID of the nodegroup
     * @param {EmrEksNodegroupOptions} options the EmrEksNodegroup [properties]{@link EmrEksNodegroupOptions}
     */
    addNodegroupCapacity(nodegroupId, options) {
        const nodegroup = this.eksCluster.addNodegroupCapacity(nodegroupId, options);
        // Adding the Amazon SSM policy
        nodegroup.role.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // Add tags for the Cluster Autoscaler IAM scoping
        aws_cdk_lib_1.Tags.of(nodegroup).add('eks:cluster-name', `${this.clusterName}`);
        // Add tags for the Cluster Autoscaler management
        aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/enabled', 'true', {
            applyToLaunchedInstances: true,
        });
        // Add tag for the AZ
        if (options.subnets && options.subnets.subnets) {
            aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/node-template/label/topology.kubernetes.io/zone', options.subnets.subnets[0].availabilityZone, {
                applyToLaunchedInstances: true,
            });
        }
        // Add tag for the lifecycle type (spot or on-demand)
        aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/node-template/label/eks.amazonaws.com/capacityType', (options.capacityType == aws_eks_1.CapacityType.SPOT) ? 'SPOT' : 'ON_DEMAND', {
            applyToLaunchedInstances: true,
        });
        // Iterate over labels and add appropriate tags
        if (options.labels) {
            for (const [key, value] of Object.entries(options.labels)) {
                aws_cdk_lib_1.Tags.of(nodegroup).add(`k8s.io/cluster-autoscaler/node-template/label/${key}`, value, {
                    applyToLaunchedInstances: true,
                });
                new aws_cdk_lib_1.CustomResource(this, `${nodegroupId}Label${key}`, {
                    serviceToken: this.nodegroupAsgTagsProviderServiceToken,
                    properties: {
                        nodegroupName: options.nodegroupName,
                        tagKey: `k8s.io/cluster-autoscaler/node-template/label/${key}`,
                        tagValue: value,
                    },
                }).node.addDependency(nodegroup);
            }
        }
        // Iterate over taints and add appropriate tags
        if (options.taints) {
            options.taints.forEach((taint) => {
                aws_cdk_lib_1.Tags.of(nodegroup).add(`k8s.io/cluster-autoscaler/node-template/taint/${taint.key}`, `${taint.value}:${taint.effect}`, {
                    applyToLaunchedInstances: true,
                });
                new aws_cdk_lib_1.CustomResource(this, `${nodegroupId}Taint${taint.key}`, {
                    serviceToken: this.nodegroupAsgTagsProviderServiceToken,
                    properties: {
                        nodegroupName: options.nodegroupName,
                        tagKey: `k8s.io/cluster-autoscaler/node-template/taint/${taint.key}`,
                        tagValue: `${taint.value}:${taint.effect}`,
                    },
                }).node.addDependency(nodegroup);
            });
        }
        return nodegroup;
    }
    /**
     * Create and configure a new Amazon IAM Role usable as an execution role.
     * This method makes the created role assumed by the Amazon EKS cluster Open ID Connect provider.
     * @param {Construct} scope of the IAM role
     * @param {string} id of the CDK resource to be created, it should be unique across the stack
     * @param {IManagedPolicy} policy the execution policy to attach to the role
     * @param {string} namespace The namespace from which the role is going to be used. MUST be the same as the namespace of the Virtual Cluster from which the job is submitted
     * @param {string} name Name to use for the role, required and is used to scope the iam role
     */
    createExecutionRole(scope, id, policy, namespace, name) {
        const stack = aws_cdk_lib_1.Stack.of(this);
        let irsaConditionkey = new aws_cdk_lib_1.CfnJson(this, `${id}irsaConditionkey'`, {
            value: {
                [`${this.eksCluster.openIdConnectProvider.openIdConnectProviderIssuer}:sub`]: 'system:serviceaccount:' + namespace + ':emr-containers-sa-*-*-' + aws_cdk_lib_1.Aws.ACCOUNT_ID.toString() + '-' + SimpleBase.base36.encode(name),
            },
        });
        // Create an execution role assumable by EKS OIDC provider
        return new aws_iam_1.Role(scope, `${id}ExecutionRole`, {
            assumedBy: new aws_iam_1.FederatedPrincipal(this.eksCluster.openIdConnectProvider.openIdConnectProviderArn, {
                StringLike: irsaConditionkey,
            }, 'sts:AssumeRoleWithWebIdentity'),
            roleName: name,
            managedPolicies: [policy],
            inlinePolicies: {
                PodTemplateAccess: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                's3:getObject',
                            ],
                            resources: [
                                stack.formatArn({
                                    region: '',
                                    account: '',
                                    service: 's3',
                                    resource: this.podTemplateLocation.bucketName,
                                    resourceName: `${this.podTemplateLocation.objectKey}/*`,
                                }),
                            ],
                        }),
                    ],
                }),
            },
        });
    }
    /**
     * Upload podTemplates to the Amazon S3 location used by the cluster.
     * @param {string} id the unique ID of the CDK resource
     * @param {string} filePath The local path of the yaml podTemplate files to upload
     */
    uploadPodTemplate(id, filePath) {
        new aws_s3_deployment_1.BucketDeployment(this, `${id}AssetDeployment`, {
            destinationBucket: this.assetBucket,
            destinationKeyPrefix: this.podTemplateLocation.objectKey,
            sources: [aws_s3_deployment_1.Source.asset(filePath)],
            role: this.assetUploadBucketRole,
        });
    }
    /**
     * Creates a new Amazon EMR on EKS job template based on the props passed
     * @param {Construct} scope the scope of the stack where job template is created
     * @param {string} id the CDK id for job template resource
     * @param {EmrEksJobTemplateDefinition} options the EmrManagedEndpointOptions to configure the Amazon EMR managed endpoint
     */
    addJobTemplate(scope, id, options) {
        // Create custom resource to execute the create job template boto3 call
        const cr = new aws_cdk_lib_1.CustomResource(scope, id, {
            serviceToken: this.jobTemplateProviderToken,
            properties: {
                name: options.name,
                jobTemplateData: options.jobTemplateData,
            },
        });
        cr.node.addDependency(this.eksCluster);
        return cr;
    }
    /**
     * Apply the provided manifest and add the CDK dependency on EKS cluster
     * @param {string} id the unique ID of the CDK resource
     * @param {any} manifest The manifest to apply.
     * You can use the Utils class that offers method to read yaml file and load it as a manifest
     */
    addKarpenterProvisioner(id, manifest) {
        if (!this.isKarpenter) {
            throw new Error(`You can\'t use this method when the autoscaler is set to ${Autoscaler.KARPENTER}`);
        }
        let manifestApply = this.eksCluster.addManifest(id, ...manifest);
        if (this.karpenterChart) {
            manifestApply.node.addDependency(this.karpenterChart);
        }
        return manifestApply;
    }
}
_a = JSII_RTTI_SYMBOL_1;
EmrEksCluster[_a] = { fqn: "aws-analytics-reference-architecture.EmrEksCluster", version: "2.12.9" };
EmrEksCluster.DEFAULT_EMR_VERSION = EmrVersion.V6_12;
EmrEksCluster.DEFAULT_EKS_VERSION = aws_eks_1.KubernetesVersion.V1_25;
EmrEksCluster.DEFAULT_CLUSTER_NAME = 'data-platform';
EmrEksCluster.DEFAULT_KARPENTER_VERSION = 'v0.20.0';
exports.EmrEksCluster = EmrEksCluster;
//# sourceMappingURL=data:application/json;base64,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