"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.clusterAutoscalerSetup = exports.karpenterSetup = exports.karpenterManifestSetup = exports.setDefaultKarpenterProvisioners = exports.setDefaultManagedNodeGroups = exports.eksClusterSetup = void 0;
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_sqs_1 = require("aws-cdk-lib/aws-sqs");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const utils_1 = require("../utils");
const emr_eks_nodegroup_1 = require("./emr-eks-nodegroup");
const emr_eks_cluster_1 = require("./emr-eks-cluster");
const IamPolicyEbsCsiDriver = require("./resources/k8s/iam-policy-ebs-csi-driver.json");
/**
 * @internal
 * Upload podTemplates to the Amazon S3 location used by the cluster.
 * @param {Cluster} cluster the unique ID of the CDK resource
 * @param {Construct} scope The local path of the yaml podTemplate files to upload
 * @param {string} eksAdminRoleArn The admin role of the EKS cluster
 */
function eksClusterSetup(cluster, scope, eksAdminRoleArn) {
    // Add the provided Amazon IAM Role as Amazon EKS Admin
    if (eksAdminRoleArn != undefined) {
        cluster.eksCluster.awsAuth.addMastersRole(aws_iam_1.Role.fromRoleArn(scope, 'AdminRole', eksAdminRoleArn), 'AdminRole');
    }
    const ebsCsiDriverIrsa = cluster.eksCluster.addServiceAccount('ebsCSIDriverRoleSA', {
        name: 'ebs-csi-controller-sa',
        namespace: 'kube-system',
    });
    const ebsCsiDriverPolicyDocument = aws_iam_1.PolicyDocument.fromJson(IamPolicyEbsCsiDriver);
    const ebsCsiDriverPolicy = new aws_iam_1.Policy(scope, 'IamPolicyEbsCsiDriverIAMPolicy', { document: ebsCsiDriverPolicyDocument });
    ebsCsiDriverPolicy.attachToRole(ebsCsiDriverIrsa.role);
    const ebsCSIDriver = new aws_eks_1.CfnAddon(scope, 'ebsCsiDriver', {
        addonName: 'aws-ebs-csi-driver',
        clusterName: cluster.eksCluster.clusterName,
        serviceAccountRoleArn: ebsCsiDriverIrsa.role.roleArn,
        addonVersion: 'v1.18.0-eksbuild.1',
        resolveConflicts: "OVERWRITE"
    });
    ebsCSIDriver.node.addDependency(ebsCsiDriverIrsa);
    // Deploy the Helm Chart for the Certificate Manager. Required for EMR Studio ALB.
    cluster.eksCluster.addHelmChart('CertManager', {
        createNamespace: true,
        namespace: 'cert-manager',
        chart: 'cert-manager',
        repository: 'https://charts.jetstack.io',
        version: '1.11.2',
        timeout: aws_cdk_lib_1.Duration.minutes(14),
        values: {
            startupapicheck: {
                timeout: '5m'
            },
            installCRDs: true
        }
    });
    // Add the kubernetes dashboard from helm chart
    cluster.eksCluster.addHelmChart('KubernetesDashboard', {
        createNamespace: true,
        namespace: 'kubernetes-dashboard',
        chart: 'kubernetes-dashboard',
        repository: 'https://kubernetes.github.io/dashboard/',
        version: 'v6.0.0',
        timeout: aws_cdk_lib_1.Duration.minutes(2),
        values: {
            fullnameOverride: 'kubernetes-dashboard',
            resources: {
                limits: {
                    memory: '600Mi',
                },
            },
        },
    });
    // Add the kubernetes dashboard service account
    cluster.eksCluster.addManifest('kubedashboard', {
        apiVersion: 'v1',
        kind: 'ServiceAccount',
        metadata: {
            name: 'eks-admin',
            namespace: 'kube-system',
        },
    });
    // Add the kubernetes dashboard cluster role binding
    cluster.eksCluster.addManifest('kubedashboardrolebinding', {
        apiVersion: 'rbac.authorization.k8s.io/v1',
        kind: 'ClusterRoleBinding',
        metadata: {
            name: 'eks-admin',
        },
        roleRef: {
            apiGroup: 'rbac.authorization.k8s.io',
            kind: 'ClusterRole',
            name: 'cluster-admin',
        },
        subjects: [
            {
                kind: 'ServiceAccount',
                name: 'eks-admin',
                namespace: 'kube-system',
            },
        ],
    });
    // Nodegroup capacity needed for all the tooling components including Karpenter
    let EmrEksNodeGroupTooling = { ...emr_eks_nodegroup_1.EmrEksNodegroup.TOOLING_ALL };
    EmrEksNodeGroupTooling.nodeRole = cluster.ec2InstanceNodeGroupRole;
    // Create the Amazon EKS Nodegroup for tooling
    cluster.addNodegroupCapacity('Tooling', EmrEksNodeGroupTooling);
    //IAM role created for the aws-node pod following AWS best practice not to use the EC2 instance role
    const awsNodeRole = new aws_iam_1.Role(scope, 'awsNodeRole', {
        assumedBy: new aws_iam_1.FederatedPrincipal(cluster.eksCluster.openIdConnectProvider.openIdConnectProviderArn, { ...[] }, 'sts:AssumeRoleWithWebIdentity'),
        roleName: `awsNodeRole-${cluster.clusterName}`,
        managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy')],
    });
    // update the aws-node service account with IAM role created for it
    new aws_eks_1.KubernetesManifest(scope, 'awsNodeServiceAccountUpdateManifest', {
        cluster: cluster.eksCluster,
        manifest: [
            {
                apiVersion: 'v1',
                kind: 'ServiceAccount',
                metadata: {
                    name: 'aws-node',
                    namespace: 'kube-system',
                    annotations: {
                        'eks.amazonaws.com/role-arn': awsNodeRole.roleArn,
                    },
                },
            }
        ],
        overwrite: true,
    });
    // Provide the Kubernetes Dashboard URL in AWS CloudFormation output
    new aws_cdk_lib_1.CfnOutput(scope, 'kubernetesDashboardURL', {
        description: 'Access Kubernetes Dashboard via kubectl proxy and this URL',
        value: 'http://localhost:8001/api/v1/namespaces/kubernetes-dashboard/services/https:kubernetes-dashboard:https/proxy/#/login',
    });
}
exports.eksClusterSetup = eksClusterSetup;
/**
 * @internal
 * Method to add the default EKS Managed Nodegroups configured for Spark workloads
 */
function setDefaultManagedNodeGroups(cluster) {
    let EmrEksNodeGroupCritical = { ...emr_eks_nodegroup_1.EmrEksNodegroup.CRITICAL_ALL };
    EmrEksNodeGroupCritical.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('criticalAll', EmrEksNodeGroupCritical);
    let EmrEksNodeGroupsharedDriver = { ...emr_eks_nodegroup_1.EmrEksNodegroup.SHARED_DRIVER };
    EmrEksNodeGroupsharedDriver.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('sharedDriver', EmrEksNodeGroupsharedDriver);
    let EmrEksNodeGroupsharedExecutor = { ...emr_eks_nodegroup_1.EmrEksNodegroup.SHARED_EXECUTOR };
    EmrEksNodeGroupsharedExecutor.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('sharedExecutor', EmrEksNodeGroupsharedExecutor);
    let EmrEksNodeGroupnotebookDriver = { ...emr_eks_nodegroup_1.EmrEksNodegroup.NOTEBOOK_DRIVER };
    EmrEksNodeGroupnotebookDriver.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('notebookDriver', EmrEksNodeGroupnotebookDriver);
    let EmrEksNodeGroupnotebookExecutor = { ...emr_eks_nodegroup_1.EmrEksNodegroup.NOTEBOOK_EXECUTOR };
    EmrEksNodeGroupnotebookExecutor.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('notebookExecutor', EmrEksNodeGroupnotebookExecutor);
}
exports.setDefaultManagedNodeGroups = setDefaultManagedNodeGroups;
/**
 * @internal
 * Method to add the default Karpenter provisioners for Spark workloads
 */
function setDefaultKarpenterProvisioners(cluster) {
    const subnets = cluster.eksCluster.vpc.selectSubnets({
        onePerAz: true,
        subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
    }).subnets;
    subnets.forEach((subnet, index) => {
        let criticalManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/critical-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterCriticalManifest-${index}`, criticalManfifestYAML);
        let sharedDriverManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/shared-driver-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterSharedDriverManifest-${index}`, sharedDriverManfifestYAML);
        let sharedExecutorManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/shared-executor-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterSharedExecutorManifest-${index}`, sharedExecutorManfifestYAML);
        let notebookDriverManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/notebook-driver-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterNotebookDriverManifest-${index}`, notebookDriverManfifestYAML);
        let notebookExecutorManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/notebook-executor-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterNotebookExecutorManifest-${index}`, notebookExecutorManfifestYAML);
    });
}
exports.setDefaultKarpenterProvisioners = setDefaultKarpenterProvisioners;
/**
 * @internal
 * Method to generate the Karpenter manifests from templates and targeted to the specific EKS cluster
 */
function karpenterManifestSetup(clusterName, path, subnet) {
    let manifest = utils_1.Utils.readYamlDocument(path);
    manifest = manifest.replace('{{subnet-id}}', subnet.subnetId);
    manifest = manifest.replace(/(\{{az}})/g, subnet.availabilityZone);
    manifest = manifest.replace('{{cluster-name}}', clusterName);
    let manfifestYAML = manifest.split("---").map((e) => utils_1.Utils.loadYaml(e));
    return manfifestYAML;
}
exports.karpenterManifestSetup = karpenterManifestSetup;
/**
 * @internal
 * Install all the required configurations of Karpenter SQS and Event rules to handle spot and unhealthy instance termination
 * Create a security group to be used by nodes created with karpenter
 * Tags the subnets and VPC to be used by karpenter
 * create a tooling provisioner that will deploy in each of the AZs, one per AZ
 */
function karpenterSetup(cluster, eksClusterName, scope, karpenterVersion) {
    const karpenterInterruptionQueue = new aws_sqs_1.Queue(scope, 'karpenterInterruptionQueue', {
        queueName: eksClusterName,
        retentionPeriod: aws_cdk_lib_1.Duration.seconds(300)
    });
    karpenterInterruptionQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: ['sqs:SendMessage'],
        principals: [new aws_iam_1.ServicePrincipal('sqs.amazonaws.com'), new aws_iam_1.ServicePrincipal('events.amazonaws.com')]
    }));
    new aws_events_1.Rule(scope, 'scheduledChangeRule', {
        eventPattern: {
            source: ['aws.heatlh'],
            detail: ['AWS Health Event']
        },
        targets: [new aws_events_targets_1.SqsQueue(karpenterInterruptionQueue)]
    });
    new aws_events_1.Rule(scope, 'instanceStateChangeRule', {
        eventPattern: {
            source: ['aws.ec2'],
            detail: ['EC2 Instance State-change Notification']
        },
        targets: [new aws_events_targets_1.SqsQueue(karpenterInterruptionQueue)]
    });
    const karpenterNodeRole = new aws_iam_1.Role(cluster, 'karpenter-node-role', {
        assumedBy: new aws_iam_1.ServicePrincipal(`ec2.${cluster.stack.urlSuffix}`),
        managedPolicies: [
            aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'),
            aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'),
            aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'),
            aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
        ],
        roleName: `KarpenterNodeRole-${eksClusterName}`,
    });
    const karpenterControllerPolicyStatementSSM = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: ['ssm:GetParameter', 'pricing:GetProducts'],
        resources: ['*'],
    });
    const karpenterControllerPolicyStatementEC2 = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: [
            'ec2:CreateLaunchTemplate',
            'ec2:DeleteLaunchTemplate',
            'ec2:CreateFleet',
            'ec2:RunInstances',
            'ec2:CreateTags',
            'ec2:TerminateInstances',
            'ec2:DescribeLaunchTemplates',
            'ec2:DescribeInstances',
            'ec2:DescribeSecurityGroups',
            'ec2:DescribeSubnets',
            'ec2:DescribeInstanceTypes',
            'ec2:DescribeInstanceTypeOfferings',
            'ec2:DescribeAvailabilityZones',
        ],
        resources: ['*'],
    });
    const karpenterControllerPolicyStatementIAM = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: ['iam:PassRole'],
        resources: [`arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/KarpenterNodeRole-${eksClusterName}`],
    });
    const karpenterInstanceProfile = new aws_iam_1.CfnInstanceProfile(cluster, 'karpenter-instance-profile', {
        roles: [karpenterNodeRole.roleName],
        instanceProfileName: `karpenterNodeInstanceProfile-${eksClusterName}`,
        path: '/'
    });
    cluster.awsAuth.addRoleMapping(karpenterNodeRole, {
        username: 'system:node:{{EC2PrivateDNSName}}',
        groups: ['system:bootstrappers', 'system:nodes'],
    });
    const karpenterNS = cluster.addManifest('karpenterNS', {
        apiVersion: 'v1',
        kind: 'Namespace',
        metadata: { name: 'karpenter' },
    });
    const karpenterAccount = cluster.addServiceAccount('Karpenter', {
        name: 'karpenter',
        namespace: 'karpenter',
    });
    karpenterAccount.node.addDependency(karpenterNS);
    karpenterAccount.addToPrincipalPolicy(karpenterControllerPolicyStatementSSM);
    karpenterAccount.addToPrincipalPolicy(karpenterControllerPolicyStatementEC2);
    karpenterAccount.addToPrincipalPolicy(karpenterControllerPolicyStatementIAM);
    //Deploy Karpenter Chart
    const karpenterChart = cluster.addHelmChart('Karpenter', {
        chart: 'karpenter',
        release: 'karpenter',
        repository: 'oci://public.ecr.aws/karpenter/karpenter',
        namespace: 'karpenter',
        version: karpenterVersion || emr_eks_cluster_1.EmrEksCluster.DEFAULT_KARPENTER_VERSION,
        timeout: aws_cdk_lib_1.Duration.minutes(14),
        wait: true,
        values: {
            serviceAccount: {
                name: 'karpenter',
                create: false,
                annotations: {
                    'eks.amazonaws.com/role-arn': karpenterAccount.role.roleArn,
                },
            },
            settings: {
                aws: {
                    defaultInstanceProfile: karpenterInstanceProfile.instanceProfileName,
                    clusterName: eksClusterName,
                    clusterEndpoint: cluster.clusterEndpoint,
                    interruptionQueueName: karpenterInterruptionQueue.queueName
                },
            }
        },
    });
    karpenterChart.node.addDependency(karpenterAccount);
    const karpenterInstancesSg = new aws_ec2_1.SecurityGroup(scope, 'karpenterSg', {
        vpc: cluster.vpc,
        allowAllOutbound: true,
        description: 'security group for a karpenter instances',
        securityGroupName: 'karpenterSg',
        disableInlineRules: true,
    });
    aws_cdk_lib_1.Tags.of(karpenterInstancesSg).add('karpenter.sh/discovery', `${eksClusterName}`);
    cluster.clusterSecurityGroup.addIngressRule(karpenterInstancesSg, aws_ec2_1.Port.allTraffic());
    karpenterInstancesSg.addIngressRule(karpenterInstancesSg, aws_ec2_1.Port.allTraffic());
    karpenterInstancesSg.addIngressRule(cluster.clusterSecurityGroup, aws_ec2_1.Port.allTraffic());
    aws_cdk_lib_1.Tags.of(cluster.vpc).add('karpenter.sh/discovery', eksClusterName);
    cluster.vpc.privateSubnets.forEach((subnet) => {
        aws_cdk_lib_1.Tags.of(subnet).add('karpenter.sh/discovery', eksClusterName);
    });
    cluster.vpc.publicSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('karpenter.sh/discovery', eksClusterName));
    const privateSubnets = cluster.vpc.selectSubnets({
        onePerAz: true,
        subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
    }).subnets;
    let listPrivateSubnets = privateSubnets.map(subnet => subnet.subnetId);
    let manifest = utils_1.Utils.readYamlDocument(`${__dirname}/resources/k8s/karpenter-provisioner-config/tooling-provisioner.yml`);
    manifest = manifest.replace(/(\{{cluster-name}})/g, eksClusterName);
    manifest = manifest.replace(/(\{{subnet-list}})/g, listPrivateSubnets.join(','));
    let manfifestYAML = manifest.split("---").map((e) => utils_1.Utils.loadYaml(e));
    const manifestApply = cluster.addManifest('provisioner-tooling', ...manfifestYAML);
    manifestApply.node.addDependency(karpenterChart);
    return karpenterChart;
}
exports.karpenterSetup = karpenterSetup;
/**
 * @internal
 * Deploy the cluster autoscaler controller in the k8s cluster
 */
function clusterAutoscalerSetup(cluster, eksClusterName, scope, k8sVersion) {
    //Version of the autoscaler, controls the image tag
    const versionMap = new Map([
        [aws_eks_1.KubernetesVersion.V1_25, "9.25.0"],
        [aws_eks_1.KubernetesVersion.V1_24, "9.25.0"],
        [aws_eks_1.KubernetesVersion.V1_23, "9.21.0"],
        [aws_eks_1.KubernetesVersion.V1_22, "9.13.1"]
    ]);
    // Create a Kubernetes Service Account for the Cluster Autoscaler with Amazon IAM Role
    const AutoscalerServiceAccount = cluster.addServiceAccount('Autoscaler', {
        name: 'cluster-autoscaler',
        namespace: 'kube-system',
    });
    //Iam policy attached to the Role used by k8s autoscaller
    let autoscalingPolicyDescribe = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: [
            'autoscaling:DescribeAutoScalingGroups',
            'autoscaling:DescribeAutoScalingInstances',
            'autoscaling:DescribeLaunchConfigurations',
            'autoscaling:DescribeTags',
            'ec2:DescribeLaunchTemplateVersions',
            'eks:DescribeNodegroup',
        ],
        resources: ['*'],
    });
    let autoscalingPolicyMutateGroup = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: [
            'autoscaling:SetDesiredCapacity',
            'autoscaling:TerminateInstanceInAutoScalingGroup',
        ],
        resources: ['*'],
        conditions: {
            StringEquals: {
                'aws:ResourceTag/eks:cluster-name': eksClusterName,
            },
        },
    });
    // Add the right Amazon IAM Policy to the Amazon IAM Role for the Cluster Autoscaler
    AutoscalerServiceAccount.addToPrincipalPolicy(autoscalingPolicyDescribe);
    AutoscalerServiceAccount.addToPrincipalPolicy(autoscalingPolicyMutateGroup);
    cluster.addHelmChart('AutoScaler', {
        chart: 'cluster-autoscaler',
        repository: 'https://kubernetes.github.io/autoscaler',
        version: versionMap.get(k8sVersion),
        namespace: 'kube-system',
        timeout: aws_cdk_lib_1.Duration.minutes(14),
        values: {
            cloudProvider: 'aws',
            awsRegion: aws_cdk_lib_1.Stack.of(scope).region,
            autoDiscovery: { clusterName: eksClusterName },
            rbac: {
                serviceAccount: {
                    name: 'cluster-autoscaler',
                    create: false,
                },
            },
            extraArgs: {
                'skip-nodes-with-local-storage': false,
                'scan-interval': '5s',
                'expander': 'least-waste',
                'balance-similar-node-groups': true,
                'skip-nodes-with-system-pods': false,
            },
        },
    });
}
exports.clusterAutoscalerSetup = clusterAutoscalerSetup;
//# sourceMappingURL=data:application/json;base64,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