"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataDomainNracWorkflow = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
/**
 * This CDK Construct creates a NRAC workflow for Producer/Consumer account.
 * It is based on an AWS Step Functions state machine. It has the following steps:
 * * checks for AWS RAM invitations
 * * accepts RAM invitations if the source is Central Gov. account
 * * creates AWS Glue Data Catalog Database and tables
 * * creates Resource-Link(s) for created tables
 *
 * This Step Functions state machine is invoked from the Central Gov. account via EventBridge Event Bus.
 * It is initiatated in {@link DataDomain}, but can be used as a standalone construct.
 *
 * Usage example:
 * ```typescript
 * import { App, Stack } from 'aws-cdk-lib';
 * import { Role } from 'aws-cdk-lib/aws-iam';
 * import { DataDomain } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new App();
 * const stack = new Stack(exampleApp, 'DataProductStack');
 *
 * const workflowRole = new Role(stack, 'myWorkflowRole', {
 *  assumedBy: ...
 * });
 *
 * new DataDomainNracWorkflow(this, 'DataDomainWorkflow', {
 *  eventBus: eventBus,
 *  workflowRole: workflowRole,
 *  centralAccountId: '1234567891011',
 *  domainName: 'domainName'
 * });
 * ```
 *
 */
class DataDomainNracWorkflow extends constructs_1.Construct {
    /**
     * Construct a new instance of DataDomainNracWorkflow.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataDomainWorkflow} props the DataDomainWorkflowProps properties
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Task to check for existing RAM invitations
        const getRamInvitations = new aws_stepfunctions_tasks_1.CallAwsService(this, 'GetResourceShareInvitations', {
            service: 'ram',
            action: 'getResourceShareInvitations',
            iamResources: ['*'],
            parameters: {},
            resultPath: '$.taskresult',
        });
        // Task to accept RAM share invitation
        const acceptRamShare = new aws_stepfunctions_tasks_1.CallAwsService(this, 'AcceptResourceShareInvitation', {
            service: 'ram',
            action: 'acceptResourceShareInvitation',
            iamResources: ['*'],
            parameters: {
                'ResourceShareInvitationArn.$': '$.ram_share.ResourceShareInvitationArn',
            },
            resultPath: '$.Response',
            resultSelector: {
                'Status.$': '$.ResourceShareInvitation.Status',
            },
        });
        const createLocalDatabase = new aws_stepfunctions_tasks_1.CallAwsService(this, 'createLocalDatabase', {
            service: 'glue',
            action: 'createDatabase',
            iamResources: ['*'],
            parameters: {
                'DatabaseInput': {
                    'Name.$': '$.detail.database_name'
                },
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        const grantCreateTable = new aws_stepfunctions_tasks_1.CallAwsService(this, 'grantCreateTable', {
            service: 'lakeformation',
            action: 'grantPermissions',
            iamResources: ['*'],
            parameters: {
                'Permissions': [
                    'ALL'
                ],
                'Principal': {
                    'DataLakePrincipalIdentifier': props.workflowRole.roleArn
                },
                'Resource': {
                    'Database': {
                        'Name.$': '$.detail.database_name'
                    },
                }
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD
        });
        // Task to create a resource-link for shared table from central gov accunt
        const createResourceLink = new aws_stepfunctions_tasks_1.CallAwsService(this, 'createResourceLink', {
            service: 'glue',
            action: 'createTable',
            iamResources: ['*'],
            parameters: {
                'DatabaseName.$': '$.database_name',
                'TableInput': {
                    'Name.$': "States.Format('rl-{}', $.table_name)",
                    'TargetTable': {
                        'CatalogId': props.centralAccountId,
                        'DatabaseName.$': '$.central_database_name',
                        'Name.$': '$.table_name',
                    },
                },
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        // Trigger crawler workflow
        const triggerCrawler = new aws_stepfunctions_tasks_1.EventBridgePutEvents(this, 'triggerCrawler', {
            entries: [{
                    detail: aws_stepfunctions_1.TaskInput.fromObject({
                        'database_name': aws_stepfunctions_1.JsonPath.stringAt("$.database_name"),
                        'table_names': aws_stepfunctions_1.JsonPath.stringAt("$.table_names"),
                        'central_account_id': aws_stepfunctions_1.JsonPath.stringAt("$.central_account_id"),
                        'central_database_name': aws_stepfunctions_1.JsonPath.stringAt("$.central_database_name"),
                        'lf_access_mode': aws_stepfunctions_1.JsonPath.stringAt("$.lf_access_mode"),
                    }),
                    detailType: 'triggerCrawler',
                    eventBus: props.eventBus,
                    source: 'com.central.stepfunction',
                }]
        });
        // Pass task to finish the workflow
        const finishWorkflow = new aws_stepfunctions_1.Pass(this, 'finishWorkflow');
        const rlMapTask = new aws_stepfunctions_1.Map(this, 'forEachTable', {
            itemsPath: '$.table_names',
            inputPath: '$.[0]',
            parameters: {
                'central_database_name.$': '$.central_database_name',
                'database_name.$': '$.database_name',
                'table_name.$': '$$.Map.Item.Value'
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        rlMapTask.iterator(createResourceLink);
        rlMapTask.next(new aws_stepfunctions_1.Choice(this, 'thisAccountIsProducer', { inputPath: '$.[0]' })
            .when(aws_stepfunctions_1.Condition.stringEquals('$.producer_acc_id', aws_cdk_lib_1.Aws.ACCOUNT_ID), triggerCrawler)
            .otherwise(finishWorkflow));
        // Task to iterate over RAM shares and check if there are PENDING invites from the central account
        const ramMapTask = new aws_stepfunctions_1.Map(this, 'forEachRamInvitation', {
            itemsPath: '$.taskresult.ResourceShareInvitations',
            parameters: {
                'ram_share.$': '$$.Map.Item.Value',
                'central_account_id.$': '$.account',
                'central_database_name.$': '$.detail.central_database_name',
                'database_name.$': '$.detail.database_name',
                'table_names.$': '$.detail.table_names',
                'producer_acc_id.$': '$.detail.producer_acc_id',
                'lf_access_mode.$': '$.detail.lf_access_mode',
            },
            resultPath: '$.map_result',
            outputPath: '$.map_result.[?(@.central_account_id)]',
        });
        const invitationAccepted = new aws_stepfunctions_1.Pass(this, 'invitationAccepted', {
            result: aws_stepfunctions_1.Result.fromObject({ 'Status': 'ACCEPTED' }),
            resultPath: '$.Response'
        });
        ramMapTask.iterator(new aws_stepfunctions_1.Choice(this, 'isInvitationPending')
            .when(aws_stepfunctions_1.Condition.and(aws_stepfunctions_1.Condition.stringEqualsJsonPath('$.ram_share.SenderAccountId', '$.central_account_id'), aws_stepfunctions_1.Condition.stringEquals('$.ram_share.Status', 'PENDING')), acceptRamShare)
            .when(aws_stepfunctions_1.Condition.and(aws_stepfunctions_1.Condition.stringEqualsJsonPath('$.ram_share.SenderAccountId', '$.central_account_id'), aws_stepfunctions_1.Condition.stringEquals('$.ram_share.Status', 'ACCEPTED')), invitationAccepted)
            .otherwise(new aws_stepfunctions_1.Pass(this, 'notValidPass', {
            result: aws_stepfunctions_1.Result.fromObject({})
        })));
        ramMapTask.next(rlMapTask);
        // Avoid possible delays in between RAM share time and EventBridge event time 
        const initWait = new aws_stepfunctions_1.Wait(this, 'InitWait', {
            time: aws_stepfunctions_1.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(5))
        });
        createLocalDatabase.addCatch(grantCreateTable, {
            errors: ['Glue.AlreadyExistsException'],
            resultPath: '$.Exception',
        }).next(grantCreateTable).next(ramMapTask);
        // Create Log group for this state machine
        const logGroup = new aws_logs_1.LogGroup(this, 'Workflow', {
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
            logGroupName: '/aws/vendedlogs/data-mesh/nrac-workflow',
        });
        logGroup.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        // State Machine workflow to accept RAM share and create resource-link for a shared table
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'nracStateMachine', {
            definition: initWait.next(getRamInvitations).next(new aws_stepfunctions_1.Choice(this, 'resourceShareInvitationsEmpty')
                .when(aws_stepfunctions_1.Condition.isPresent('$.taskresult.ResourceShareInvitations[0]'), createLocalDatabase)
                .otherwise(finishWorkflow)),
            role: props.workflowRole,
            logs: {
                destination: logGroup,
                level: aws_stepfunctions_1.LogLevel.ALL,
            },
        });
    }
}
exports.DataDomainNracWorkflow = DataDomainNracWorkflow;
//# sourceMappingURL=data:application/json;base64,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