"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PreparedDataset = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * PreparedDataset is used by the [BatchReplayer]{@link BatchReplayer} to generate data in different targets.
 *
 * One of the startDatetime or offset parameter needs to be passed to the constructor:
 *  * StartDatetime is used for prepared datasets provided by the Analytics Reference Architecture because they are known during synthetize time.
 *  * Offset is used when a PreparedDataset is created from a CustomDataset because the startDatetime is not known during synthetize time.
 *
 * A PreparedDataset has following properties:
 *
 * 1. Data is partitioned by timestamp (a range in seconds). Each folder stores data within a given range.
 * There is no constraint on how long the timestamp range can be. But each file must not be larger than 100MB.
 * Creating new PreparedDataset requires to find the right balance between number of partitions and the amount of data read by each BatchReplayer (micro-)batch
 * The available PreparedDatasets have a timestamp range that fit the total dataset time range (see each dataset documentation below) to avoid having too many partitions.
 *
 * Here is an example:
 *
 * |- time_range_start=16000000000
 *
 *    |- file1.csv 100MB
 *
 *    |- file2.csv 50MB
 *
 * |- time_range_start=16000000300 // 5 minute range (300 sec)
 *
 *    |- file1.csv 1MB
 *
 * |- time_range_start=16000000600
 *
 *    |- file1.csv 100MB
 *
 *    |- file2.csv 100MB
 *
 *    |- whichever-file-name-is-fine-as-we-have-manifest-files.csv 50MB
 *
 * 2. It has a manifest CSV file with two columns: start and path. Start is the timestamp
 *
 * start        , path
 *
 * 16000000000  , s3://<path>/<to>/<folder>/time_range_start=16000000000/file1.csv
 *
 * 16000000000  , s3://<path>/<to>/<folder>/time_range_start=16000000000/file2.csv
 *
 * 16000000300  , s3://<path>/<to>/<folder>/time_range_start=16000000300/file1.csv
 *
 * 16000000600  , s3://<path>/<to>/<folder>/time_range_start=16000000600/file1.csv
 *
 * 16000000600  , s3://<path>/<to>/<folder>/time_range_start=16000000600/file2.csv
 *
 * 16000000600  , s3://<path>/<to>/<folder>/time_range_start=16000000600/whichever-file....csv
 *
 * If the stack is deployed in another region than eu-west-1, data transfer costs will apply.
 * The pre-defined PreparedDataset access is recharged to the consumer via Amazon S3 Requester Pay feature.
 */
class PreparedDataset {
    /**
     * Calculate the offset in Seconds from the startDatetime value and the current datetime
     * @param {string} startDatetime the min datetime value used as reference for offset
     * @access private
     */
    static getOffset(startDatetime) {
        const now = new Date().getTime();
        const minDatetime = new Date(startDatetime).getTime();
        return Math.floor((now - minDatetime) / 1000).toString();
    }
    /**
     * Constructs a new instance of the Dataset class
     * @param {DatasetProps} props the DatasetProps
     * @access public
     */
    constructor(props) {
        if (props.startDatetime === undefined && props.offset === undefined) {
            throw new Error('[PreparedDataset] One of startDatetime or offset parameter must be passed');
        }
        if (props.startDatetime !== undefined && props.offset !== undefined) {
            throw new Error('[PreparedDataset] Only one of startDatetime or offset parameter must be passed');
        }
        this.startDateTime = props.startDatetime;
        this.offset = props.startDatetime ? PreparedDataset.getOffset(props.startDatetime) : props.offset;
        this.location = props.location;
        this.tableName = this.sqlTable();
        this.manifestLocation = props.manifestLocation;
        this.dateTimeColumnToFilter = props.dateTimeColumnToFilter;
        this.dateTimeColumnsToAdjust = props.dateTimeColumnsToAdjust;
    }
    /**
     * Extract the last part of the object key (with / delimiter) and replace '-' with '_' for SQL compatibility
     * @access private
     */
    sqlTable() {
        const parsedPrefix = this.location.objectKey.split('/');
        const re = /\-/gi;
        return parsedPrefix[parsedPrefix.length - 1].replace(re, '_');
    }
}
_a = JSII_RTTI_SYMBOL_1;
PreparedDataset[_a] = { fqn: "aws-analytics-reference-architecture.PreparedDataset", version: "2.12.9" };
/**
 * The bucket name of the AWS Analytics Reference Architecture datasets.
 * Data transfer costs will aply if the stack is deployed in another region than eu-west-1.
 * The pre-defined PreparedDataset access is recharged to the consumer via Amazon S3 Requester Pay feature.
 */
PreparedDataset.DATASETS_BUCKET = 'aws-analytics-reference-architecture';
/**
 * The web sale dataset part of 1GB retail datasets.
 * The time range is one week from min(sale_datetime) to max(sale_datetime)
 *
 * | Column name           | Column type | Example                  |
 * |-----------------------|-------------|--------------------------|
 * | item_id               | bigint      | 3935                     |
 * | order_id              | bigint      | 81837                    |
 * | quantity              | bigint      | 65                       |
 * | wholesale_cost        | double      | 32.98                    |
 * | list_price            | double      | 47.82                    |
 * | sales_price           | double      | 36.34                    |
 * | ext_discount_amt      | double      | 2828.8                   |
 * | ext_sales_price       | double      | 2362.1                   |
 * | ext_wholesale_cost    | double      | 2143.7                   |
 * | ext_list_price        | double      | 3108.3                   |
 * | ext_tax               | double      | 0.0                      |
 * | coupon_amt            | double      | 209.62                   |
 * | ext_ship_cost         | double      | 372.45                   |
 * | net_paid              | double      | 2152.48                  |
 * | net_paid_inc_tax      | double      | 2152.48                  |
 * | net_paid_inc_ship     | double      | 442.33                   |
 * | net_paid_inc_ship_tax | double      | 442.33                   |
 * | net_profit            | double      | 8.78                     |
 * | bill_customer_id      | string      | AAAAAAAALNLFAAAA         |
 * | ship_customer_id      | string      | AAAAAAAALPPJAAAA         |
 * | warehouse_id          | string      | AAAAAAAABAAAAAAA         |
 * | promo_id              | string      | AAAAAAAAPCAAAAAA         |
 * | ship_delay            | string      | OVERNIGHT                |
 * | ship_mode             | string      | SEA                      |
 * | ship_carrier          | string      | GREAT EASTERN            |
 * | sale_datetime         | string      | 2021-01-06T15:00:19.373Z |
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 */
PreparedDataset.RETAIL_1_GB_WEB_SALE = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/web-sale',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/web-sale-manifest.csv',
    },
    dateTimeColumnToFilter: 'sale_datetime',
    dateTimeColumnsToAdjust: ['sale_datetime'],
});
/**
 * The store sale dataset part of 1GB retail datasets.
 * The time range is one week from min(sale_datetime) to max(sale_datetime)
 *
 * | Column name        | Column type | Example                  |
 * |--------------------|-------------|--------------------------|
 * | item_id            | bigint      | 3935                     |
 * | ticket_id          | bigint      | 81837                    |
 * | quantity           | bigint      | 96                       |
 * | wholesale_cost     | double      | 21.15                    |
 * | list_price         | double      | 21.78                    |
 * | sales_price        | double      | 21.18                    |
 * | ext_discount_amt   | double      | 0.0                      |
 * | ext_sales_price    | double      | 2033.28                  |
 * | ext_wholesale_cost | double      | 2030.4                   |
 * | ext_list_price     | double      | 2090.88                  |
 * | ext_tax            | double      | 81.1                     |
 * | coupon_amt         | double      | 0.0                      |
 * | net_paid           | double      | 2033.28                  |
 * | net_paid_inc_tax   | double      | 2114.38                  |
 * | net_profit         | double      | 2.88                     |
 * | customer_id        | string      | AAAAAAAAEOIDAAAA         |
 * | store_id           | string      | AAAAAAAABAAAAAAA         |
 * | promo_id           | string      | AAAAAAAAEEAAAAAA         |
 * | sale_datetime      | string      | 2021-01-04T22:20:04.144Z |
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 */
PreparedDataset.RETAIL_1_GB_STORE_SALE = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/store-sale',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/store-sale-manifest.csv',
    },
    dateTimeColumnToFilter: 'sale_datetime',
    dateTimeColumnsToAdjust: ['sale_datetime'],
});
/**
 * The customer dataset part of 1GB retail datasets.
 * The time range is one week from min(customer_datetime) to max(customer_datetime)
 *
 * | Column name       	| Column type 	| Example                    	|
 * |-------------------	|-------------	|----------------------------	|
 * | customer_id       	| string      	| AAAAAAAAHCLFOHAA           	|
 * | salutation        	| string      	| Miss                       	|
 * | first_name        	| string      	| Tina                       	|
 * | last_name         	| string      	| Frias                      	|
 * | birth_country     	| string      	| GEORGIA                    	|
 * | email_address     	| string      	| Tina.Frias@jdK4TZ1qJXB.org 	|
 * | birth_date        	| string      	| 1924-06-14                 	|
 * | gender            	| string      	| F                          	|
 * | marital_status    	| string      	| D                          	|
 * | education_status  	| string      	| 2 yr Degree                	|
 * | purchase_estimate 	| bigint      	| 2500                       	|
 * | credit_rating     	| string      	| Low Risk                   	|
 * | buy_potential     	| string      	| 1001-5000                  	|
 * | vehicle_count     	| bigint      	| 1                          	|
 * | lower_bound       	| bigint      	| 170001                     	|
 * | upper_bound       	| bigint      	| 180000                     	|
 * | address_id        	| string      	| AAAAAAAALAFINEAA           	|
 * | customer_datetime 	| string      	| 2021-01-19T08:07:47.140Z   	|
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 */
PreparedDataset.RETAIL_1_GB_CUSTOMER = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/customer',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/customer-manifest.csv',
    },
    dateTimeColumnToFilter: 'customer_datetime',
    dateTimeColumnsToAdjust: ['customer_datetime'],
});
/**
 * The customer address dataset part of 1GB retail datasets.
 * It can be joined with customer dataset on address_id column.
 * The time range is one week from min(address_datetime) to max(address_datetime)
 *
 * | Column name      | Column type | Example                  |
 * |------------------|-------------|--------------------------|
 * | address_id       | string      | AAAAAAAAINDKAAAA         |
 * | city             | string      | Farmington               |
 * | county           | string      | Greeley County           |
 * | state            | string      | KS                       |
 * | zip              | bigint      | 69145                    |
 * | country          | string      | United States            |
 * | gmt_offset       | double      | -6.0                     |
 * | location_type    | string      | apartment                |
 * | street           | string      | 390 Pine South Boulevard |
 * | address_datetime | string      | 2021-01-03T02:25:52.826Z |
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 *
 */
PreparedDataset.RETAIL_1_GB_CUSTOMER_ADDRESS = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/customer-address',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/customer-address-manifest.csv',
    },
    dateTimeColumnToFilter: 'address_datetime',
    dateTimeColumnsToAdjust: ['address_datetime'],
});
/**
 * The item dataset part of 1GB retail datasets
 * The time range is one week from min(item_datetime) to max(item_datetime)
 *
 * | Column name   | Column type | Example                                        |
 * |---------------|-------------|------------------------------------------------|
 * |       item_id |      bigint |                                          15018 |
 * |     item_desc |      string | Even ready materials tell with a ministers; un |
 * |         brand |      string |                                 scholarmaxi #9 |
 * |         class |      string |                                        fishing |
 * |      category |      string |                                         Sports |
 * |      manufact |      string |                                    eseoughtpri |
 * |          size |      string |                                            N/A |
 * |         color |      string |                                        thistle |
 * |         units |      string |                                         Bundle |
 * |     container |      string |                                        Unknown |
 * |  product_name |      string |                          eingoughtbarantiought |
 * | item_datetime |      string |                       2021-01-01T18:17:56.718Z |
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 */
PreparedDataset.RETAIL_1_GB_ITEM = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/item',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/item-manifest.csv',
    },
    dateTimeColumnToFilter: 'item_datetime',
    dateTimeColumnsToAdjust: ['item_datetime'],
});
/**
 * The promo dataset part of 1GB retail datasets
 * The time range is one week from min(promo_datetime) to max(promo_datetime)
 *
 * | Column name     | Column type | Example                  |
 * |-----------------|-------------|--------------------------|
 * |        promo_id |      string |         AAAAAAAAHIAAAAAA |
 * |            cost |      double |                   1000.0 |
 * | response_target |      bigint |                        1 |
 * |      promo_name |      string |                     anti |
 * |         purpose |      string |                  Unknown |
 * |  start_datetime |      string | 2021-01-01 00:00:35.890Z |
 * |    end_datetime |      string | 2021-01-02 13:16:09.785Z |
 * |  promo_datetime |      string | 2021-01-01 00:00:16.104Z |
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 */
PreparedDataset.RETAIL_1_GB_PROMO = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/promo',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/promo-manifest.csv',
    },
    dateTimeColumnToFilter: 'promo_datetime',
    dateTimeColumnsToAdjust: ['promo_datetime'],
});
/**
 * The store dataset part of 1GB retail datasets
 * The time range is one week from min(store_datetime) to max(store_datetime)
 *
 * | Column name      | Column type | Example                  |
 * |------------------|-------------|--------------------------|
 * |         store_id |      string |         AAAAAAAAKAAAAAAA |
 * |       store_name |      string |                      bar |
 * | number_employees |      bigint |                      219 |
 * |      floor_space |      bigint |                  6505323 |
 * |            hours |      string |                 8AM-12AM |
 * |          manager |      string |             David Trahan |
 * |        market_id |      bigint |                       10 |
 * |   market_manager |      string |      Christopher Maxwell |
 * |             city |      string |                   Midway |
 * |           county |      string |        Williamson County |
 * |            state |      string |                       TN |
 * |              zip |      bigint |                    31904 |
 * |          country |      string |            United States |
 * |       gmt_offset |      double |                     -5.0 |
 * |   tax_percentage |      double |                      0.0 |
 * |           street |      string |            71 Cedar Blvd |
 * |   store_datetime |      string | 2021-01-01T00:00:00.017Z |
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 */
PreparedDataset.RETAIL_1_GB_STORE = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/store',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/store-manifest.csv',
    },
    dateTimeColumnToFilter: 'store_datetime',
    dateTimeColumnsToAdjust: ['store_datetime'],
});
/**
 * The store dataset part of 1GB retail datasets
 * The time range is one week from min(warehouse_datetime) to max(warehouse_datetime)
 *
 * | Column name        | Column type | Example                  |
 * |--------------------|-------------|--------------------------|
 * |       warehouse_id |      string |         AAAAAAAAEAAAAAAA |
 * |     warehouse_name |      string |               Operations |
 * |             street |      string |    461 Second Johnson Wy |
 * |               city |      string |                 Fairview |
 * |                zip |      bigint |                    35709 |
 * |             county |      string |        Williamson County |
 * |              state |      string |                       TN |
 * |            country |      string |            United States |
 * |         gmt_offset |      double |                     -5.0 |
 * | warehouse_datetime |      string | 2021-01-01T00:00:00.123Z |
 *
 */
PreparedDataset.RETAIL_1_GB_WAREHOUSE = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/warehouse',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/warehouse-manifest.csv',
    },
    dateTimeColumnToFilter: 'warehouse_datetime',
    dateTimeColumnsToAdjust: ['warehouse_datetime'],
});
exports.PreparedDataset = PreparedDataset;
//# sourceMappingURL=data:application/json;base64,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