import { Duration } from 'aws-cdk-lib';
import { IChainable } from 'aws-cdk-lib/aws-stepfunctions';
import { Construct } from 'constructs';
import { PreparedDataset } from './prepared-dataset';
import { ISecurityGroup, IVpc } from 'aws-cdk-lib/aws-ec2';
import { S3Sink, DbSink, DynamoDbSink } from './batch-replayer-helpers';
/**
 * The properties for the BatchReplayer construct
 */
export interface BatchReplayerProps {
    /**
     * The [PreparedDataset]{@link PreparedDataset} used to replay data
     */
    readonly dataset: PreparedDataset;
    /**
     * The frequency of the replay
     * @default - The BatchReplayer is triggered every 60 seconds
     */
    readonly frequency?: Duration;
    /**
     * Parameters to write to S3 target
     */
    readonly s3Props?: S3Sink;
    /**
     * Parameters to write to DynamoDB target
     */
    readonly ddbProps?: DynamoDbSink;
    /**
     * Parameters to write to Redshift target
     */
    readonly redshiftProps?: DbSink;
    /**
     * Parameters to write to Aurora target
     */
    readonly auroraProps?: DbSink;
    /**
     * Parameters to write to RDS target
     */
    readonly rdsProps?: DbSink;
    /**
     * Security group for the WriteInBatch Lambda function
     */
    readonly secGroup?: ISecurityGroup;
    /**
     * VPC for the WriteInBatch Lambda function
     */
    readonly vpc?: IVpc;
    /**
     * Additional StupFunction Tasks to run sequentially after the BatchReplayer finishes
     * @default - The BatchReplayer do not have additional Tasks
     *
     * The expected input for the first Task in this sequence is:
     *
     * input = [
     *  {
     *    "processedRecords": Int,
     *    "outputPaths": String [],
     *    "startTimeinIso": String,
     *    "endTimeinIso": String
     *  }
     * ]
     *
     * Each element in input represents the output of each lambda iterator that replays the data.
     *
     * param: processedRecods -> Number of records processed
     * param: ouputPaths -> List of files created in S3
     *  **  eg. "s3://<sinkBucket name>/<s3ObjectKeySink prefix, if any>/<dataset name>/ingestion_start=<timestamp>/ingestion_end=<timestamp>/<s3 filename>.csv",
  
     * param: startTimeinIso -> Start Timestamp on original dataset
     * param: endTimeinIso -> End Timestamp on original dataset
     *
     * *outputPaths* can be used to extract and aggregate new partitions on data and
     * trigger additional Tasks.
     */
    readonly additionalStepFunctionTasks?: IChainable[];
}
/**
 * Replay the data in the given PartitionedDataset.
 *
 * It will dump files into the target based on the given `frequency`.
 * The computation is in a Step Function with two Lambda steps.
 *
 * 1. resources/lambdas/find-file-paths
 * Read the manifest file and output a list of S3 file paths within that batch time range
 *
 * 2. resources/lambdas/write-in-batch
 * Take a file path, filter only records within given time range, adjust the time with offset to
 * make it looks like just being generated. Then write the output to the target
 *
 * Usage example:
 * ```typescript
 *
 * const myBucket = new Bucket(stack, "MyBucket")
 *
 * let myProps: S3Sink = {
 *  sinkBucket: myBucket,
 *  sinkObjectKey: 'some-prefix',
 *  outputFileMaxSizeInBytes: 10000000,
 * }
 *
 * new BatchReplayer(stack, "WebSalesReplayer", {
 *   dataset: PreparedDataset.RETAIL_1_GB_WEB_SALE,
 *   s3Props: myProps,
 *   frequency: 120,
 * });
 * ```
 *
 * :warning: **If the Bucket is encrypted with KMS, the Key must be managed by this stack.
 */
export declare class BatchReplayer extends Construct {
    /**
     * Dataset used for replay
     */
    readonly dataset: PreparedDataset;
    /**
     * Frequency (in Seconds) of the replaying. The batch job will start
     * for every given frequency and replay the data in that period
     */
    readonly frequency: number;
    /**
     * Parameters to write to S3 target
     */
    readonly s3Props?: S3Sink;
    /**
     * Parameters to write to DynamoDB target
     */
    readonly ddbProps?: DynamoDbSink;
    /**
     * Parameters to write to Redshift target
     */
    readonly redshiftProps?: DbSink;
    /**
     * Parameters to write to Aurora target
     */
    readonly auroraProps?: DbSink;
    /**
     * Parameters to write to RDS target
     */
    readonly rdsProps?: DbSink;
    /**
     * Security group for the WriteInBatch Lambda function
     */
    readonly secGroup?: ISecurityGroup;
    /**
     * VPC for the WriteInBatch Lambda function
     */
    readonly vpc?: IVpc;
    /**
     * Optional Sequence of additional Tasks to append at the end of the Step Function
     * that replays data that will execute after data has been replayed
     */
    private readonly additionalStepFunctionTasks?;
    /**
     * Constructs a new instance of the BatchReplayer construct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {BatchReplayerProps} props the BatchReplayer [properties]{@link BatchReplayerProps}
     */
    constructor(scope: Construct, id: string, props: BatchReplayerProps);
    private chainStepFunctionTasks;
}
