"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.PreBundledLayer = void 0;
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * A Layer with prebundled dependencies that can be reused accross multiple [PreBundledFunction]{@link PreBundledFunction} resources.
 * This layer reduces the total size of the Analytics Reference Architecture library by factorizing common dependencies in one shared layer.
 *
 * Do not include packages in your Lambda function `requirements.txt` if they are already part of the PreBundledLayer.
 *
 * Here is the list of bundled dependencies:
 * * boto3==1.26.37
 * * botocore==1.29.37
 * * jmespath==1.0.1
 * * python-dateutil==2.8.2
 * * s3transfer==0.6.0
 * * six==1.16.0
 * * urllib3==1.26.13
 */
class PreBundledLayer extends aws_lambda_1.LayerVersion {
    /**
     * Get an existing PreBundledLayer if it already exists in the CDK scope or create a new one
     * @param {Construct} scope the CDK scope used to search or create the cluster
     * @param {string} codePath the code path used to create the layer
     */
    static getOrCreate(scope, codePath) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = 'boto3Layer';
        // eslint-disable-next-line max-len
        const boto3Layer = stack.nestedStackParent ? stack.nestedStackParent.node.tryFindChild(id) : stack.node.tryFindChild(id);
        return boto3Layer || new PreBundledLayer(stack, id, {
            codePath: codePath,
            compatibleRuntimes: [aws_lambda_1.Runtime.PYTHON_3_9, aws_lambda_1.Runtime.PYTHON_3_10],
            layerVersionName: 'ara-boto3-layer',
        });
    }
    /**
     * Constructs a new instance of the PreBundledLayer construct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {PreBundledFunctionProps} props the PreBundledFunction [properties]{@link PreBundledFunctionProps}
     */
    constructor(scope, id, props) {
        if (props.code) {
            throw new Error('Pass "codePath" prop instead of "code" . See CONTRIB_FAQ.md on how to create prebundled Lambda function.');
        }
        let layerProps = { ...props };
        // __dirname is where this file is. In JSII, it is <jsii_tmp_path>/lib/common.
        // When running unit tests, it is ./src/common). In both case, we need to go up one level.
        let assetPath = path.join(__dirname, `../${props.codePath}`);
        layerProps.code = aws_lambda_1.Code.fromAsset(assetPath);
        //delete props that were added to force user input
        delete layerProps.codePath;
        super(scope, id, { ...layerProps });
    }
}
exports.PreBundledLayer = PreBundledLayer;
//# sourceMappingURL=data:application/json;base64,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