"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.PreBundledFunction = void 0;
const path = require("path");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * A Lambda function with prebundled dependencies.
 *
 * It changes of the code path by based on the environment that `cdk synth` is running on.
 *
 * This class is used together with a Projen custom task "copy-resources", and "pip-install".
 * The tasks will ensure that all Python and libraries files are available in "lib" folder,
 * with the same relative path.
 *
 * When this construct is being run in JSII, this file will be in `node_modules` folder
 * (as it's installed as a 3rd party library.) So we need to change reference based on __dirname.
 *
 * Additionally, the Lambda function is using a common Lambda layer with common dependencies when the runtime is Python.
 * Refer to [PreBundledLayer]{@link PreBundledLayer} construct for packaged Python dependencies.
 *
 *  * Usage example:
 * ```typescript
 * import { PreBundledFunction } from 'aws-analytics-reference-architecture';
 *
 * new PreBundledFunction(this, 'PreBundledFunction', {
 *   codePath: 'construct-dir/resources/lambdas/lambda_dir',
 *   lambdaPolicyStatements: findFilePathsFnPolicy,
 *   // you can use any property available in Function CDK Construct including
 *   memorySize: 1024,
 *   runtime: Runtime.PYTHON_3_8,
 *   handler: 'lambda-file-name.handler',
 *   logRetention: RetentionDays.ONE_WEEK,
 *   timeout: Duration.minutes(15),
 * });
 * ```
 */
class PreBundledFunction extends aws_lambda_1.Function {
    /**
     * Constructs a new instance of the PreBundledFunction construct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {PreBundledFunctionProps} props the PreBundledFunction [properties]{@link PreBundledFunctionProps}
     */
    constructor(scope, id, props) {
        if (props.code) {
            throw new Error('Pass "codePath" prop instead of "code" . See CONTRIB_FAQ.md on how to create prebundled Lambda function.');
        }
        let functionProps = { ...props };
        // __dirname is where this file is. In JSII, it is <jsii_tmp_path>/lib/common.
        // When running unit tests, it is ./src/common). In both case, we need to go up one level.
        let assetPath = path.join(__dirname, `../${props.codePath}`);
        functionProps.code = aws_lambda_1.Code.fromAsset(assetPath);
        functionProps.functionName = `${scope.node.id}${id}`;
        let lambdaPolicyStatement = [];
        lambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:CreateLogGroup'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/${functionProps.functionName}`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        lambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:CreateLogStream'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/${functionProps.functionName}*`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        lambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:PutLogEvents'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/${functionProps.functionName}:log-stream:*`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        functionProps.lambdaPolicyStatements?.forEach((element) => {
            lambdaPolicyStatement.push(element);
        });
        //Policy to allow lambda access to cloudwatch logs
        const lambdaExecutionRolePolicy = new aws_iam_1.ManagedPolicy(scope, 'LambdaExecutionRolePolicy' + functionProps.functionName, {
            statements: lambdaPolicyStatement,
            description: 'Policy similar to lambda execution role but scoped down',
        });
        //Use role from props or create an execution role for the lambda and attach to it a policy formed from user input
        const lambdaExecutionRole = props.role ? props.role : new aws_iam_1.Role(scope, 'LambdaExecutionRole' + functionProps.functionName, {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            description: 'Role used by lambda in ARA',
            //roleName: 'LambdaExecutionRole' + functionProps.functionName,
        });
        lambdaExecutionRole.addManagedPolicy(lambdaExecutionRolePolicy);
        let logRetentionLambdaPolicyStatement = [];
        logRetentionLambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:PutRetentionPolicy', 'logs:DeleteRetentionPolicy'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/${functionProps.functionName}`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        logRetentionLambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        //Policy to allow lambda access to cloudwatch logs
        const logRetentionLambdaExecutionRolePolicy = new aws_iam_1.ManagedPolicy(scope, 'LogRetentionLambdaExecutionRolePolicy' + functionProps.functionName, {
            statements: logRetentionLambdaPolicyStatement,
            description: 'Policy used to allow CR for log retention',
        });
        //Create an execution role for the lambda and attach to it a policy formed from user input
        const logRetentionLambdaExecutionRole = new aws_iam_1.Role(scope, 'LogRetentionLambdaExecutionRole' + functionProps.functionName, {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            description: 'Role used by lambda to modify log retention',
            managedPolicies: [logRetentionLambdaExecutionRolePolicy],
            //roleName: 'LogRetLambdaExec' + functionProps.functionName,
        });
        functionProps.role = lambdaExecutionRole;
        functionProps.logRetentionRole = logRetentionLambdaExecutionRole;
        //delete props that were added to force user input
        delete functionProps.codePath;
        delete functionProps.name;
        delete functionProps.lambdaPolicyStatements;
        super(scope, id, { ...functionProps });
    }
}
exports.PreBundledFunction = PreBundledFunction;
//# sourceMappingURL=data:application/json;base64,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