# -*- coding: utf-8 -*-
import re
import uuid
from dateutil import parser
from datetime import datetime
import logging
from django.conf import settings
from django.db import models
from django.dispatch import receiver
from django.db.models.signals import post_save, post_delete
from django.db.models.signals import class_prepared
from django.forms.models import model_to_dict, fields_for_model, ALL_FIELDS
from django.shortcuts import resolve_url
from django.core import validators
from django.utils import timezone
from django.utils import dateformat
from django.utils.timezone import make_aware, make_naive, is_aware, is_naive, now, utc
from django.utils.encoding import smart_str, smart_unicode
from django.utils.encoding import force_text, python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _
from django.contrib.auth import get_user_model

from .rsa_encrypt import encryptor

logger = logging.getLogger(__name__)


def format_exception(status_code, detail):
    data = {
        "code": status_code,
        "msg": detail
    }
    return data


def get_client_ip(request):
    x_forwarded_for = request.META.get('HTTP_X_FORWARDED_FOR', '')
    if x_forwarded_for:
        ip = x_forwarded_for.split(',')[0]
    else:
        ip = request.META.get('REMOTE_ADDR', '')
    return ip


def to_python_boolean(string):
    """Convert a string to boolean.
    """
    string = string.lower()
    if string in ('t', 'true', '1'):
        return True
    if string in ('f', 'false', '0'):
        return False
    raise ValueError("Invalid boolean value: '%s'" % string)


def within_time_range(d1, d2, maxdiff_seconds):
    '''Return true if two datetime.datetime object differs less than the given seconds'''
    delta = d2 - d1 if d2 > d1 else d1 - d2
    # delta.total_seconds() is only available in python 2.7+
    diff = (delta.microseconds + (delta.seconds + delta.days*24*3600) * 1e6) / 1e6
    return diff < maxdiff_seconds


def get_keyword(request, keyword, default=None):
    value = request.GET.get(keyword, default)
    return value or request.POST.get(keyword, default)


def get_header(request, key, default=None):
    key = u"HTTP_{}".format(key).upper()
    return request.META.get(key, default)


def get_data(request, key, default=None):
    return request.data.get(key, default)


def to_datetime(timestamp):
    if isinstance(timestamp, (int, long, float)):
        dt = datetime.fromtimestamp(timestamp, timezone.get_current_timezone())
    else:
        dt = parser.parse(timestamp)
    return dt


def encrypt_rsa(message, public_key):
    try:
        return encryptor.encrypt(message, public_key=public_key)
    except Exception as e:
        logger.debug(e)
        return None


def decrpyt_rsa(message, private_key):
    try:
        return encryptor.decrypt(message, private_key=private_key)
    except Exception as e:
        logger.debug(e)
        return None
