"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynchronousCrawler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const pre_bundled_layer_1 = require("../common/pre-bundled-layer");
/**
 * CrawlerStartWait Construct to start an AWS Glue Crawler execution and asynchronously wait for completion
 */
class SynchronousCrawler extends constructs_1.Construct {
    /**
    * Constructs a new instance of the DataGenerator class
    * @param {Construct} scope the Scope of the CDK Construct
    * @param {string} id the ID of the CDK Construct
    * @param {SynchronousCrawlerProps} props the CrawlerStartWait [properties]{@link SynchronousCrawlerProps}
    * @access public
    */
    constructor(scope, id, props) {
        super(scope, id);
        const stack = aws_cdk_lib_1.Stack.of(this);
        //Lambda policy to allow starting a crawler
        const lambdaCRPolicy = [new aws_iam_1.PolicyStatement({
                resources: [
                    stack.formatArn({
                        region: aws_cdk_lib_1.Aws.REGION,
                        account: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        service: 'glue',
                        resource: 'crawler',
                        resourceName: props.crawlerName,
                    }),
                ],
                actions: [
                    'glue:StartCrawler',
                    'glue:GetCrawler',
                    'glue:StopCrawler',
                ],
            })];
        // AWS Lambda function for the AWS CDK Custom Resource responsible to start crawler
        const crawlerStartFn = new pre_bundled_function_1.PreBundledFunction(this, 'StartFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'synchronous-crawler/resources/lambdas',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: lambdaCRPolicy,
            handler: 'lambda.on_event',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        // AWS Lambda function for the AWS CDK Custom Resource responsible to wait for crawler completion
        const crawlerWaitFn = new pre_bundled_function_1.PreBundledFunction(this, 'WaitFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'synchronous-crawler/resources/lambdas',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: lambdaCRPolicy,
            handler: 'lambda.is_complete',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        // Create an AWS CDK Custom Resource Provider for starting the source crawler and waiting for completion
        const crawlerStartWaitCRP = new custom_resources_1.Provider(this, 'synchronousCrawlerCRP', {
            //onEventFnName: 'SynchronousCrawlerStartFn',
            //isCompleteFnName: 'SynchronousCrawlerWaitFn',
            onEventHandler: crawlerStartFn,
            isCompleteHandler: crawlerWaitFn,
            queryInterval: aws_cdk_lib_1.Duration.seconds(60),
            totalTimeout: aws_cdk_lib_1.Duration.seconds(props.timeout || 600),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        // Create an AWS CDK Custom Resource for starting the source crawler and waiting for completion
        new aws_cdk_lib_1.CustomResource(this, 'synchronousCrawlerCR', {
            serviceToken: crawlerStartWaitCRP.serviceToken,
            properties: {
                CrawlerName: props.crawlerName,
            },
        });
    }
}
exports.SynchronousCrawler = SynchronousCrawler;
_a = JSII_RTTI_SYMBOL_1;
SynchronousCrawler[_a] = { fqn: "aws-analytics-reference-architecture.SynchronousCrawler", version: "2.10.1" };
//# sourceMappingURL=data:application/json;base64,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