"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NotebookPlatform = exports.SSOIdentityType = exports.StudioAuthMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_emr_1 = require("aws-cdk-lib/aws-emr");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const ara_bucket_1 = require("../ara-bucket");
const context_options_1 = require("../common/context-options");
const tracked_construct_1 = require("../common/tracked-construct");
const singleton_kms_key_1 = require("../singleton-kms-key");
const utils_1 = require("../utils");
const notebook_platform_helpers_1 = require("./notebook-platform-helpers");
const emr_eks_cluster_1 = require("../emr-eks-platform/emr-eks-cluster");
/**
 * Enum to define authentication mode for Amazon EMR Studio
 */
var StudioAuthMode;
(function (StudioAuthMode) {
    StudioAuthMode["IAM"] = "IAM";
    StudioAuthMode["SSO"] = "SSO";
})(StudioAuthMode = exports.StudioAuthMode || (exports.StudioAuthMode = {}));
/**
 * Enum to define the type of identity Type in EMR studio
 */
var SSOIdentityType;
(function (SSOIdentityType) {
    SSOIdentityType["USER"] = "USER";
    SSOIdentityType["GROUP"] = "GROUP";
})(SSOIdentityType = exports.SSOIdentityType || (exports.SSOIdentityType = {}));
/**
 * A CDK construct to create a notebook infrastructure based on Amazon EMR Studio and assign users to it
 *
 * This construct is initialized through a constructor that takes as argument an interface defined in {@link NotebookPlatformProps}
 * The construct has a method to add users {@link addUser} the method take as argument {@link NotebookUserOptions}
 *
 * Resources deployed:
 *
 * * An S3 Bucket used by EMR Studio to store the Jupyter notebooks
 * * A KMS encryption Key used to encrypt an S3 bucket used by EMR Studio to store jupyter notebooks
 * * An EMR Studio service Role as defined here, and allowed to access the S3 bucket and KMS key created above
 * * An EMR Studio User Role as defined here - The policy template which is leveraged is the Basic one from the Amazon EMR Studio documentation
 * * Multiple EMR on EKS Managed Endpoints, each for a user or a group of users
 * * An execution role to be passed to the Managed endpoint from a policy provided by the user
 * * Multiple Session Policies that are used to map an EMR Studio user or group to a set of resources they are allowed to access. These resources are:
 *   * EMR Virtual Cluster - created above
 *   * ManagedEndpoint
 *
 *
 * Usage example:
 *
 * ```typescript
 * const emrEks = EmrEksCluster.getOrCreate(stack, {
 *   eksAdminRoleArn: 'arn:aws:iam::012345678912:role/Admin-Admin',
 *   eksClusterName: 'cluster',
 * });
 *
 * const notebookPlatform = new NotebookPlatform(stack, 'platform-notebook', {
 *   emrEks: emrEks,
 *   eksNamespace: 'platformns',
 *   studioName: 'platform',
 *   studioAuthMode: StudioAuthMode.SSO,
 * });
 *
 * // If the S3 bucket is encrypted, add policy to the key for the role
 * const policy1 = new ManagedPolicy(stack, 'MyPolicy1', {
 *   statements: [
 *     new PolicyStatement({
 *       resources: <BUCKET ARN(s)>,
 *       actions: ['s3:*'],
 *     }),
 *     new PolicyStatement({
 *       resources: [
 *         stack.formatArn({
 *           account: Aws.ACCOUNT_ID,
 *           region: Aws.REGION,
 *           service: 'logs',
 *           resource: '*',
 *           arnFormat: ArnFormat.NO_RESOURCE_NAME,
 *         }),
 *       ],
 *       actions: [
 *         'logs:*',
 *       ],
 *     }),
 *   ],
 * });
 *
 * notebookPlatform.addUser([{
 *   identityName: 'user1',
 *   identityType: SSOIdentityType.USER,
 *   notebookManagedEndpoints: [{
 *     emrOnEksVersion: EmrVersion.V6_9,
 *     executionPolicy: policy1,
 *   }],
 * }]);
 *
 * ```
 */
class NotebookPlatform extends tracked_construct_1.TrackedConstruct {
    /**
     * @public
     * Constructs a new instance of the DataPlatform class
     * @param {Construct} scope the Scope of the AWS CDK Construct
     * @param {string} id the ID of the AWS CDK Construct
     * @param {NotebookPlatformProps} props the DataPlatformNotebooks [properties]{@link NotebookPlatformProps}
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingCode: context_options_1.ContextOptions.DATA_ENG_PLATFORM_ID,
        };
        super(scope, id, trackedConstructProps);
        this.studioServicePolicy = [];
        this.studioUserPolicy = [];
        this.studioSubnetList = [];
        this.managedEndpointExecutionPolicyArnMapping = new Map();
        this.authMode = props.studioAuthMode;
        this.vcNamespace = props.eksNamespace ? props.eksNamespace : 'default';
        if (props.idpArn !== undefined) {
            this.federatedIdPARN = props.idpArn;
        }
        //Create encryption key to use with cloudwatch loggroup and S3 bucket storing notebooks and
        this.notebookPlatformEncryptionKey = singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey');
        this.emrVirtualClusterName = 'emr-vc-' + utils_1.Utils.stringSanitizer(props.studioName);
        this.emrEks = props.emrEks;
        //Get the list of private subnets in VPC
        this.studioSubnetList = this.emrEks.eksCluster.vpc.selectSubnets({
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        }).subnetIds;
        //Create a virtual cluster a give it a name of 'emr-vc-'+studioName provided by user
        this.emrVirtCluster = this.emrEks.addEmrVirtualCluster(this, {
            createNamespace: true,
            eksNamespace: props.eksNamespace,
            name: utils_1.Utils.stringSanitizer(this.emrVirtualClusterName),
        });
        //Create a security group to be attached to the studio workspaces
        this.workSpaceSecurityGroup = new aws_ec2_1.SecurityGroup(this, `workspaceSecurityGroup-${props.studioName}`, {
            vpc: this.emrEks.eksCluster.vpc,
            securityGroupName: 'workSpaceSecurityGroup-' + props.studioName,
            allowAllOutbound: false,
        });
        //Tag workSpaceSecurityGroup to be used with EMR Studio
        aws_cdk_lib_1.Tags.of(this.workSpaceSecurityGroup).add('for-use-with-amazon-emr-managed-policies', 'true');
        //Create a security group to be attached to the engine for EMR
        //This is mandatory for Amazon EMR Studio although we are not using EMR on EC2
        this.engineSecurityGroup = new aws_ec2_1.SecurityGroup(this, `engineSecurityGroup-${props.studioName}`, {
            vpc: this.emrEks.eksCluster.vpc,
            securityGroupName: 'engineSecurityGroup-' + props.studioName,
            allowAllOutbound: false,
        });
        //Tag engineSecurityGroup to be used with EMR Studio
        aws_cdk_lib_1.Tags.of(this.engineSecurityGroup).add('for-use-with-amazon-emr-managed-policies', 'true');
        //Create S3 bucket to store EMR Studio workspaces
        //Bucket is kept after destroying the construct
        this.workspacesBucket = ara_bucket_1.AraBucket.getOrCreate(this, {
            bucketName: 'workspaces-bucket-' + utils_1.Utils.stringSanitizer(props.studioName),
            encryptionKey: this.notebookPlatformEncryptionKey,
            serverAccessLogsPrefix: `${props.studioName}-workspace`,
        });
        this.notebookPlatformEncryptionKey.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            principals: [new aws_iam_1.ServicePrincipal('elasticmapreduce.amazonaws.com')],
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'kms:Encrypt*',
                'kms:Decrypt*',
                'kms:ReEncrypt*',
                'kms:GenerateDataKey*',
                'kms:Describe*',
            ],
            conditions: {
                ArnEquals: {
                    'aws:s3:arn': `arn:aws:s3:::${'ara-workspaces-bucket-' + aws_cdk_lib_1.Aws.ACCOUNT_ID + '-' + utils_1.Utils.stringSanitizer(props.studioName)}/*`,
                },
            },
            resources: ['*'],
        }), false);
        //Create a Managed policy for Studio service role
        this.studioServicePolicy.push(aws_iam_1.ManagedPolicy.fromManagedPolicyArn(this, `StudioServiceManagedPolicy-${props.studioName}`, notebook_platform_helpers_1.createStudioServiceRolePolicy(this, this.notebookPlatformEncryptionKey.keyArn, this.workspacesBucket.bucketName, props.studioName)));
        //Create a role for the Studio
        this.studioServiceRole = new aws_iam_1.Role(this, `studioServiceRole-${props.studioName}`, {
            assumedBy: new aws_iam_1.ServicePrincipal(NotebookPlatform.STUDIO_PRINCIPAL),
            roleName: 'studioServiceRole+' + utils_1.Utils.stringSanitizer(props.studioName),
            managedPolicies: this.studioServicePolicy,
        });
        // Create an EMR Studio user role only if the user uses SSO as authentication mode
        if (props.studioAuthMode === 'SSO') {
            //Get Managed policy for Studio user role and put it in an array to be assigned to a user role
            this.studioUserPolicy.push(aws_iam_1.ManagedPolicy.fromManagedPolicyArn(this, `StudioUserManagedPolicy-${props.studioName}`, notebook_platform_helpers_1.createStudioUserRolePolicy(this, props.studioName, this.studioServiceRole.roleName)));
            //Create a role for the EMR Studio user, this roles is further restricted by session policy for each user
            this.studioUserRole = new aws_iam_1.Role(this, `studioUserRole-${props.studioName}`, {
                assumedBy: new aws_iam_1.ServicePrincipal(NotebookPlatform.STUDIO_PRINCIPAL),
                roleName: 'studioUserRole+' + utils_1.Utils.stringSanitizer(props.studioName),
                managedPolicies: this.studioUserPolicy,
            });
        }
        // Create the EMR Studio
        this.studioInstance = new aws_emr_1.CfnStudio(this, `Studio-${props.studioName}`, {
            authMode: props.studioAuthMode,
            defaultS3Location: 's3://' + this.workspacesBucket.bucketName + '/',
            engineSecurityGroupId: this.engineSecurityGroup.securityGroupId,
            name: props.studioName,
            serviceRole: this.studioServiceRole.roleArn,
            subnetIds: this.studioSubnetList,
            userRole: this.studioUserRole ? this.studioUserRole.roleArn : undefined,
            vpcId: this.emrEks.eksCluster.vpc.vpcId,
            workspaceSecurityGroupId: this.workSpaceSecurityGroup.securityGroupId,
            idpAuthUrl: props.idpAuthUrl ? props.idpAuthUrl : undefined,
            idpRelayStateParameterName: props.idpRelayStateParameterName ? props.idpRelayStateParameterName : undefined,
        });
        // Set the Studio URL and Studio Id this is used in session Mapping for
        // EMR Studio when {@linkcode addFederatedUsers} or {@linkcode addSSOUsers} are called
        this.studioName = props.studioName;
        //Set the Studio Id to use for SessionMapping
        this.studioId = this.studioInstance.attrStudioId;
        new aws_cdk_lib_1.CfnOutput(this, `URL for EMR Studio: ${this.studioName}`, {
            value: this.studioInstance.attrUrl,
        });
        /*//Return EMR Studio URL as CfnOutput
        if (this.nestedStackParent != undefined) {
          new CfnOutput(this.nestedStackParent, `URL for EMR Studio: ${this.studioName}`, {
            value: this.studioInstance.attrUrl,
          });
        }*/
    }
    /**
     * @public
     * Method to add users, take a list of userDefinition and will create a managed endpoints for each user
     * and create an IAM Policy scoped to the list managed endpoints
     * @param {NotebookUserOptions} userList list of users
     */
    addUser(userList) {
        //Initialize the managedEndpointArns
        //Used to store the arn of managed endpoints after creation for each users
        //This is used to update the IAM policy
        let managedEndpointArns = [];
        //let managedEndpointObjects: Map<string, CustomResource> = new Map <string, CustomResource> ();
        let iamRolePolicy;
        let iamRoleList = [];
        //Loop through each user and create its managed endpoint(s) as defined by the user
        for (let user of userList) {
            //For each policy create a role and then pass it to addManageEndpoint to create an endpoint
            user.notebookManagedEndpoints.forEach((notebookManagedEndpoint, index) => {
                if (!this.managedEndpointExecutionPolicyArnMapping.has(notebookManagedEndpoint.managedEndpointName)) {
                    //For each user or group, create a new managedEndpoint
                    //ManagedEndpoint ARN is used to update and scope the session policy of the user or group
                    let emrOnEksVersion = user.notebookManagedEndpoints[index].emrOnEksVersion;
                    let configOverride = user.notebookManagedEndpoints[index].configurationOverrides;
                    let execRole = this.emrEks.createExecutionRole(this, `${user.identityName}${index}`, notebookManagedEndpoint.executionPolicy, this.vcNamespace, `${notebookManagedEndpoint.managedEndpointName}-execRole`);
                    iamRoleList.push(execRole);
                    let managedEndpoint = this.emrEks.addManagedEndpoint(this, `${this.studioName}${utils_1.Utils.stringSanitizer(notebookManagedEndpoint.managedEndpointName)}`, {
                        managedEndpointName: `${this.studioName}-${notebookManagedEndpoint.managedEndpointName}`,
                        virtualClusterId: this.emrVirtCluster.attrId,
                        executionRole: execRole,
                        emrOnEksVersion: emrOnEksVersion ? emrOnEksVersion : NotebookPlatform.DEFAULT_EMR_VERSION,
                        configurationOverrides: configOverride ? configOverride : undefined,
                    });
                    managedEndpoint.node.addDependency(this.emrEks);
                    //Get the Security Group of the ManagedEndpoint which is the Engine Security Group
                    let engineSecurityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(this, `engineSecurityGroup${user.identityName}${index}`, managedEndpoint.getAttString('securityGroup'));
                    aws_cdk_lib_1.Tags.of(engineSecurityGroup).add('for-use-by-analytics-reference-architecture', 'true');
                    let vpcCidrBlock = this.emrEks.eksCluster.vpc.vpcCidrBlock;
                    //Update workspace Security Group to allow outbound traffic on port 18888 toward Engine Security Group
                    this.workSpaceSecurityGroup.addEgressRule(aws_ec2_1.Peer.ipv4(vpcCidrBlock), aws_ec2_1.Port.tcp(18888), 'Allow traffic to EMR');
                    this.engineSecurityGroup?.addIngressRule(aws_ec2_1.Peer.ipv4(vpcCidrBlock), aws_ec2_1.Port.tcp(18888), 'Allow traffic from EMR Studio');
                    this.workSpaceSecurityGroup.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
                    //Tag the Security Group of the ManagedEndpoint to be used with EMR Studio
                    aws_cdk_lib_1.Tags.of(engineSecurityGroup).add('for-use-with-amazon-emr-managed-policies', 'true');
                    //Add the managedendpointArn to @managedEndpointExcutionPolicyArnMapping
                    //This is to avoid the creation an endpoint with the same policy twice
                    //Save resources and reduce the deployment time
                    // TODO check the emr version is the same => to be fixed on a later commit need to solve adding a tuple to a JS map
                    this.managedEndpointExecutionPolicyArnMapping.set(notebookManagedEndpoint.managedEndpointName, managedEndpoint.getAttString('arn'));
                    //Push the managedendpoint arn to be used in to build the policy to attach to it
                    managedEndpointArns.push(managedEndpoint.getAttString('arn'));
                }
                else {
                    managedEndpointArns.push(this.managedEndpointExecutionPolicyArnMapping.get(notebookManagedEndpoint.managedEndpointName));
                }
            });
            if (this.authMode === 'IAM' && this.federatedIdPARN === undefined) {
                //Create the role policy and gets its ARN
                iamRolePolicy = notebook_platform_helpers_1.createIAMRolePolicy(this, user, this.studioServiceRole.roleName, managedEndpointArns, this.studioId);
                let iamUser = user.iamUser ? user.iamUser : aws_iam_1.User.fromUserName(this, `IAMUSER-${user.identityName}`, user.identityName);
                iamRolePolicy.attachToUser(iamUser);
            }
            else if (this.authMode === 'IAM' && this.federatedIdPARN != undefined) {
                //Create the role policy and gets its ARN
                iamRolePolicy = notebook_platform_helpers_1.createIAMRolePolicy(this, user, this.studioServiceRole.roleName, managedEndpointArns, this.studioId);
                notebook_platform_helpers_1.createIAMFederatedRole(this, iamRolePolicy, this.federatedIdPARN, user.identityName, this.studioId);
            }
            else if (this.authMode === 'SSO') {
                //Create the session policy and gets its ARN
                let sessionPolicyArn = notebook_platform_helpers_1.createUserSessionPolicy(this, user, this.studioServiceRole.roleName, managedEndpointArns, this.studioId);
                if (user.identityType == 'USER' || user.identityType == 'GROUP') {
                    //Map a session to user or group
                    new aws_emr_1.CfnStudioSessionMapping(this, 'studioUser' + user.identityName + user.identityName, {
                        identityName: user.identityName,
                        identityType: user.identityType,
                        sessionPolicyArn: sessionPolicyArn,
                        studioId: this.studioId,
                    });
                }
                else {
                    throw new Error(`identityType should be either USER or GROUP not ${user.identityType}`);
                }
            }
        }
        return iamRoleList;
    }
}
exports.NotebookPlatform = NotebookPlatform;
_a = JSII_RTTI_SYMBOL_1;
NotebookPlatform[_a] = { fqn: "aws-analytics-reference-architecture.NotebookPlatform", version: "2.10.1" };
NotebookPlatform.DEFAULT_EMR_VERSION = emr_eks_cluster_1.EmrVersion.V6_8;
NotebookPlatform.STUDIO_PRINCIPAL = 'elasticmapreduce.amazonaws.com';
//# sourceMappingURL=data:application/json;base64,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