"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrEksCluster = exports.EmrVersion = exports.Autoscaler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const aws_emrcontainers_1 = require("aws-cdk-lib/aws-emrcontainers");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const SimpleBase = require("simple-base");
const ara_bucket_1 = require("../ara-bucket");
const context_options_1 = require("../common/context-options");
const tracked_construct_1 = require("../common/tracked-construct");
const singleton_kms_key_1 = require("../singleton-kms-key");
const emr_managed_endpoint_1 = require("./emr-managed-endpoint");
const CriticalDefaultConfig = require("./resources/k8s/emr-eks-config/critical.json");
const NotebookDefaultConfig = require("./resources/k8s/emr-eks-config/notebook-pod-template-ready.json");
const SharedDefaultConfig = require("./resources/k8s/emr-eks-config/shared.json");
const K8sRoleBinding = require("./resources/k8s/rbac/emr-containers-role-binding.json");
const K8sRole = require("./resources/k8s/rbac/emr-containers-role.json");
const emr_eks_cluster_helpers_1 = require("./emr-eks-cluster-helpers");
const singleton_launch_template_1 = require("../singleton-launch-template");
const emr_eks_nodegroup_asg_tag_1 = require("./emr-eks-nodegroup-asg-tag");
const emr_eks_job_template_1 = require("./emr-eks-job-template");
const vpc_helper_1 = require("./vpc-helper");
/**
 * The different autoscaler available with EmrEksCluster
 */
var Autoscaler;
(function (Autoscaler) {
    Autoscaler["KARPENTER"] = "KARPENTER";
    Autoscaler["CLUSTER_AUTOSCALER"] = "CLUSTER_AUTOSCALER";
})(Autoscaler = exports.Autoscaler || (exports.Autoscaler = {}));
/**
 * The different EMR versions available on EKS
 */
var EmrVersion;
(function (EmrVersion) {
    EmrVersion["V6_10"] = "emr-6.10.0-latest";
    EmrVersion["V6_9"] = "emr-6.9.0-latest";
    EmrVersion["V6_8"] = "emr-6.8.0-latest";
    EmrVersion["V6_7"] = "emr-6.7.0-latest";
    EmrVersion["V6_6"] = "emr-6.6.0-latest";
    EmrVersion["V6_5"] = "emr-6.5.0-latest";
    EmrVersion["V6_4"] = "emr-6.4.0-latest";
    EmrVersion["V6_3"] = "emr-6.3.0-latest";
    EmrVersion["V6_2"] = "emr-6.2.0-latest";
    EmrVersion["V5_33"] = "emr-5.33.0-latest";
    EmrVersion["V5_32"] = "emr-5.32.0-latest";
})(EmrVersion = exports.EmrVersion || (exports.EmrVersion = {}));
/**
 * EmrEksCluster Construct packaging all the resources and configuration required to run Amazon EMR on EKS.
 * It deploys:
 * * An EKS cluster (VPC configuration can be customized)
 * * A tooling nodegroup to run tools including the Kubedashboard and the Cluster Autoscaler
 * * Optionally multiple nodegroups (one per AZ) for critical/shared/notebook EMR workloads
 * * Additional nodegroups can be configured
 *
 * The construct will upload on S3 the Pod templates required to run EMR jobs on the default nodegroups.
 * It will also parse and store the configuration of EMR on EKS jobs for each default nodegroup in object parameters
 *
 * Methods are available to add EMR Virtual Clusters to the EKS cluster and to create execution roles for the virtual clusters.
 *
 * Usage example:
 *
 * ```typescript
 * const emrEks: EmrEksCluster = EmrEksCluster.getOrCreate(stack, {
 *   eksAdminRoleArn: <ROLE_ARN>,
 *   eksClusterName: <CLUSTER_NAME>,
 * });
 *
 * const virtualCluster = emrEks.addEmrVirtualCluster(stack, {
 *   name: <Virtual_Cluster_Name>,
 *   createNamespace: <TRUE OR FALSE>,
 *   eksNamespace: <K8S_namespace>,
 * });
 *
 * const role = emrEks.createExecutionRole(stack, 'ExecRole',{
 *   policy: <POLICY>,
 * })
 *
 * // EMR on EKS virtual cluster ID
 * cdk.CfnOutput(self, 'VirtualClusterId',value = virtualCluster.attr_id)
 * // Job config for each nodegroup
 * cdk.CfnOutput(self, "CriticalConfig", value = emrEks.criticalDefaultConfig)
 * cdk.CfnOutput(self, "SharedConfig", value = emrEks.sharedDefaultConfig)
 * // Execution role arn
 * cdk.CfnOutput(self,'ExecRoleArn', value = role.roleArn)
 * ```
 *
 */
class EmrEksCluster extends tracked_construct_1.TrackedConstruct {
    /**
     * Constructs a new instance of the EmrEksCluster construct.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps}
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingCode: context_options_1.ContextOptions.EMR_EKS_TRACKING_ID,
        };
        super(scope, id, trackedConstructProps);
        this.clusterName = props.eksClusterName ?? EmrEksCluster.DEFAULT_CLUSTER_NAME;
        //Define EKS cluster logging
        const eksClusterLogging = [
            aws_eks_1.ClusterLoggingTypes.API,
            aws_eks_1.ClusterLoggingTypes.AUTHENTICATOR,
            aws_eks_1.ClusterLoggingTypes.SCHEDULER,
            aws_eks_1.ClusterLoggingTypes.CONTROLLER_MANAGER,
            aws_eks_1.ClusterLoggingTypes.AUDIT,
        ];
        //Set the autoscaler mechanism flag
        this.isKarpenter = props.autoscaling == Autoscaler.KARPENTER ? true : false;
        this.defaultNodes = props.defaultNodes == undefined ? true : props.defaultNodes;
        // Create a role to be used as instance profile for nodegroups
        this.ec2InstanceNodeGroupRole = new aws_iam_1.Role(this, 'ec2InstanceNodeGroupRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
        });
        //attach policies to the role to be used by the nodegroups
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        // Create the custom resource provider for tagging the EC2 Auto Scaling groups
        this.nodegroupAsgTagsProviderServiceToken = new emr_eks_nodegroup_asg_tag_1.EmrEksNodegroupAsgTagProvider(this, 'AsgTagProvider', {
            eksClusterName: this.clusterName,
        }).provider.serviceToken;
        // Create the custom resource provider for tagging the EC2 Auto Scaling groups
        this.jobTemplateProviderToken = new emr_eks_job_template_1.EmrEksJobTemplateProvider(this, 'jobTemplateProvider').provider.serviceToken;
        // create an Amazon EKS CLuster with default parameters if not provided in the properties
        if (props.eksCluster == undefined) {
            let eksVpc = props.vpcCidr ? vpc_helper_1.vpcBootstrap(scope, props.vpcCidr, this.clusterName) : props.eksVpc;
            this.eksCluster = new aws_eks_1.Cluster(scope, `${this.clusterName}Cluster`, {
                defaultCapacity: 0,
                clusterName: this.clusterName,
                version: props.kubernetesVersion || EmrEksCluster.DEFAULT_EKS_VERSION,
                clusterLogging: eksClusterLogging,
                kubectlLayer: props.kubectlLambdaLayer ?? undefined,
                vpc: eksVpc
            });
            //Create VPC flow log for the EKS VPC
            let eksVpcFlowLogLogGroup = new aws_logs_1.LogGroup(this, 'eksVpcFlowLogLogGroup', {
                logGroupName: `/aws/emr-eks-vpc-flow/${this.clusterName}`,
                encryptionKey: singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey'),
                retention: aws_logs_1.RetentionDays.ONE_WEEK,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            });
            //Allow vpc flowlog to access KMS key to encrypt logs
            singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey').addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                principals: [new aws_iam_1.ServicePrincipal(`logs.${aws_cdk_lib_1.Stack.of(this).region}.amazonaws.com`)],
                actions: [
                    'kms:Encrypt*',
                    'kms:Decrypt*',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:Describe*',
                ],
                conditions: {
                    ArnLike: {
                        'kms:EncryptionContext:aws:logs:arn': `arn:aws:logs:${aws_cdk_lib_1.Stack.of(this).region}:${aws_cdk_lib_1.Stack.of(this).account}:*`,
                    },
                },
                resources: ['*'],
            }));
            //Setup the VPC flow logs
            const iamRoleforFlowLog = new aws_iam_1.Role(this, 'iamRoleforFlowLog', {
                assumedBy: new aws_iam_1.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
            this.eksCluster.vpc.addFlowLog('eksVpcFlowLog', {
                destination: aws_ec2_1.FlowLogDestination.toCloudWatchLogs(eksVpcFlowLogLogGroup, iamRoleforFlowLog),
            });
            //Setting up the cluster with the required controller
            emr_eks_cluster_helpers_1.eksClusterSetup(this, this, props.eksAdminRoleArn);
            //Deploy the right autoscaler using the flag set earlier 
            if (this.isKarpenter) {
                this.karpenterChart = emr_eks_cluster_helpers_1.karpenterSetup(this.eksCluster, this.clusterName, this, props.karpenterVersion || EmrEksCluster.DEFAULT_KARPENTER_VERSION);
            }
            else {
                const kubernetesVersion = props.kubernetesVersion ?? EmrEksCluster.DEFAULT_EKS_VERSION;
                emr_eks_cluster_helpers_1.clusterAutoscalerSetup(this.eksCluster, this.clusterName, this, kubernetesVersion);
            }
        }
        else {
            //Initialize with the provided EKS Cluster
            this.eksCluster = props.eksCluster;
        }
        //Check if the user want to use the default nodegroup and
        //Add the default nodegroup configured and optimized to run Spark workloads
        if (this.defaultNodes && props.autoscaling == Autoscaler.CLUSTER_AUTOSCALER) {
            emr_eks_cluster_helpers_1.setDefaultManagedNodeGroups(this);
        }
        //Check if there user want to use the default Karpenter provisioners and
        //Add the defaults pre-configured and optimized to run Spark workloads
        if (this.defaultNodes && props.autoscaling == Autoscaler.KARPENTER) {
            emr_eks_cluster_helpers_1.setDefaultKarpenterProvisioners(this);
        }
        ara_bucket_1.AraBucket.getOrCreate(this, { bucketName: 's3-access-logs', objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_PREFERRED });
        // Tags the Amazon VPC and Subnets of the Amazon EKS Cluster
        aws_cdk_lib_1.Tags.of(this.eksCluster.vpc).add('for-use-with-amazon-emr-managed-policies', 'true');
        this.eksCluster.vpc.privateSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('for-use-with-amazon-emr-managed-policies', 'true'));
        this.eksCluster.vpc.publicSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('for-use-with-amazon-emr-managed-policies', 'true'));
        // Create Amazon IAM ServiceLinkedRole for Amazon EMR and add to kubernetes configmap
        // required to add a dependency on the Amazon EMR virtual cluster
        this.emrServiceRole = new aws_iam_1.CfnServiceLinkedRole(this, 'EmrServiceRole', {
            awsServiceName: 'emr-containers.amazonaws.com',
        });
        this.eksCluster.awsAuth.addRoleMapping(aws_iam_1.Role.fromRoleArn(this, 'ServiceRoleForAmazonEMRContainers', `arn:aws:iam::${aws_cdk_lib_1.Stack.of(this).account}:role/AWSServiceRoleForAmazonEMRContainers`), {
            username: 'emr-containers',
            groups: ['']
        });
        // Create an Amazon S3 Bucket for default podTemplate assets
        this.assetBucket = ara_bucket_1.AraBucket.getOrCreate(this, { bucketName: `${this.clusterName.toLowerCase()}-emr-eks-assets`, encryption: aws_s3_1.BucketEncryption.KMS_MANAGED });
        // Configure the podTemplate location
        this.podTemplateLocation = {
            bucketName: this.assetBucket.bucketName,
            objectKey: `${this.clusterName}/pod-template`,
        };
        aws_cdk_lib_1.Tags.of(this.assetBucket).add('for-use-with', 'cdk-analytics-reference-architecture');
        let s3DeploymentLambdaPolicyStatement = [];
        s3DeploymentLambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        //Policy to allow lambda access to cloudwatch logs
        const lambdaExecutionRolePolicy = new aws_iam_1.ManagedPolicy(this, 's3BucketDeploymentPolicy', {
            statements: s3DeploymentLambdaPolicyStatement,
            description: 'Policy used by S3 deployment cdk construct',
        });
        //Create an execution role for the lambda and attach to it a policy formed from user input
        this.assetUploadBucketRole = new aws_iam_1.Role(this, 's3BucketDeploymentRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            description: 'Role used by S3 deployment cdk construct',
            managedPolicies: [lambdaExecutionRolePolicy],
        });
        // Upload the default podTemplate to the Amazon S3 asset bucket
        this.uploadPodTemplate('defaultPodTemplates', path_1.join(__dirname, 'resources/k8s/pod-template'));
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the notebook default config
        NotebookDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-driver.yaml`);
        NotebookDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-executor.yaml`);
        this.notebookDefaultConfig = JSON.parse(JSON.stringify(NotebookDefaultConfig));
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the critical default config
        CriticalDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-driver.yaml`);
        CriticalDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-executor.yaml`);
        this.criticalDefaultConfig = JSON.stringify(CriticalDefaultConfig);
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the shared default config
        SharedDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-driver.yaml`);
        SharedDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-executor.yaml`);
        this.sharedDefaultConfig = JSON.stringify(SharedDefaultConfig);
        // Set the custom resource provider service token here to avoid circular dependencies
        this.managedEndpointProviderServiceToken = new emr_managed_endpoint_1.EmrManagedEndpointProvider(this, 'ManagedEndpointProvider', {
            assetBucket: this.assetBucket,
        }).provider.serviceToken;
        // Provide the podTemplate location on Amazon S3
        new aws_cdk_lib_1.CfnOutput(this, 'podTemplateLocation', {
            description: 'Use podTemplates in Amazon EMR jobs from this Amazon S3 Location',
            value: this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}`),
        });
    }
    /**
     * Get an existing EmrEksCluster based on the cluster name property or create a new one
     * only one EKS cluster can exist per stack
     * @param {Construct} scope the CDK scope used to search or create the cluster
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps} if created
     */
    static getOrCreate(scope, props) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = props.eksClusterName || EmrEksCluster.DEFAULT_CLUSTER_NAME;
        let emrEksCluster;
        if (stack.node.tryFindChild(id) == undefined) {
            emrEksCluster = new EmrEksCluster(stack, id, props);
        }
        return stack.node.tryFindChild(id) || emrEksCluster;
    }
    /**
     * Add a new Amazon EMR Virtual Cluster linked to Amazon EKS Cluster.
     * @param {Construct} scope of the stack where virtual cluster is deployed
     * @param {EmrVirtualClusterOptions} options the EmrVirtualClusterProps [properties]{@link EmrVirtualClusterProps}
     */
    addEmrVirtualCluster(scope, options) {
        const eksNamespace = options.eksNamespace ?? 'default';
        const regex = /^[a-z0-9]+$/g;
        if (!eksNamespace.match(regex)) {
            throw new Error(`Namespace provided violates the constraints of Namespace naming ${eksNamespace}`);
        }
        const ns = options.createNamespace
            ? this.eksCluster.addManifest(`${options.name}Namespace`, {
                apiVersion: 'v1',
                kind: 'Namespace',
                metadata: { name: eksNamespace },
            })
            : null;
        // deep clone the Role template object and replace the namespace
        const k8sRole = JSON.parse(JSON.stringify(K8sRole));
        k8sRole.metadata.namespace = eksNamespace;
        const role = this.eksCluster.addManifest(`${options.name}Role`, k8sRole);
        role.node.addDependency(this.emrServiceRole);
        if (ns)
            role.node.addDependency(ns);
        // deep clone the Role Binding template object and replace the namespace
        const k8sRoleBinding = JSON.parse(JSON.stringify(K8sRoleBinding));
        k8sRoleBinding.metadata.namespace = eksNamespace;
        const roleBinding = this.eksCluster.addManifest(`${options.name}RoleBinding`, k8sRoleBinding);
        roleBinding.node.addDependency(role);
        const virtCluster = new aws_emrcontainers_1.CfnVirtualCluster(scope, `${options.name}VirtualCluster`, {
            name: options.name,
            containerProvider: {
                id: this.clusterName,
                type: 'EKS',
                info: { eksInfo: { namespace: options.eksNamespace || 'default' } },
            },
            tags: [{
                    key: 'for-use-with',
                    value: 'cdk-analytics-reference-architecture',
                }],
        });
        virtCluster.node.addDependency(roleBinding);
        virtCluster.node.addDependency(this.emrServiceRole);
        if (ns)
            virtCluster.node.addDependency(ns);
        aws_cdk_lib_1.Tags.of(virtCluster).add('for-use-with', 'cdk-analytics-reference-architecture');
        return virtCluster;
    }
    /**
     * Creates a new Amazon EMR managed endpoint to be used with Amazon EMR Virtual Cluster .
     * CfnOutput can be customized.
     * @param {Construct} scope the scope of the stack where managed endpoint is deployed
     * @param {string} id the CDK id for endpoint
     * @param {EmrManagedEndpointOptions} options the EmrManagedEndpointOptions to configure the Amazon EMR managed endpoint
     */
    addManagedEndpoint(scope, id, options) {
        if (options.managedEndpointName.length > 64) {
            throw new Error(`error managed endpoint name length is greater than 64 ${id}`);
        }
        if (this.notebookDefaultConfig == undefined) {
            throw new Error('error empty configuration override is not supported on non-default nodegroups');
        }
        let jsonConfigurationOverrides;
        // TODO this need to be broadended to all possible emr configuration
        // try {
        //   //Check if the configOverride provided by user is valid
        //   let isConfigOverrideValid: boolean = validateSchema(JSON.stringify(configOverrideSchema), options.configurationOverrides);
        //   jsonConfigurationOverrides = isConfigOverrideValid ? options.configurationOverrides : this.notebookDefaultConfig;
        // } catch (error) {
        //   throw new Error(`The configuration override is not valid JSON : ${options.configurationOverrides}`);
        // }
        jsonConfigurationOverrides = options.configurationOverrides ? options.configurationOverrides : this.notebookDefaultConfig;
        // Create custom resource with async waiter until the Amazon EMR Managed Endpoint is created
        const cr = new aws_cdk_lib_1.CustomResource(scope, id, {
            serviceToken: this.managedEndpointProviderServiceToken,
            properties: {
                clusterId: options.virtualClusterId,
                executionRoleArn: options.executionRole.roleArn,
                endpointName: options.managedEndpointName,
                releaseLabel: options.emrOnEksVersion || EmrEksCluster.DEFAULT_EMR_VERSION,
                configurationOverrides: jsonConfigurationOverrides,
            },
        });
        cr.node.addDependency(this.eksCluster);
        return cr;
    }
    /**
   * Add new nodegroups to the cluster for Amazon EMR on EKS. This method overrides Amazon EKS nodegroup options then create the nodegroup.
   * If no subnet is provided, it creates one nodegroup per private subnet in the Amazon EKS Cluster.
   * If NVME local storage is used, the user_data is modified.
   * @param {string} id the CDK ID of the resource
   * @param {EmrEksNodegroupOptions} props the EmrEksNodegroupOptions [properties]{@link EmrEksNodegroupOptions}
   */
    addEmrEksNodegroup(id, props) {
        if (this.isKarpenter) {
            throw new Error(`You can\'t use this method when the autoscaler is set to ${Autoscaler.KARPENTER}`);
        }
        // Get the subnet from Properties or one private subnet for each AZ
        const subnetList = props.subnet ? [props.subnet] : this.eksCluster.vpc.selectSubnets({
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
        }).subnets;
        // Add Amazon SSM agent to the user data
        var userData = [
            'yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm',
            'systemctl enable amazon-ssm-agent',
            'systemctl start amazon-ssm-agent',
        ];
        var launchTemplateName = `EmrEksLaunch-${this.clusterName}`;
        // If the Nodegroup uses NVMe, add user data to configure them
        if (props.mountNvme) {
            userData = userData.concat([
                'INSTANCE_TYPE=$(ec2-metadata -t)',
                'if [[ $INSTANCE_TYPE == *"2xlarge"* ]]; then',
                'DEVICE="/dev/nvme1n1"',
                'mkfs.ext4 $DEVICE',
                'else',
                'yum install -y mdadm',
                'SSD_NVME_DEVICE_LIST=("/dev/nvme1n1" "/dev/nvme2n1")',
                'SSD_NVME_DEVICE_COUNT=${#SSD_NVME_DEVICE_LIST[@]}',
                'RAID_DEVICE=${RAID_DEVICE:-/dev/md0}',
                'RAID_CHUNK_SIZE=${RAID_CHUNK_SIZE:-512}  # Kilo Bytes',
                'FILESYSTEM_BLOCK_SIZE=${FILESYSTEM_BLOCK_SIZE:-4096}  # Bytes',
                'STRIDE=$((RAID_CHUNK_SIZE * 1024 / FILESYSTEM_BLOCK_SIZE))',
                'STRIPE_WIDTH=$((SSD_NVME_DEVICE_COUNT * STRIDE))',
                'mdadm --create --verbose "$RAID_DEVICE" --level=0 -c "${RAID_CHUNK_SIZE}" --raid-devices=${#SSD_NVME_DEVICE_LIST[@]} "${SSD_NVME_DEVICE_LIST[@]}"',
                'while [ -n "$(mdadm --detail "$RAID_DEVICE" | grep -ioE \'State :.*resyncing\')" ]; do',
                'echo "Raid is resyncing.."',
                'sleep 1',
                'done',
                'echo "Raid0 device $RAID_DEVICE has been created with disks ${SSD_NVME_DEVICE_LIST[*]}"',
                'mkfs.ext4 -m 0 -b "$FILESYSTEM_BLOCK_SIZE" -E "stride=$STRIDE,stripe-width=$STRIPE_WIDTH" "$RAID_DEVICE"',
                'DEVICE=$RAID_DEVICE',
                'fi',
                'systemctl stop docker',
                'mkdir -p /var/lib/kubelet/pods',
                'mount $DEVICE /var/lib/kubelet/pods',
                'chmod 750 /var/lib/docker',
                'systemctl start docker',
            ]);
            launchTemplateName = `EmrEksNvmeLaunch-${this.clusterName}`;
        }
        // Add headers and footers to user data
        const userDataMime = aws_cdk_lib_1.Fn.base64(`MIME-Version: 1.0
Content-Type: multipart/mixed; boundary="==MYBOUNDARY=="

--==MYBOUNDARY==
Content-Type: text/x-shellscript; charset="us-ascii"

#!/bin/bash
${userData.join('\r\n')}

--==MYBOUNDARY==--\\
`);
        // Create a new LaunchTemplate or reuse existing one
        const lt = singleton_launch_template_1.SingletonCfnLaunchTemplate.getOrCreate(this, launchTemplateName, userDataMime);
        // Create one Amazon EKS Nodegroup per subnet
        subnetList.forEach((subnet, index) => {
            // Make the ID unique across AZ using the index of subnet in the subnet list
            const resourceId = `${id}-${index}`;
            const nodegroupName = props.nodegroupName ? `${props.nodegroupName}-${index}` : resourceId;
            // Add the user data to the NodegroupOptions
            const nodeGroupParameters = {
                ...props,
                ...{
                    launchTemplateSpec: {
                        id: lt.ref,
                        version: lt.attrLatestVersionNumber,
                    },
                    subnets: {
                        subnets: [subnet],
                    },
                    nodegroupName: nodegroupName,
                },
            };
            // Create the Amazon EKS Nodegroup
            this.addNodegroupCapacity(resourceId, nodeGroupParameters);
        });
    }
    /**
     * Add a new Amazon EKS Nodegroup to the cluster.
     * This method is used to add a nodegroup to the Amazon EKS cluster and automatically set tags based on labels and taints
     *  so it can be used for the cluster autoscaler.
     * @param {string} nodegroupId the ID of the nodegroup
     * @param {EmrEksNodegroupOptions} options the EmrEksNodegroup [properties]{@link EmrEksNodegroupOptions}
     */
    addNodegroupCapacity(nodegroupId, options) {
        const nodegroup = this.eksCluster.addNodegroupCapacity(nodegroupId, options);
        // Adding the Amazon SSM policy
        nodegroup.role.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // Add tags for the Cluster Autoscaler IAM scoping
        aws_cdk_lib_1.Tags.of(nodegroup).add('eks:cluster-name', `${this.clusterName}`);
        // Add tags for the Cluster Autoscaler management
        aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/enabled', 'true', {
            applyToLaunchedInstances: true,
        });
        // Add tag for the AZ
        if (options.subnets && options.subnets.subnets) {
            aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/node-template/label/topology.kubernetes.io/zone', options.subnets.subnets[0].availabilityZone, {
                applyToLaunchedInstances: true,
            });
        }
        // Add tag for the lifecycle type (spot or on-demand)
        aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/node-template/label/eks.amazonaws.com/capacityType', (options.capacityType == aws_eks_1.CapacityType.SPOT) ? 'SPOT' : 'ON_DEMAND', {
            applyToLaunchedInstances: true,
        });
        // Iterate over labels and add appropriate tags
        if (options.labels) {
            for (const [key, value] of Object.entries(options.labels)) {
                aws_cdk_lib_1.Tags.of(nodegroup).add(`k8s.io/cluster-autoscaler/node-template/label/${key}`, value, {
                    applyToLaunchedInstances: true,
                });
                new aws_cdk_lib_1.CustomResource(this, `${nodegroupId}Label${key}`, {
                    serviceToken: this.nodegroupAsgTagsProviderServiceToken,
                    properties: {
                        nodegroupName: options.nodegroupName,
                        tagKey: `k8s.io/cluster-autoscaler/node-template/label/${key}`,
                        tagValue: value,
                    },
                }).node.addDependency(nodegroup);
            }
        }
        // Iterate over taints and add appropriate tags
        if (options.taints) {
            options.taints.forEach((taint) => {
                aws_cdk_lib_1.Tags.of(nodegroup).add(`k8s.io/cluster-autoscaler/node-template/taint/${taint.key}`, `${taint.value}:${taint.effect}`, {
                    applyToLaunchedInstances: true,
                });
                new aws_cdk_lib_1.CustomResource(this, `${nodegroupId}Taint${taint.key}`, {
                    serviceToken: this.nodegroupAsgTagsProviderServiceToken,
                    properties: {
                        nodegroupName: options.nodegroupName,
                        tagKey: `k8s.io/cluster-autoscaler/node-template/taint/${taint.key}`,
                        tagValue: `${taint.value}:${taint.effect}`,
                    },
                }).node.addDependency(nodegroup);
            });
        }
        return nodegroup;
    }
    /**
     * Create and configure a new Amazon IAM Role usable as an execution role.
     * This method makes the created role assumed by the Amazon EKS cluster Open ID Connect provider.
     * @param {Construct} scope of the IAM role
     * @param {string} id of the CDK resource to be created, it should be unique across the stack
     * @param {IManagedPolicy} policy the execution policy to attach to the role
     * @param {string} namespace The namespace from which the role is going to be used. MUST be the same as the namespace of the Virtual Cluster from which the job is submitted
     * @param {string} name Name to use for the role, required and is used to scope the iam role
     */
    createExecutionRole(scope, id, policy, namespace, name) {
        const stack = aws_cdk_lib_1.Stack.of(this);
        let irsaConditionkey = new aws_cdk_lib_1.CfnJson(this, `${id}irsaConditionkey'`, {
            value: {
                [`${this.eksCluster.openIdConnectProvider.openIdConnectProviderIssuer}:sub`]: 'system:serviceaccount:' + namespace + ':emr-containers-sa-*-*-' + aws_cdk_lib_1.Aws.ACCOUNT_ID.toString() + '-' + SimpleBase.base36.encode(name),
            },
        });
        // Create an execution role assumable by EKS OIDC provider
        return new aws_iam_1.Role(scope, `${id}ExecutionRole`, {
            assumedBy: new aws_iam_1.FederatedPrincipal(this.eksCluster.openIdConnectProvider.openIdConnectProviderArn, {
                StringLike: irsaConditionkey,
            }, 'sts:AssumeRoleWithWebIdentity'),
            roleName: name,
            managedPolicies: [policy],
            inlinePolicies: {
                PodTemplateAccess: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                's3:getObject',
                            ],
                            resources: [
                                stack.formatArn({
                                    region: '',
                                    account: '',
                                    service: 's3',
                                    resource: this.podTemplateLocation.bucketName,
                                    resourceName: `${this.podTemplateLocation.objectKey}/*`,
                                }),
                            ],
                        }),
                    ],
                }),
            },
        });
    }
    /**
     * Upload podTemplates to the Amazon S3 location used by the cluster.
     * @param {string} id the unique ID of the CDK resource
     * @param {string} filePath The local path of the yaml podTemplate files to upload
     */
    uploadPodTemplate(id, filePath) {
        new aws_s3_deployment_1.BucketDeployment(this, `${id}AssetDeployment`, {
            destinationBucket: this.assetBucket,
            destinationKeyPrefix: this.podTemplateLocation.objectKey,
            sources: [aws_s3_deployment_1.Source.asset(filePath)],
            role: this.assetUploadBucketRole,
        });
    }
    /**
     * Creates a new Amazon EMR on EKS job template based on the props passed
     * @param {Construct} scope the scope of the stack where job template is created
     * @param {string} id the CDK id for job template resource
     * @param {EmrEksJobTemplateDefinition} options the EmrManagedEndpointOptions to configure the Amazon EMR managed endpoint
     */
    addJobTemplate(scope, id, options) {
        // Create custom resource to execute the create job template boto3 call
        const cr = new aws_cdk_lib_1.CustomResource(scope, id, {
            serviceToken: this.jobTemplateProviderToken,
            properties: {
                name: options.name,
                jobTemplateData: options.jobTemplateData,
            },
        });
        cr.node.addDependency(this.eksCluster);
        return cr;
    }
    /**
     * Apply the provided manifest and add the CDK dependency on EKS cluster
     * @param {string} id the unique ID of the CDK resource
     * @param {any} manifest The manifest to apply.
     * You can use the Utils class that offers method to read yaml file and load it as a manifest
     */
    addKarpenterProvisioner(id, manifest) {
        if (!this.isKarpenter) {
            throw new Error(`You can\'t use this method when the autoscaler is set to ${Autoscaler.KARPENTER}`);
        }
        let manifestApply = this.eksCluster.addManifest(id, ...manifest);
        if (this.karpenterChart) {
            manifestApply.node.addDependency(this.karpenterChart);
        }
        return manifestApply;
    }
}
exports.EmrEksCluster = EmrEksCluster;
_a = JSII_RTTI_SYMBOL_1;
EmrEksCluster[_a] = { fqn: "aws-analytics-reference-architecture.EmrEksCluster", version: "2.10.1" };
EmrEksCluster.DEFAULT_EMR_VERSION = EmrVersion.V6_8;
EmrEksCluster.DEFAULT_EKS_VERSION = aws_eks_1.KubernetesVersion.V1_22;
EmrEksCluster.DEFAULT_CLUSTER_NAME = 'data-platform';
EmrEksCluster.DEFAULT_KARPENTER_VERSION = 'v0.20.0';
//# sourceMappingURL=data:application/json;base64,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