"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrEksImageBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const constructs_1 = require("constructs");
const ara_bucket_1 = require("../ara-bucket");
const utils_1 = require("../utils");
const docker_builder_util_1 = require("./docker-builder-util");
/**
 * A CDK construct to create build and publish EMR on EKS custom image
 * The construct will create an ECR repository to publish the images
 * It provide a method {@link publishImage} to build a docker file and publish it to the ECR repository
 *
 * Resources deployed:
 *
 * * Multiple Session Policies that are used to map an EMR Studio user or group to a set of resources they are allowed to access. These resources are:
 *   * ECR Repository
 *   * Codebuild project
 *   * A custom resource to build and publish a custom EMR on EKS image
 *
 *
 * Usage example:
 *
 * ```typescript
 *
 * const app = new App();
 *
 * const account = process.env.CDK_DEFAULT_ACCOUNT;
 * const region = process.env.CDK_DEFAULT_REGION;
 *
 * const stack = new Stack(app, 'EmrEksImageBuilderStack', {
 * env: { account: account, region: region },
 * });
 *
 * const publish = new EmrEksImageBuilder(stack, 'EmrEksImageBuilder', {
 *  repositoryName: 'my-repo',
 *  ecrRemovalPolicy: RemovalPolicy.RETAIN
 * });
 *
 * publish.publishImage('PATH-TO-DOCKER-FILE-FOLDER', 'v4');
 *
 * ```
 */
class EmrEksImageBuilder extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.assetBucket = ara_bucket_1.AraBucket.getOrCreate(this, { bucketName: `${utils_1.Utils.stringSanitizer(aws_cdk_lib_1.Stack.of(this).stackName)}-ara-docker-assets`, encryption: aws_s3_1.BucketEncryption.KMS_MANAGED });
        let codeBuildRole = new aws_iam_1.Role(this, 'codebuildarn', {
            assumedBy: new aws_iam_1.ServicePrincipal('codebuild.amazonaws.com'),
        });
        const ecrRepo = new aws_ecr_1.Repository(this, `ecr-${props.repositoryName}`, {
            repositoryName: props.repositoryName,
            removalPolicy: props.ecrRemovalPolicy ? props.ecrRemovalPolicy : aws_cdk_lib_1.RemovalPolicy.RETAIN,
            imageScanOnPush: true
        });
        this.ecrURI = ecrRepo.repositoryUri;
        this.ecrName = props.repositoryName;
        let dockerImageAccount = docker_builder_util_1.emrOnEksImageMap.get(aws_cdk_lib_1.Stack.of(this).region);
        if (dockerImageAccount === undefined)
            throw new Error('Docker Image is not available in the selected region');
        let commands = [
            'echo logging into docker',
            `aws ecr get-login-password --region ${aws_cdk_lib_1.Aws.REGION} | docker login --username AWS --password-stdin ${dockerImageAccount}.dkr.ecr.${aws_cdk_lib_1.Aws.REGION}.amazonaws.com`,
            'echo Build start',
            'echo $DOCKER_FILE_S3_PATH',
            'aws s3 cp $DOCKER_FILE_S3_PATH Dockerfile',
            'docker build -t local .',
            'docker logout',
            'echo $ecrURI',
            `aws ecr get-login-password --region ${aws_cdk_lib_1.Aws.REGION} | docker login --username AWS --password-stdin ${this.ecrURI}`,
            'docker tag local $ecrURI:$tag',
            'docker push $ecrURI:$tag',
            'docker logout'
        ];
        const codebuildProject = new aws_codebuild_1.Project(this, `DockerImageDeployProject-${props.repositoryName}`, {
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    build: {
                        commands: commands,
                    },
                },
            }),
            environment: {
                privileged: true,
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
                computeType: aws_codebuild_1.ComputeType.SMALL
            },
            role: codeBuildRole,
        });
        ecrRepo.grantPullPush(codeBuildRole);
        this.assetBucket.grantRead(codeBuildRole);
        codeBuildRole.addToPolicy(new aws_iam_1.PolicyStatement({
            resources: [`arn:aws:ecr:${aws_cdk_lib_1.Aws.REGION}:${dockerImageAccount}:repository/*`],
            actions: [
                'ecr:BatchGetImage',
                'ecr:GetAuthorizationToken',
                'ecr:BatchCheckLayerAvailability',
                'ecr:GetDownloadUrlForLayer',
            ]
        }));
        this.codebuildProjectName = codebuildProject.projectName;
        this.dockerBuildPublishCrToken = docker_builder_util_1.EmrEksImageBuilderCRProviderSetup(this, codebuildProject.projectArn);
    }
    /**
     * A method to build and publish the custom image from a Dockerfile
     * The method invoke the custom resource deployed by the construct
     * and publish the **URI** of the published custom image as Cloudformation output
     * @param {string} dockerfilePath Path to the folder for Dockerfile
     * @param {string} tag The tag used to publish to the ECR repository
     */
    publishImage(dockerfilePath, tag) {
        new aws_s3_deployment_1.BucketDeployment(this, `DockerfileAssetDeployment-${tag}`, {
            destinationBucket: this.assetBucket,
            destinationKeyPrefix: `${this.ecrName}/${tag}`,
            sources: [aws_s3_deployment_1.Source.asset(dockerfilePath)],
        });
        const containerImage = new aws_cdk_lib_1.CustomResource(this, `EmrEksImageBuild-${tag}`, {
            serviceToken: this.dockerBuildPublishCrToken,
            properties: {
                s3Path: `s3://${this.assetBucket.bucketName}/${this.ecrName}/${tag}/Dockerfile`,
                tag: tag,
                ecrURI: this.ecrURI,
                codebuildProjectName: this.codebuildProjectName,
                ecrName: this.ecrName,
            },
        });
        new aws_cdk_lib_1.CfnOutput(this, `URI-${tag}`, {
            value: containerImage.getAttString('ContainerUri'),
        });
    }
}
exports.EmrEksImageBuilder = EmrEksImageBuilder;
_a = JSII_RTTI_SYMBOL_1;
EmrEksImageBuilder[_a] = { fqn: "aws-analytics-reference-architecture.EmrEksImageBuilder", version: "2.10.1" };
//# sourceMappingURL=data:application/json;base64,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