"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareRdsTarget = exports.prepareAuroraTarget = exports.prepareRedshiftTarget = exports.prepareDdbTarget = exports.prepareS3Target = void 0;
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
/**
 * Set up permissions to write to S3 target and params required for WriteInBatch Lambda
 *
 * @param S3Props
 * @param dataBucketName
 * @param dataObjectKey
 */
function prepareS3Target(S3Props, dataBucketName, dataObjectKey) {
    // Add policy to allow access to bucket
    const policy = new aws_iam_1.PolicyStatement({
        actions: [
            's3:GetObject',
            's3:ListBucket',
        ],
        resources: [
            `arn:aws:s3:::${dataBucketName}/${dataObjectKey}/*`,
            `arn:aws:s3:::${dataBucketName}`,
        ],
    });
    // Add params to allow data to be output to S3 target
    const taskInputParams = {
        sinkPath: S3Props.sinkBucket.s3UrlForObject(S3Props.sinkObjectKey),
        outputFileMaxSizeInBytes: S3Props.outputFileMaxSizeInBytes,
    };
    return { policy, taskInputParams };
}
exports.prepareS3Target = prepareS3Target;
/**
 * Set up permissions to write to DynamoDB target and params required for WriteInBatch Lambda
 *
 * @param ddbProps
 */
function prepareDdbTarget(ddbProps) {
    // Add policy to allow access to table
    const policy = new aws_iam_1.PolicyStatement({
        actions: [
            'dynamodb:DescribeTable',
            'dynamodb:PutItem',
            'dynamodb:BatchWriteItem',
        ],
        resources: [ddbProps.table.tableArn],
    });
    // Add params to allow data to be output to DynamoDB target
    const taskInputParams = {
        ddbTableName: ddbProps.table.tableName,
    };
    return { policy, taskInputParams };
}
exports.prepareDdbTarget = prepareDdbTarget;
/**
 * Set up permissions to write to Redshift target and params required for WriteInBatch Lambda
 *
 * @param redshiftProps
 */
function prepareRedshiftTarget(redshiftProps) {
    // Add policy to allow access to table
    const policy = new aws_iam_1.PolicyStatement({
        actions: [
            'secretsmanager:GetSecretValue',
        ],
        resources: [redshiftProps.connection],
    });
    // Add params to allow data to be output to Redshift target
    const taskInputParams = {
        redshiftTableName: redshiftProps.table,
        redshiftConnection: redshiftProps.connection,
        redshiftSchema: redshiftProps.schema,
    };
    return { policy, taskInputParams };
}
exports.prepareRedshiftTarget = prepareRedshiftTarget;
/**
 * Set up permissions to write to S3 target and params required for WriteInBatch Lambda
 *
 * @param auroraProps
 */
function prepareAuroraTarget(auroraProps) {
    // Add policy to allow access to table
    const policy = new aws_iam_1.PolicyStatement({
        actions: [
            'secretsmanager:GetSecretValue',
        ],
        resources: [auroraProps.connection],
    });
    // Add params to allow data to be output to Aurora MySQL target
    const taskInputParams = {
        auroraTableName: auroraProps.table,
        auroraConnection: auroraProps.connection,
        auroraSchema: auroraProps.schema,
        dbType: auroraProps.type,
    };
    return { policy, taskInputParams };
}
exports.prepareAuroraTarget = prepareAuroraTarget;
/**
 * Set up permissions to write to RDS target and params required for WriteInBatch Lambda
 *
 * @param rdsProps
 */
function prepareRdsTarget(rdsProps) {
    // Add policy to allow access to table
    const policy = new aws_iam_1.PolicyStatement({
        actions: [
            'secretsmanager:GetSecretValue',
        ],
        resources: [rdsProps.connection],
    });
    // Add params to allow data to be output to RDS target
    const taskInputParams = {
        rdsTableName: rdsProps.table,
        rdsConnection: rdsProps.connection,
        rdsSchema: rdsProps.schema,
        dbType: rdsProps.type,
    };
    return { policy, taskInputParams };
}
exports.prepareRdsTarget = prepareRdsTarget;
//# sourceMappingURL=data:application/json;base64,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