"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkDeployer = exports.DeploymentType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const cdk = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const singleton_kms_key_1 = require("../singleton-kms-key");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const utils_1 = require("../utils");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const cdk_deployer_build_1 = require("./cdk-deployer-build");
/**
 * @deprecated The enum should not be used. Use https://github.com/flochaz/cdk-standalone-deployer
 */
var DeploymentType;
(function (DeploymentType) {
    DeploymentType[DeploymentType["WORKSHOP_STUDIO"] = 0] = "WORKSHOP_STUDIO";
    DeploymentType[DeploymentType["CLICK_TO_DEPLOY"] = 1] = "CLICK_TO_DEPLOY";
})(DeploymentType = exports.DeploymentType || (exports.DeploymentType = {}));
/**
 * A custom CDK Stack that can be synthetized as a CloudFormation Stack to deploy a CDK application hosted on GitHub or on S3 as a Zip file.
 * This stack is self contained and can be one-click deployed to any AWS account.
 * It can be used for AWS workshop or AWS blog examples deployment when CDK is not supported/desired.
 * The stack supports passing the CDK application stack name to deploy (in case there are multiple stacks in the CDK app) and CDK parameters.
 *
 * It contains the necessary resources to synchronously deploy a CDK application from a GitHub repository:
 *  * A CodeBuild project to effectively deploy the CDK application
 *  * A StartBuild custom resource to synchronously triggers the build using a callback pattern based on Event Bridge
 *  * The necessary roles and permissions
 *
 * The StartBuild CFN custom resource is using the callback pattern to wait for the build completion:
 *  1. a Lambda function starts the build but doesn't return any value to the CFN callback URL. Instead, the callback URL is passed to the build project.
 *  2. the completion of the build triggers an Event and a second Lambda function which checks the result of the build and send information to the CFN callback URL
 *
 *  * Usage example:
 * ```typescript
 * new CdkDeployer(AwsNativeRefArchApp, 'AwsNativeRefArchDeployer', {
 *  githubRepository: 'aws-samples/aws-analytics-reference-architecture',
 *  cdkAppLocation: 'refarch/aws-native',
 *  cdkParameters: {
 *    QuickSightUsername: {
 *      default: 'myuser',
 *      type: 'String',
 *    },
 *    QuickSightIdentityRegion: {
 *      default: 'us-east-1',
 *      type: 'String',
 *    },
 *  },
 * });
 * ```
 */
class CdkDeployer extends cdk.Stack {
    /**
     * Constructs a new instance of the TrackedConstruct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {CdkDeployerProps} props the CdkDeployer [properties]{@link CdkDeployerProps}
     */
    constructor(scope, props) {
        super(scope, 'CDKDeployer', {
            // Change the Stack Synthetizer to remove the CFN parameters for the CDK version
            synthesizer: new aws_cdk_lib_1.DefaultStackSynthesizer({
                generateBootstrapVersionRule: false,
            }),
        });
        // Add parameters to the stack so it can be transfered to the CDK application
        var parameters = '';
        for (let name in props.cdkParameters) {
            let param = props.cdkParameters[name];
            let cfnParam = new cdk.CfnParameter(this, name, param);
            parameters = parameters.concat(` -c ${name}=${cfnParam.value}`);
        }
        // Name of the stack to deploy in codebuild
        const stackName = props.stackName ? props.stackName : '';
        // Role used by the CodeBuild project
        const buildRole = new aws_iam_1.Role(this, 'CodeBuildRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('codebuild.amazonaws.com'),
        });
        // We need the CDK execution role so the CodeBuild role can assume it for CDK deployment
        const cdkDeployRole = utils_1.Utils.getCdkDeployRole(this, 'CdkDeployRole');
        const cdkPublishRole = utils_1.Utils.getCdkFilePublishRole(this, 'CdkPublishRole');
        buildRole.addManagedPolicy(new aws_iam_1.ManagedPolicy(this, 'CdkBuildPolicy', {
            statements: [
                new aws_iam_1.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        'kms:CreateKey',
                        'kms:DisableKey',
                        'kms:EnableKeyRotation',
                        'kms:TagResource',
                        'kms:DescribeKey',
                        'kms:ScheduleKeyDeletion',
                        'kms:CreateAlias',
                        'kms:DeleteAlias',
                        'kms:CreateGrant',
                        'kms:DescribeKey',
                        'kms:RetireGrant'
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        's3:CreateBucket',
                        's3:PutBucketAcl',
                        's3:PutEncryptionConfiguration',
                        's3:PutBucketPublicAccessBlock',
                        's3:PutBucketVersioning',
                        's3:DeleteBucket',
                        's3:PutBucketPolicy',
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: [
                        `arn:aws:cloudformation:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:stack/CDKToolkit*`
                    ],
                    actions: [
                        'cloudformation:DescribeStacks',
                        'cloudformation:DeleteStack',
                        'cloudformation:DeleteChangeSet',
                        'cloudformation:CreateChangeSet',
                        'cloudformation:DescribeChangeSet',
                        'cloudformation:ExecuteChangeSet',
                        'cloudformation:DescribeStackEvents',
                        'cloudformation:GetTemplate',
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: [
                        cdkDeployRole.roleArn,
                        cdkPublishRole.roleArn,
                    ],
                    actions: [
                        'sts:AssumeRole',
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: [
                        `arn:aws:ssm:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:parameter/cdk-bootstrap/*/*`,
                    ],
                    actions: [
                        'ssm:PutParameter',
                        'ssm:GetParameters',
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: [
                        `arn:aws:ecr:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:repository/cdk*`,
                    ],
                    actions: [
                        'ecr:SetRepositoryPolicy',
                        'ecr:GetLifecyclePolicy',
                        'ecr:PutImageTagMutability',
                        'ecr:DescribeRepositories',
                        'ecr:ListTagsForResource',
                        'ecr:PutImageScanningConfiguration',
                        'ecr:CreateRepository',
                        'ecr:PutLifecyclePolicy',
                        'ecr:SetRepositoryPolicy',
                        'ecr:DeleteRepository',
                        'ecr:TagResource',
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: [
                        `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk*`,
                    ],
                    actions: [
                        'iam:GetRole',
                        'iam:CreateRole',
                        'iam:TagRole',
                        'iam:DeleteRole',
                        'iam:AttachRolePolicy',
                        'iam:DetachRolePolicy',
                        'iam:GetRolePolicy',
                        'iam:PutRolePolicy',
                        'iam:DeleteRolePolicy',
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    resources: [
                        `arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/codebuild/*`,
                    ],
                    actions: [
                        'logs:PutLogEvents',
                    ],
                }),
            ]
        }));
        let source;
        if (props.deploymentType === DeploymentType.WORKSHOP_STUDIO) {
            const cdkAppSourceCodeBucketName = new aws_cdk_lib_1.CfnParameter(this, 'CDKAppSourceCodeBucketName', {
                type: 'String',
            });
            const cdkAppSourceCodeBucketPrefix = new aws_cdk_lib_1.CfnParameter(this, 'CDKAppSourceCodeBucketPrefix', {
                type: 'String',
            });
            source = aws_codebuild_1.Source.s3({
                bucket: aws_s3_1.Bucket.fromBucketName(this, 'CdkAppBucket', cdkAppSourceCodeBucketName.valueAsString),
                path: `${cdkAppSourceCodeBucketPrefix.valueAsString}cdk_app.zip`,
            });
        }
        else {
            if (props.githubRepository) {
                source = aws_codebuild_1.Source.gitHub({
                    owner: props.githubRepository.split('/')[0],
                    repo: props.githubRepository.split('/')[1],
                    branchOrRef: props.gitBranch ? props.gitBranch : undefined,
                    reportBuildStatus: true,
                });
            }
            else if (props.s3Repository) {
                source = aws_codebuild_1.Source.s3({
                    bucket: aws_s3_1.Bucket.fromBucketName(this, 'CdkAppBucket', props.s3Repository.bucketName),
                    path: props.s3Repository.objectKey,
                });
            }
            else {
                throw new Error('githubRepository or s3Repository is required for CLICK_TO_DEPLOY deployment type');
            }
        }
        const codeBuildProject = new aws_codebuild_1.Project(this, 'CodeBuildProject', {
            source,
            encryptionKey: singleton_kms_key_1.SingletonKey.getOrCreate(this, 'DefaultKmsKey'),
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
                computeType: aws_codebuild_1.ComputeType.SMALL,
                environmentVariables: {
                    PARAMETERS: {
                        value: parameters,
                    },
                    STACKNAME: {
                        value: stackName,
                    },
                    CDK_APP_LOCATION: {
                        value: props.cdkAppLocation ? props.cdkAppLocation : '',
                    },
                },
            },
            role: buildRole,
        });
        if (props.deploymentType === DeploymentType.WORKSHOP_STUDIO) {
            codeBuildProject.node.defaultChild.addPropertyOverride('EncryptionKey', 'alias/aws/s3');
        }
        const startBuildRole = new aws_iam_1.Role(this, 'StartBuildRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')],
            inlinePolicies: {
                StartBuild: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            resources: [codeBuildProject.projectArn],
                            actions: ['codebuild:StartBuild'],
                        }),
                    ],
                }),
            },
        });
        const startBuildFunction = new aws_lambda_1.Function(this, 'StartBuildFunction', {
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            code: aws_lambda_1.Code.fromInline(cdk_deployer_build_1.startBuild(props.deployBuildSpec, props.destroyBuildSpec)),
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.seconds(60),
            role: startBuildRole,
        });
        const reportBuildRole = new aws_iam_1.Role(this, 'ReportBuildRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')],
            inlinePolicies: {
                ReportBuild: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            resources: [codeBuildProject.projectArn],
                            actions: ['codebuild:BatchGetBuilds', 'codebuild:ListBuildsForProject'],
                        }),
                    ],
                }),
            },
        });
        const reportBuildFunction = new aws_lambda_1.Function(this, 'ReportBuildFunction', {
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            code: aws_lambda_1.Code.fromInline(cdk_deployer_build_1.reportBuild),
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.seconds(60),
            role: reportBuildRole,
        });
        const buildCompleteRule = new aws_events_1.Rule(this, 'BuildCompleteEvent', {
            eventPattern: {
                source: ['aws.codebuild'],
                detailType: ['CodeBuild Build State Change'],
                detail: {
                    'build-status': ['SUCCEEDED', 'FAILED', 'STOPPED'],
                    'project-name': [codeBuildProject.projectName],
                },
            },
            targets: [new aws_events_targets_1.LambdaFunction(reportBuildFunction)],
        });
        const buildTrigger = new aws_cdk_lib_1.CustomResource(this, 'CodeBuildTriggerCustomResource', {
            serviceToken: startBuildFunction.functionArn,
            properties: {
                ProjectName: codeBuildProject.projectName,
                BuildRoleArn: buildRole.roleArn,
                Parameters: parameters,
                StackName: stackName,
            },
        });
        buildTrigger.node.addDependency(buildCompleteRule);
        buildTrigger.node.addDependency(buildRole);
        this.deployResult = buildTrigger.getAttString('BuildStatus');
    }
}
exports.CdkDeployer = CdkDeployer;
_a = JSII_RTTI_SYMBOL_1;
CdkDeployer[_a] = { fqn: "aws-analytics-reference-architecture.CdkDeployer", version: "2.10.1" };
//# sourceMappingURL=data:application/json;base64,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