"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AraBucket = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const singleton_kms_key_1 = require("./singleton-kms-key");
/**
* An Amazon S3 Bucket following best practices for the AWS Analytics Reference Architecture.
* The bucket name is mandatory and is used as the CDK id.
* The bucket name is postfixed with the AWS account ID and the AWS region.
*
* The bucket has the following default properties:
*  * the encryption mode is KMS managed by AWS
*  * if the encryption mode is KMS customer managed, the encryption key is a default and unique KMS key for ARA
*  * the KMS key is used as a bucket key
*  * the SSL is enforced
*  * the objects are automatically deleted when the bucket is deleted
*  * the access are logged in a default and unique S3 bucket for ARA if serverAccessLogsPrefix is provided
*  * the access are not logged if serverAccessLogsPrefix is  not provided
*  * the public access is blocked and no bucket policy or object permission can grant public access
*
* All standard S3 Bucket properties can be provided to not use the defaults.
* Usage example:
* ```typescript
* import * as cdk from 'aws-cdk-lib';
* import { AraBucket } from 'aws-analytics-reference-architecture';
*
* const exampleApp = new cdk.App();
* const stack = new cdk.Stack(exampleApp, 'AraBucketStack');
*
* new AraBucket(stack, {
*  bucketName: 'test-bucket',
*  serverAccessLogsPrefix: 'test-bucket',
* });
* ```
*/
class AraBucket extends aws_s3_1.Bucket {
    /**
    * Constructs a new instance of the AraBucket class
    * @param {Construct} scope the Scope of the CDK Construct
    * @param {AraBucketProps} props the AraBucketProps [properties]{@link AraBucketProps}
    * @access private
    */
    constructor(scope, props) {
        if (scope.node.tryGetContext('@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy') === true) {
            throw new Error("Using @aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy is not supported please switch it to false in your cdk.json");
        }
        var serverAccessLogsBucket = undefined;
        if (props.serverAccessLogsPrefix) {
            serverAccessLogsBucket = props.serverAccessLogsBucket || AraBucket.getOrCreate(scope, {
                bucketName: 's3-access-logs',
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_PREFERRED,
            });
        }
        // If using KMS encryption then use a customer managed key, if not set the key to undefined
        let bucketEncryptionKey = aws_s3_1.BucketEncryption.KMS == props.encryption ? props.encryptionKey || singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey') : undefined;
        // If the bucket is for s3 access logs, we remove the bucketname to ensure uniqueness across stacks
        let bucketName = (props.bucketName == 's3-access-logs') ? undefined : `${props.bucketName}-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`;
        // set the right default parameters in the S3 bucket props
        const bucketProps = {
            ...props,
            ...{
                bucketName: bucketName,
                encryption: props.encryption ? props.encryption : aws_s3_1.BucketEncryption.KMS_MANAGED,
                encryptionKey: bucketEncryptionKey,
                bucketKeyEnabled: aws_s3_1.BucketEncryption.KMS == props.encryption ? true : false,
                enforceSSL: props.enforceSSL || true,
                removalPolicy: props.removalPolicy || aws_cdk_lib_1.RemovalPolicy.DESTROY,
                autoDeleteObjects: props.autoDeleteObjects || true,
                serverAccessLogsBucket: serverAccessLogsBucket,
                serverAccessLogsPrefix: props.serverAccessLogsPrefix,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                lifecycleRules: props.lifecycleRules || [{ abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(1) }],
            },
        };
        // build the S3 bucket
        super(scope, props.bucketName, bucketProps);
        // Used to force the cleaning of the access logs bucket after deletion of this 
        if (serverAccessLogsBucket) {
            this.node.addDependency(serverAccessLogsBucket);
        }
    }
    /**
    * Get the Amazon S3 Bucket from the AWS CDK Stack based on the provided name.
    * If no bucket exists, it creates a new one based on the provided properties.
    */
    static getOrCreate(scope, props) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = `${props.bucketName}`;
        const stackBucket = stack.node.tryFindChild(id) ?? (stack.nestedStackParent ? stack.nestedStackParent.node.tryFindChild(id) : undefined);
        return stackBucket || new AraBucket(stack, props);
    }
}
exports.AraBucket = AraBucket;
_a = JSII_RTTI_SYMBOL_1;
AraBucket[_a] = { fqn: "aws-analytics-reference-architecture.AraBucket", version: "2.10.1" };
//# sourceMappingURL=data:application/json;base64,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