import os
from openai import OpenAI

class ExplainabilityGuide:
    """
    Explainability (XAI) engine:
    - Decision explanations
    - Transparency summaries
    - Justification logs
    """

    def __init__(self, api_key=None, model_name="gpt-4o-mini"):
        self.api_key = api_key or os.getenv("OPENAI_API_KEY")
        if not self.api_key:
            raise ValueError("OpenAI API key missing. Provide api_key= or set OPENAI_API_KEY.")
        self.client = OpenAI(api_key=self.api_key)
        self.model_name = model_name

    # ------------------
    # Primary Functions
    # ------------------

    def explain_decision(self, decision, context=None):
        prompt = f"""
        Provide a clear, human-readable explanation for this AI decision:

        Decision: {decision}
        Context: {context}
        """

        response = self.client.chat.completions.create(
            model=self.model_name,
            messages=[{"role": "user", "content": prompt}]
        )
        return response.choices[0].message.content

    def transparency_summary(self, metadata):
        summary = f"""
        MODEL TRANSPARENCY SUMMARY
        ---------------------------
        Name: {metadata.get('model_name')}
        Purpose: {metadata.get('purpose')}
        Training Data: {metadata.get('training_data')}
        Version: {metadata.get('version')}
        Known Risks: {metadata.get('risks')}
        """

        return "\n".join(line.strip() for line in summary.split("\n") if line.strip())
