#!/usr/bin/env python3
import json
import os
from guide_framework.guide import GUIDEAuditor

def main():
    print("🌐 GUIDE AI Ethics Assessment Ecosystem")
    print("=" * 50)
    
    # Check if dependencies are installed
    try:
        import pandas as pd
        import numpy as np
        from colorama import init, Fore, Back, Style
        init()  # Initialize colorama
    except ImportError as e:
        print(f"❌ Missing dependency: {e}")
        print("Please run: pip install -r requirements.txt")
        return
    
    # Create output directory
    os.makedirs("assessments", exist_ok=True)
    
    # Run assessment
    auditor = GUIDEAuditor()
    report = auditor.run_complete_assessment()
    
    # Save detailed report
    report_file = f"assessments/guide_assessment_{report['assessment_summary']['situation']}.json"
    with open(report_file, 'w') as f:
        json.dump(report, f, indent=2)
    
    # Display summary
    display_summary(report)
    
    print(f"\n📊 Detailed report saved to: {report_file}")
    print("🎯 Use this report to guide your AI ethics improvement efforts!")

def display_summary(report):
    from colorama import Fore, Back, Style
    
    print(f"\n{Fore.CYAN}{'='*60}{Style.RESET_ALL}")
    print(f"{Fore.CYAN}📋 GUIDE ASSESSMENT SUMMARY{Style.RESET_ALL}")
    print(f"{Fore.CYAN}{'='*60}{Style.RESET_ALL}")
    
    summary = report['assessment_summary']
    
    # Color code overall maturity
    maturity_color = {
        'Advanced': Fore.GREEN,
        'Established': Fore.BLUE, 
        'Developing': Fore.YELLOW,
        'Initial Stage': Fore.RED,
        'Not Implemented': Fore.RED
    }
    
    print(f"\nOverall Ethical Maturity: {maturity_color[summary['overall_maturity']]}{summary['overall_maturity']}{Style.RESET_ALL}")
    print(f"Overall Score: {summary['overall_score']}/5.0")
    print(f"Priority Level: {summary['priority_level']}")
    
    print(f"\n{Fore.WHITE}Principle Scores:{Style.RESET_ALL}")
    print(f"{Fore.WHITE}{'-' * 40}{Style.RESET_ALL}")
    
    principle_names = {
        "G": f"{Fore.BLUE}Governance{Style.RESET_ALL}",
        "U": f"{Fore.GREEN}Understanding{Style.RESET_ALL}", 
        "I": f"{Fore.YELLOW}Integrity{Style.RESET_ALL}",
        "D": f"{Fore.MAGENTA}Disclosure{Style.RESET_ALL}",
        "E": f"{Fore.CYAN}Equity{Style.RESET_ALL}"
    }
    
    for principle, data in report['principle_scores'].items():
        score_color = Fore.GREEN if data['score'] >= 4.0 else Fore.YELLOW if data['score'] >= 3.0 else Fore.RED
        print(f"{principle_names[principle]}: {score_color}{data['score']}/5.0{Style.RESET_ALL} ({data['maturity']})")
    
    print(f"\n{Fore.WHITE}🚀 RECOMMENDED ACTIONS:{Style.RESET_ALL}")
    print(f"{Fore.WHITE}{'-' * 40}{Style.RESET_ALL}")
    
    action_plan = report['action_plan']
    
    if action_plan['immediate_actions']:
        print(f"\n{Fore.RED}🔴 IMMEDIATE ACTIONS (Critical):{Style.RESET_ALL}")
        for action in action_plan['immediate_actions']:
            print(f"  • {action}")
    
    if action_plan['short_term_goals']:
        print(f"\n{Fore.YELLOW}🟡 SHORT-TERM GOALS (3-6 months):{Style.RESET_ALL}")
        for action in action_plan['short_term_goals']:
            print(f"  • {action}")
    
    if action_plan['long_term_improvements']:
        print(f"\n{Fore.GREEN}🟢 LONG-TERM IMPROVEMENTS (6+ months):{Style.RESET_ALL}")
        for action in action_plan['long_term_improvements']:
            print(f"  • {action}")
    
    print(f"\n{Fore.CYAN}{'='*60}{Style.RESET_ALL}")
    print(f"{Fore.CYAN}BUILD AI THAT EARNS TRUST THROUGH ETHICAL EXCELLENCE{Style.RESET_ALL}")
    print(f"{Fore.CYAN}{'='*60}{Style.RESET_ALL}")

if __name__ == "__main__":
    main()