"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Packaging = exports.DependenciesFile = void 0;
const fs = require("fs");
const path = require("path");
var DependenciesFile;
(function (DependenciesFile) {
    DependenciesFile["PIP"] = "requirements.txt";
    DependenciesFile["POETRY"] = "poetry.lock";
    DependenciesFile["PIPENV"] = "Pipfile.lock";
    DependenciesFile["NONE"] = "";
})(DependenciesFile || (exports.DependenciesFile = DependenciesFile = {}));
class Packaging {
    /**
     * Standard packaging with `pip`.
     */
    static withPip() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIP,
        });
    }
    /**
     * Packaging with `pipenv`.
     */
    static withPipenv() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIPENV,
            // By default, pipenv creates a virtualenv in `/.local`, so we force it to create one in the package directory.
            // At the end, we remove the virtualenv to avoid creating a duplicate copy in the Lambda package.
            exportCommand: `PIPENV_VENV_IN_PROJECT=1 pipenv requirements > ${DependenciesFile.PIP} && rm -rf .venv`,
        });
    }
    /**
     * Packaging with `poetry`.
     */
    static withPoetry(props) {
        return new Packaging({
            dependenciesFile: DependenciesFile.POETRY,
            // Export dependencies with credentials available in the bundling image.
            exportCommand: [
                'poetry', 'export',
                ...props?.poetryIncludeHashes ? [] : ['--without-hashes'],
                ...props?.poetryWithoutUrls ? ['--without-urls'] : [],
                '--with-credentials',
                '--format', DependenciesFile.PIP,
                '--output', DependenciesFile.PIP,
            ].join(' '),
        });
    }
    /**
     * No dependencies or packaging.
     */
    static withNoPackaging() {
        return new Packaging({ dependenciesFile: DependenciesFile.NONE });
    }
    static fromEntry(entry, poetryIncludeHashes, poetryWithoutUrls) {
        if (fs.existsSync(path.join(entry, DependenciesFile.PIPENV))) {
            return this.withPipenv();
        }
        if (fs.existsSync(path.join(entry, DependenciesFile.POETRY))) {
            return this.withPoetry({ poetryIncludeHashes, poetryWithoutUrls });
        }
        else if (fs.existsSync(path.join(entry, DependenciesFile.PIP))) {
            return this.withPip();
        }
        else {
            return this.withNoPackaging();
        }
    }
    constructor(props) {
        this.dependenciesFile = props.dependenciesFile;
        this.exportCommand = props.exportCommand;
    }
}
exports.Packaging = Packaging;
//# sourceMappingURL=data:application/json;base64,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