# Copyright 2018 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================
"""Library for converting from hdf5 to json + binary weights.

Used primarily to convert saved weights, or saved_models from their
hdf5 format to a JSON + binary weights format that the TS codebase can use.
."""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import json
import os
import tempfile

import h5py
import numpy as np
import six

from tensorflowjs import write_weights  # pylint: disable=import-error

# File name for the indexing JSON file in an artifact directory.
ARTIFACT_MODEL_JSON_FILE_NAME = 'model.json'

# JSON string keys for fields of the indexing JSON.
ARTIFACT_MODEL_TOPOLOGY_KEY = 'modelTopology'
ARTIFACT_WEIGHTS_MANIFEST_KEY = 'weightsManifest'


def normalize_weight_name(weight_name):
  """Remove suffix ":0" (if present) from weight name."""
  name = as_text(weight_name)
  if name.endswith(':0'):
    # Python TensorFlow weight names ends with the output slot, which is
    # not applicable to TensorFlow.js.
    name = name[:-2]
  return name


def as_text(bytes_or_text, encoding='utf-8'):
  if isinstance(bytes_or_text, six.text_type):
    return bytes_or_text
  elif isinstance(bytes_or_text, bytes):
    return bytes_or_text.decode(encoding)
  else:
    raise TypeError('Expected binary or unicode string, got %r' %
                    bytes_or_text)


class HDF5Converter(object):
  """Helper class to convert HDF5 format to JSON + binary weights format

  Used primarily to allow easy migration of a Python Keras trained
  and saved model to a JSON format for use in js based implementations.
  """

  def _convert_h5_group(self, group):
    """Construct a weights group entry.

    Args:
      group: The HDF5 group data, possibly nested.

    Returns:
      An array of weight groups (see `write_weights` in TensorFlow.js).
    """
    group_out = []
    if 'weight_names' in group.attrs:
      # This is a leaf node in namespace (e.g., 'Dense' in 'foo/bar/Dense').
      names = [name for name in group.attrs['weight_names']]
      if not names:
        return group_out

      names = [as_text(name) for name in names]
      weight_values = [
          np.array(group[weight_name]) for weight_name in names]
      group_out += [{
          'name': normalize_weight_name(weight_name),
          'data': weight_value
      } for (weight_name, weight_value) in zip(names, weight_values)]
    else:
      # This is *not* a leaf level in the namespace (e.g., 'foo' in
      # 'foo/bar/Dense').
      for key in group.keys():
        # Call this method recursively.
        group_out += self._convert_h5_group(group[key])

    return group_out

  def _check_version(self, h5file):
    """Check version compatiility.
    Args:
      h5file: An h5file object.
    Raises:
      ValueError: if the KerasVersion of the HDF5 file is unsupported.
    """
    keras_version = as_text(h5file.attrs['keras_version'])
    if keras_version.split('.')[0] != '2':
      raise ValueError(
          'Expected Keras version 2; got Keras version %s' % keras_version)

  def _initialize_output_dictionary(self, h5file):
    """Prepopulate required fields for all data foramts.

    Args:
      h5file: Valid h5file object.

    Returns:
      A dictionary with common fields sets, shared across formats.
    """
    out = dict()
    out['keras_version'] = as_text(h5file.attrs['keras_version'])
    out['backend'] = as_text(h5file.attrs['backend'])
    return out

  def _ensure_h5file(self, h5file):
    if not isinstance(h5file, h5py.File):
      return h5py.File(h5file)
    else:
      return h5file

  def _ensure_json_dict(self, item):
    return item if isinstance(item, dict) else json.loads(as_text(item))

  def h5_merged_saved_model_to_tfjs_format(self, h5file, split_by_layer=False):
    """Load topology & weight values from HDF5 file and convert.

    The HDF5 file is one generated by Keras' save_model method or model.save()

    N.B.:
    1) This function works only on HDF5 values from Keras version 2.
    2) This function does not perform conversion for special weights including
       ConvLSTM2D and CuDNNLSTM.

    Args:
      h5file: An instance of h5py.File, or the path to an h5py file.
      split_by_layer: (Optional) whether the weights of different layers are
        to be stored in separate weight groups (Default: `False`).

    Returns:
      (model_json, groups)
        model_json: a JSON dictionary holding topology and system metadata.
        group: an array of group_weights as defined in tfjs write_weights.

    Raises:
      ValueError: If the Keras version of the HDF5 file is not supported.
    """
    h5file = self._ensure_h5file(h5file)
    self._check_version(h5file)
    model_json = self._initialize_output_dictionary(h5file)

    model_json['model_config'] = self._ensure_json_dict(
        h5file.attrs['model_config'])
    if 'training_config' in h5file.attrs:
      model_json['training_config'] = self._ensure_json_dict(
          h5file.attrs['training_config'])

    groups = [] if split_by_layer else [[]]

    model_weights = h5file['model_weights']
    layer_names = [as_text(n) for n in model_weights]
    for layer_name in layer_names:
      layer = model_weights[layer_name]
      group = self._convert_h5_group(layer)
      if group:
        if split_by_layer:
          groups.append(group)
        else:
          groups[0] += group
    return model_json, groups

  def h5_weights_to_tfjs_format(self, h5file, split_by_layer=False):
    """Load weight values from a Keras HDF5 file and to a binary format.

    The HDF5 file is one generated by Keras' Model.save_weights() method.

    N.B.:
    1) This function works only on HDF5 values from Keras version 2.
    2) This function does not perform conversion for special weights including
       ConvLSTM2D and CuDNNLSTM.

    Args:
      h5file: An instance of h5py.File, or the path to an h5py file.
      split_by_layer: (Optional) whether the weights of different layers are
        to be stored in separate weight groups (Default: `False`).

    Returns:
      An array of group_weights as defined in tfjs write_weights.

    Raises:
      ValueError: If the Keras version of the HDF5 file is not supported
    """
    h5file = self._ensure_h5file(h5file)
    self._check_version(h5file)

    groups = [] if split_by_layer else [[]]

    # pylint: disable=not-an-iterable
    layer_names = [as_text(n) for n in h5file.attrs['layer_names']]
    # pylint: enable=not-an-iterable
    for layer_name in layer_names:
      layer = h5file[layer_name]
      group = self._convert_h5_group(layer)
      if group:
        if split_by_layer:
          groups.append(group)
        else:
          groups[0] += group
    return groups

  def write_artifacts(self,
                      topology,
                      weights,
                      output_dir,
                      quantization_dtype=None):
    """Writes weights and topology to the output_dir.

    If `topology` is Falsy (e.g., `None`), only emit weights to output_dir.

    Args:
      topology: a JSON dictionary, representing the Keras config.
      weights: an array of weight groups (as defined in tfjs write_weights).
      output_dir: the directory to hold all the contents.
      quantization_dtype: An optional numpy dtype to quantize weights to for
        compression. Only np.uint8 and np.uint16 are supported.
    """
    # TODO(cais, nielsene): This method should allow optional arguments of
    #   `write_weights.write_weights` (e.g., shard size) and forward them.
    # We write the topology after since write_weights makes no promises about
    # preserving directory contents.
    if os.path.isfile(output_dir):
      raise ValueError(
          'Path "%d" already exists as a file (not a directory).' % output_dir)

    model_json = {}

    model_json[ARTIFACT_MODEL_TOPOLOGY_KEY] = topology or None
    weights_manifest = write_weights.write_weights(
        weights, output_dir, write_manifest=False,
        quantization_dtype=quantization_dtype)
    if not isinstance(weights_manifest, list):
      weights_manifest = json.loads(weights_manifest)
    assert isinstance(weights_manifest, list)
    model_json[ARTIFACT_WEIGHTS_MANIFEST_KEY] = weights_manifest

    model_json_path = os.path.join(output_dir, ARTIFACT_MODEL_JSON_FILE_NAME)
    with open(model_json_path, 'wt') as f:
      json.dump(model_json, f)


def save_keras_model(model, artifacts_dir, quantization_dtype=None):
  r"""Save a Keras model and its weights in TensorFlow.js format.

  Args:
    model: An instance of `keras.Model`.
    artifacts_dir: The directory in which the artifacts will be saved.
      The artifacts to be saved include:
        - model.json: A JSON representing the model. It has the following
          fields:
          - 'modelTopology': A JSON object describing the topology of the model,
            along with additional information such as training. It is obtained
            through calling `model.save()`.
          - 'weightsManifest': A TensorFlow.js-format JSON manifest for the
            model's weights.
        - files containing weight values in groups, with the file name pattern
          group(\d+)-shard(\d+)of(\d+).
      If the directory does not exist, this function will attempt to create it.
    quantization_dtype: An optional numpy dtype to quantize weights to for
        compression. Only np.uint8 and np.uint16 are supported.

  Raises:
    ValueError: If `artifacts_dir` already exists as a file (not a directory).
  """
  temp_h5_path = tempfile.mktemp() + '.h5'
  model.save(temp_h5_path)
  # TODO(cais): Maybe get rid of the class HDF5Converter to simplify the code.
  converter = HDF5Converter()
  topology_json, weight_groups = (
      converter.h5_merged_saved_model_to_tfjs_format(temp_h5_path))
  if os.path.isfile(artifacts_dir):
    raise ValueError('Path "%s" already exists as a file.' % artifacts_dir)
  elif not os.path.isdir(artifacts_dir):
    os.makedirs(artifacts_dir)
  converter.write_artifacts(
      topology_json, weight_groups, artifacts_dir,
      quantization_dtype=quantization_dtype)
  os.remove(temp_h5_path)
