"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const ecs = require("aws-cdk-lib/aws-ecs");
const ec2 = require("aws-cdk-lib/aws-ec2");
const assertions_1 = require("aws-cdk-lib/assertions");
const clusterName = "custom-cluster-name";
const containerName = "custom-container-name";
const serviceName = "custom-service-name";
const familyName = "family-name";
const testCidr = "172.0.0.0/16";
test('Check for construct properties', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const construct = createFargateConstructWithNewResources(stack, publicApi);
    expect(construct.vpc).toBeDefined();
    expect(construct.service).toBeDefined();
    expect(construct.container).toBeDefined();
    expect(construct.stateMachine).toBeDefined();
    expect(construct.stateMachineLogGroup).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
});
test('Check for new service', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: serviceName,
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
});
test('Check for an existing service', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const existingVpc = defaults.getTestVpc(stack);
    const createFargateServiceResponse = defaults.CreateFargateService(stack, 'test', {
        constructVpc: existingVpc,
        clientClusterProps: {
            clusterName
        },
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clientFargateTaskDefinitionProps: {
            family: familyName
        },
        clientContainerDefinitionProps: {
            containerName
        },
        clientFargateServiceProps: {
            serviceName
        }
    });
    new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        existingFargateServiceObject: createFargateServiceResponse.service,
        existingContainerDefinitionObject: createFargateServiceResponse.containerDefinition,
        existingVpc,
        stateMachineProps: testStateMachineProps(stack)
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: serviceName,
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
});
test('Check for IAM startExecution policy', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "states:StartExecution",
                    Effect: "Allow",
                    Resource: {
                        Ref: "testconstructStateMachine3333AAA9"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testconstructtaskdefTaskRoleDefaultPolicyF34A1535",
        Roles: [
            {
                Ref: "testconstructtaskdefTaskRoleC60414C4"
            }
        ]
    });
});
test('Check for public/private VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    createFargateConstructWithNewResources(stack, publicApi);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: testCidr
    });
    template.hasResourceProperties('AWS::EC2::InternetGateway', {});
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::StepFunctions::StateMachine', 1);
    template.resourceCountIs('AWS::ECS::Service', 1);
});
test('Check for isolated VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: testCidr
    });
    defaults.expectNonexistence(stack, 'AWS::EC2::InternetGateway', {});
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::StepFunctions::StateMachine', 1);
    template.resourceCountIs('AWS::ECS::Service', 1);
});
test('Check for an existing VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    const existingVpc = defaults.getTestVpc(stack, publicApi);
    new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        existingVpc,
        stateMachineProps: testStateMachineProps(stack)
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: "172.168.0.0/16"
    });
    template.resourceCountIs("AWS::EC2::VPC", 1);
});
test('Check for custom ARN resource', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const customEnvName = "TEST_CUSTOM_ARN";
    new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
        stateMachineEnvironmentVariableName: customEnvName
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        Family: familyName,
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: customEnvName,
                        Value: {
                            Ref: "testconstructStateMachine3333AAA9"
                        }
                    }
                ],
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: containerName,
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
});
test('Check for no cloudwatch creation', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const construct = new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
        createCloudWatchAlarms: false
    });
    expect(construct.cloudwatchAlarms).not.toBeDefined();
    defaults.expectNonexistence(stack, "AWS::CloudWatch::Alarm", {
        ComparisonOperator: "GreaterThanOrEqualToThreshold",
        EvaluationPeriods: 1,
        AlarmDescription: "Alarm for the number of executions that aborted exceeded the threshold of 1. ",
        Dimensions: [
            {
                Name: "StateMachineArn",
                Value: {
                    Ref: "testconstructStateMachine3333AAA9"
                }
            }
        ],
        MetricName: "ExecutionsAborted",
        Namespace: "AWS/States",
        Period: 300,
        Statistic: "Maximum",
        Threshold: 1
    });
});
test('Check for custom log group props', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const logGroupName = "custom-log-group";
    new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
        logGroupProps: {
            logGroupName
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Logs::LogGroup", {
        LogGroupName: logGroupName
    });
});
test('Confirm that CheckVpcProps was called', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const props = {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
        existingVpc: defaults.getTestVpc(stack),
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
    };
    const app = () => {
        new lib_1.FargateToStepfunctions(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
function createFargateConstructWithNewResources(stack, publicApi) {
    return new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
    });
}
function testStateMachineProps(stack, userProps) {
    const defaultTestProp = {
        definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'farstp-test')
    };
    return defaults.consolidateProps(defaultTestProp, userProps);
}
//# sourceMappingURL=data:application/json;base64,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