/*******************************************************************************
 * The MIT License (MIT)
 *
 * Copyright (c) 2015 Camilo Sanchez (Camiloasc1) 2020 Martin Mirchev (Marti2203)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and
 * associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT
 * NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * ****************************************************************************
 */

// $antlr-format alignTrailingComments true, columnLimit 150, minEmptyLines 1, maxEmptyLinesToKeep 1, reflowComments false, useTab false
// $antlr-format allowShortRulesOnASingleLine false, allowShortBlocksOnASingleLine true, alignSemicolons hanging, alignColons hanging

parser grammar CPP14Parser;

options {
    superClass = CPP14ParserBase;
    tokenVocab = CPP14Lexer;
}

// Insert here @header for C++ parser.

/*Basic concepts*/

translationUnit
    : declarationseq? EOF
    ;

/*Expressions*/

primaryExpression
    : literal+
    | This
    | LeftParen expression RightParen
    | idExpression
    | lambdaExpression
    ;

idExpression
    : unqualifiedId
    | qualifiedId
    ;

unqualifiedId
    : Identifier
    | operatorFunctionId
    | conversionFunctionId
    | literalOperatorId
    | Tilde (className | decltypeSpecifier)
    | templateId
    ;

qualifiedId
    : nestedNameSpecifier Template? unqualifiedId
    ;

nestedNameSpecifier
    : (theTypeName | namespaceName | decltypeSpecifier)? Doublecolon
    | nestedNameSpecifier ( Identifier | Template? simpleTemplateId) Doublecolon
    ;

lambdaExpression
    : lambdaIntroducer lambdaDeclarator? compoundStatement
    ;

lambdaIntroducer
    : LeftBracket lambdaCapture? RightBracket
    ;

lambdaCapture
    : captureList
    | captureDefault (Comma captureList)?
    ;

captureDefault
    : And
    | Assign
    ;

captureList
    : capture (Comma capture)* Ellipsis?
    ;

capture
    : simpleCapture
    | initcapture
    ;

simpleCapture
    : And? Identifier
    | This
    ;

initcapture
    : And? Identifier initializer
    ;

lambdaDeclarator
    : LeftParen parameterDeclarationClause? RightParen Mutable? exceptionSpecification? attributeSpecifierSeq? trailingReturnType?
    ;

postfixExpression
    : primaryExpression
    | postfixExpression LeftBracket (expression | bracedInitList) RightBracket
    | postfixExpression LeftParen expressionList? RightParen
    | (simpleTypeSpecifier | typeNameSpecifier) (
        LeftParen expressionList? RightParen
        | bracedInitList
    )
    | postfixExpression (Dot | Arrow) (Template? idExpression | pseudoDestructorName)
    | postfixExpression (PlusPlus | MinusMinus)
    | (Dynamic_cast | Static_cast | Reinterpret_cast | Const_cast) Less theTypeId Greater LeftParen expression RightParen
    | typeIdOfTheTypeId LeftParen (expression | theTypeId) RightParen
    ;

/*
 add a middle layer to eliminate duplicated function declarations
 */

typeIdOfTheTypeId
    : Typeid_
    ;

expressionList
    : initializerList
    ;

pseudoDestructorName
    : nestedNameSpecifier? (theTypeName Doublecolon)? Tilde theTypeName
    | nestedNameSpecifier Template simpleTemplateId Doublecolon Tilde theTypeName
    | Tilde decltypeSpecifier
    ;

unaryExpression
    : postfixExpression
    | (PlusPlus | MinusMinus | unaryOperator | Sizeof) unaryExpression
    | Sizeof (LeftParen theTypeId RightParen | Ellipsis LeftParen Identifier RightParen)
    | Alignof LeftParen theTypeId RightParen
    | noExceptExpression
    | newExpression_
    | deleteExpression
    ;

unaryOperator
    : Or
    | Star
    | And
    | Plus
    | Tilde
    | Minus
    | Not
    ;

newExpression_
    : Doublecolon? New newPlacement? (newTypeId | LeftParen theTypeId RightParen) newInitializer_?
    ;

newPlacement
    : LeftParen expressionList RightParen
    ;

newTypeId
    : typeSpecifierSeq newDeclarator_?
    ;

newDeclarator_
    : pointerOperator newDeclarator_?
    | noPointerNewDeclarator
    ;

noPointerNewDeclarator
    : LeftBracket expression RightBracket attributeSpecifierSeq?
    | noPointerNewDeclarator LeftBracket constantExpression RightBracket attributeSpecifierSeq?
    ;

newInitializer_
    : LeftParen expressionList? RightParen
    | bracedInitList
    ;

deleteExpression
    : Doublecolon? Delete (LeftBracket RightBracket)? castExpression
    ;

noExceptExpression
    : Noexcept LeftParen expression RightParen
    ;

castExpression
    : unaryExpression
    | LeftParen theTypeId RightParen castExpression
    ;

pointerMemberExpression
    : castExpression ((DotStar | ArrowStar) castExpression)*
    ;

multiplicativeExpression
    : pointerMemberExpression ((Star | Div | Mod) pointerMemberExpression)*
    ;

additiveExpression
    : multiplicativeExpression ((Plus | Minus) multiplicativeExpression)*
    ;

shiftExpression
    : additiveExpression (shiftOperator additiveExpression)*
    ;

shiftOperator
    : Greater Greater
    | Less Less
    ;

relationalExpression
    : shiftExpression ((Less | Greater | LessEqual | GreaterEqual) shiftExpression)*
    ;

equalityExpression
    : relationalExpression ((Equal | NotEqual) relationalExpression)*
    ;

andExpression
    : equalityExpression (And equalityExpression)*
    ;

exclusiveOrExpression
    : andExpression (Caret andExpression)*
    ;

inclusiveOrExpression
    : exclusiveOrExpression (Or exclusiveOrExpression)*
    ;

logicalAndExpression
    : inclusiveOrExpression (AndAnd inclusiveOrExpression)*
    ;

logicalOrExpression
    : logicalAndExpression (OrOr logicalAndExpression)*
    ;

conditionalExpression
    : logicalOrExpression (Question expression Colon assignmentExpression)?
    ;

assignmentExpression
    : conditionalExpression
    | logicalOrExpression assignmentOperator initializerClause
    | throwExpression
    ;

assignmentOperator
    : Assign
    | StarAssign
    | DivAssign
    | ModAssign
    | PlusAssign
    | MinusAssign
    | RightShiftAssign
    | LeftShiftAssign
    | AndAssign
    | XorAssign
    | OrAssign
    ;

expression
    : assignmentExpression (Comma assignmentExpression)*
    ;

constantExpression
    : conditionalExpression
    ;

/*Statements*/

statement
    : labeledStatement
    | declarationStatement
    | attributeSpecifierSeq? (
        expressionStatement
        | compoundStatement
        | selectionStatement
        | iterationStatement
        | jumpStatement
        | tryBlock
    )
    ;

labeledStatement
    : attributeSpecifierSeq? (Identifier | Case constantExpression | Default) Colon statement
    ;

expressionStatement
    : expression? Semi
    ;

compoundStatement
    : LeftBrace statementSeq? RightBrace
    ;

statementSeq
    : statement+
    ;

selectionStatement
    : If LeftParen condition RightParen statement (Else statement)?
    | Switch LeftParen condition RightParen statement
    ;

condition
    : expression
    | attributeSpecifierSeq? declSpecifierSeq declarator (
        Assign initializerClause
        | bracedInitList
    )
    ;

iterationStatement
    : While LeftParen condition RightParen statement
    | Do statement While LeftParen expression RightParen Semi
    | For LeftParen (
        forInitStatement condition? Semi expression?
        | forRangeDeclaration Colon forRangeInitializer
    ) RightParen statement
    ;

forInitStatement
    : expressionStatement
    | simpleDeclaration
    ;

forRangeDeclaration
    : attributeSpecifierSeq? declSpecifierSeq declarator
    ;

forRangeInitializer
    : expression
    | bracedInitList
    ;

jumpStatement
    : (Break | Continue | Return (expression | bracedInitList)? | Goto Identifier) Semi
    ;

declarationStatement
    : blockDeclaration
    ;

/*Declarations*/

declarationseq
    : declaration+
    ;

declaration
    : blockDeclaration
    | functionDefinition
    | templateDeclaration
    | explicitInstantiation
    | explicitSpecialization
    | linkageSpecification
    | namespaceDefinition
    | emptyDeclaration_
    | attributeDeclaration
    ;

blockDeclaration
    : simpleDeclaration
    | asmDefinition
    | namespaceAliasDefinition
    | usingDeclaration
    | usingDirective
    | staticAssertDeclaration
    | aliasDeclaration
    | opaqueEnumDeclaration
    ;

aliasDeclaration
    : Using Identifier attributeSpecifierSeq? Assign theTypeId Semi
    ;

simpleDeclaration
    : declSpecifierSeq? initDeclaratorList? Semi
    | attributeSpecifierSeq declSpecifierSeq? initDeclaratorList Semi
    ;

staticAssertDeclaration
    : Static_assert LeftParen constantExpression Comma StringLiteral RightParen Semi
    ;

emptyDeclaration_
    : Semi
    ;

attributeDeclaration
    : attributeSpecifierSeq Semi
    ;

declSpecifier
    : storageClassSpecifier
    | typeSpecifier
    | functionSpecifier
    | Friend
    | Typedef
    | Constexpr
    ;

declSpecifierSeq
    : declSpecifier+? attributeSpecifierSeq?
    ;

storageClassSpecifier
    : Register
    | Static
    | Thread_local
    | Extern
    | Mutable
    ;

functionSpecifier
    : Inline
    | Virtual
    | Explicit
    ;

typedefName
    : Identifier
    ;

typeSpecifier
    : trailingTypeSpecifier
    | classSpecifier
    | enumSpecifier
    ;

trailingTypeSpecifier
    : simpleTypeSpecifier
    | elaboratedTypeSpecifier
    | typeNameSpecifier
    | cvQualifier
    ;

typeSpecifierSeq
    : typeSpecifier+ attributeSpecifierSeq?
    ;

trailingTypeSpecifierSeq
    : trailingTypeSpecifier+ attributeSpecifierSeq?
    ;

simpleTypeLengthModifier
    : Short
    | Long
    ;

simpleTypeSignednessModifier
    : Unsigned
    | Signed
    ;

simpleTypeSpecifier
    : nestedNameSpecifier? theTypeName
    | nestedNameSpecifier Template simpleTemplateId
    | Char
    | Char16
    | Char32
    | Wchar
    | Bool
    | Short
    | Int
    | Long
    | Float
    | Signed
    | Unsigned
    | Float
    | Double
    | Void
    | Auto
    | decltypeSpecifier
    ;

theTypeName
    : className
    | enumName
    | typedefName
    | simpleTemplateId
    ;

decltypeSpecifier
    : Decltype LeftParen (expression | Auto) RightParen
    ;

elaboratedTypeSpecifier
    : classKey (
        attributeSpecifierSeq? nestedNameSpecifier? Identifier
        | simpleTemplateId
        | nestedNameSpecifier Template? simpleTemplateId
    )
    | Enum nestedNameSpecifier? Identifier
    ;

enumName
    : Identifier
    ;

enumSpecifier
    : enumHead LeftBrace (enumeratorList Comma?)? RightBrace
    ;

enumHead
    : enumkey attributeSpecifierSeq? (nestedNameSpecifier? Identifier)? enumbase?
    ;

opaqueEnumDeclaration
    : enumkey attributeSpecifierSeq? Identifier enumbase? Semi
    ;

enumkey
    : Enum (Class | Struct)?
    ;

enumbase
    : Colon typeSpecifierSeq
    ;

enumeratorList
    : enumeratorDefinition (Comma enumeratorDefinition)*
    ;

enumeratorDefinition
    : enumerator (Assign constantExpression)?
    ;

enumerator
    : Identifier
    ;

namespaceName
    : originalNamespaceName
    | namespaceAlias
    ;

originalNamespaceName
    : Identifier
    ;

namespaceDefinition
    : Inline? Namespace (Identifier | originalNamespaceName)? LeftBrace namespaceBody = declarationseq? RightBrace
    ;

namespaceAlias
    : Identifier
    ;

namespaceAliasDefinition
    : Namespace Identifier Assign qualifiednamespacespecifier Semi
    ;

qualifiednamespacespecifier
    : nestedNameSpecifier? namespaceName
    ;

usingDeclaration
    : Using (Typename_? nestedNameSpecifier | Doublecolon) unqualifiedId Semi
    ;

usingDirective
    : attributeSpecifierSeq? Using Namespace nestedNameSpecifier? namespaceName Semi
    ;

asmDefinition
    : Asm LeftParen StringLiteral RightParen Semi
    ;

linkageSpecification
    : Extern StringLiteral (LeftBrace declarationseq? RightBrace | declaration)
    ;

attributeSpecifierSeq
    : attributeSpecifier+
    ;

attributeSpecifier
    : LeftBracket LeftBracket attributeList? RightBracket RightBracket
    | alignmentspecifier
    ;

alignmentspecifier
    : Alignas LeftParen (theTypeId | constantExpression) Ellipsis? RightParen
    ;

attributeList
    : attribute (Comma attribute)* Ellipsis?
    ;

attribute
    : (attributeNamespace Doublecolon)? Identifier attributeArgumentClause?
    ;

attributeNamespace
    : Identifier
    ;

attributeArgumentClause
    : LeftParen balancedTokenSeq? RightParen
    ;

balancedTokenSeq
    : balancedtoken+
    ;

balancedtoken
    : LeftParen balancedTokenSeq RightParen
    | LeftBracket balancedTokenSeq RightBracket
    | LeftBrace balancedTokenSeq RightBrace
    | ~(LeftParen | RightParen | LeftBrace | RightBrace | LeftBracket | RightBracket)+
    ;

/*Declarators*/

initDeclaratorList
    : initDeclarator (Comma initDeclarator)*
    ;

initDeclarator
    : declarator initializer?
    ;

declarator
    : pointerDeclarator
    | noPointerDeclarator parametersAndQualifiers trailingReturnType
    ;

pointerDeclarator
    : (pointerOperator Const?)* noPointerDeclarator
    ;

noPointerDeclarator
    : declaratorid attributeSpecifierSeq?
    | noPointerDeclarator (
        parametersAndQualifiers
        | LeftBracket constantExpression? RightBracket attributeSpecifierSeq?
    )
    | LeftParen pointerDeclarator RightParen
    ;

parametersAndQualifiers
    : LeftParen parameterDeclarationClause? RightParen cvqualifierseq? refqualifier? exceptionSpecification? attributeSpecifierSeq?
    ;

trailingReturnType
    : Arrow trailingTypeSpecifierSeq abstractDeclarator?
    ;

pointerOperator
    : (And | AndAnd) attributeSpecifierSeq?
    | nestedNameSpecifier? Star attributeSpecifierSeq? cvqualifierseq?
    ;

cvqualifierseq
    : cvQualifier+
    ;

cvQualifier
    : Const
    | Volatile
    ;

refqualifier
    : And
    | AndAnd
    ;

declaratorid
    : Ellipsis? idExpression
    ;

theTypeId
    : typeSpecifierSeq abstractDeclarator?
    ;

abstractDeclarator
    : pointerAbstractDeclarator
    | noPointerAbstractDeclarator? parametersAndQualifiers trailingReturnType
    | abstractPackDeclarator
    ;

pointerAbstractDeclarator
    : pointerOperator* (noPointerAbstractDeclarator | pointerOperator)
    ;

noPointerAbstractDeclarator
    : (parametersAndQualifiers | LeftParen pointerAbstractDeclarator RightParen) (
        parametersAndQualifiers
        | LeftBracket constantExpression? RightBracket attributeSpecifierSeq?
    )*
    ;

abstractPackDeclarator
    : pointerOperator* noPointerAbstractPackDeclarator
    ;

noPointerAbstractPackDeclarator
    : Ellipsis (
        parametersAndQualifiers
        | LeftBracket constantExpression? RightBracket attributeSpecifierSeq?
    )*
    ;

parameterDeclarationClause
    : parameterDeclarationList (Comma? Ellipsis)?
    ;

parameterDeclarationList
    : parameterDeclaration (Comma parameterDeclaration)*
    ;

parameterDeclaration
    : attributeSpecifierSeq? declSpecifierSeq (declarator | abstractDeclarator?) (
        Assign initializerClause
    )?
    ;

functionDefinition
    : attributeSpecifierSeq? declSpecifierSeq? declarator virtualSpecifierSeq? functionBody
    ;

functionBody
    : constructorInitializer? compoundStatement
    | functionTryBlock
    | Assign (Default | Delete) Semi
    ;

initializer
    : braceOrEqualInitializer
    | LeftParen expressionList RightParen
    ;

braceOrEqualInitializer
    : Assign initializerClause
    | bracedInitList
    ;

initializerClause
    : assignmentExpression
    | bracedInitList
    ;

initializerList
    : initializerClause Ellipsis? (Comma initializerClause Ellipsis?)*
    ;

bracedInitList
    : LeftBrace (initializerList Comma?)? RightBrace
    ;

/*Classes*/

className
    : Identifier
    | simpleTemplateId
    ;

classSpecifier
    : classHead LeftBrace memberSpecification? RightBrace
    ;

classHead
    : classKey attributeSpecifierSeq? (classHeadName classVirtSpecifier?)? baseClause?
    | Union attributeSpecifierSeq? ( classHeadName classVirtSpecifier?)?
    ;

classHeadName
    : nestedNameSpecifier? className
    ;

classVirtSpecifier
    : Final
    ;

classKey
    : Class
    | Struct
    ;

memberSpecification
    : (memberdeclaration | accessSpecifier Colon)+
    ;

memberdeclaration
    : attributeSpecifierSeq? declSpecifierSeq? memberDeclaratorList? Semi
    | functionDefinition
    | usingDeclaration
    | staticAssertDeclaration
    | templateDeclaration
    | aliasDeclaration
    | emptyDeclaration_
    ;

memberDeclaratorList
    : memberDeclarator (Comma memberDeclarator)*
    ;

memberDeclarator
    : declarator (
        virtualSpecifierSeq
        | { self.IsPureSpecifierAllowed() }? pureSpecifier
        | { self.IsPureSpecifierAllowed() }? virtualSpecifierSeq pureSpecifier
        | braceOrEqualInitializer
    )
    | declarator
    | Identifier? attributeSpecifierSeq? Colon constantExpression
    ;

virtualSpecifierSeq
    : virtualSpecifier+
    ;

virtualSpecifier
    : Override
    | Final
    ;

/*
 purespecifier: Assign '0'//Conflicts with the lexer ;
 */

pureSpecifier
    : Assign IntegerLiteral
    ;

/*Derived classes*/

baseClause
    : Colon baseSpecifierList
    ;

baseSpecifierList
    : baseSpecifier Ellipsis? (Comma baseSpecifier Ellipsis?)*
    ;

baseSpecifier
    : attributeSpecifierSeq? (
        baseTypeSpecifier
        | Virtual accessSpecifier? baseTypeSpecifier
        | accessSpecifier Virtual? baseTypeSpecifier
    )
    ;

classOrDeclType
    : nestedNameSpecifier? className
    | decltypeSpecifier
    ;

baseTypeSpecifier
    : classOrDeclType
    ;

accessSpecifier
    : Private
    | Protected
    | Public
    ;

/*Special member functions*/

conversionFunctionId
    : Operator conversionTypeId
    ;

conversionTypeId
    : typeSpecifierSeq conversionDeclarator?
    ;

conversionDeclarator
    : pointerOperator conversionDeclarator?
    ;

constructorInitializer
    : Colon memInitializerList
    ;

memInitializerList
    : memInitializer Ellipsis? (Comma memInitializer Ellipsis?)*
    ;

memInitializer
    : meminitializerid (LeftParen expressionList? RightParen | bracedInitList)
    ;

meminitializerid
    : classOrDeclType
    | Identifier
    ;

/*Overloading*/

operatorFunctionId
    : Operator theOperator
    ;

literalOperatorId
    : Operator (StringLiteral Identifier | UserDefinedStringLiteral)
    ;

/*Templates*/

templateDeclaration
    : Template Less templateparameterList Greater declaration
    ;

templateparameterList
    : templateParameter (Comma templateParameter)*
    ;

templateParameter
    : typeParameter
    | parameterDeclaration
    ;

typeParameter
    : ((Template Less templateparameterList Greater)? Class | Typename_) (
        Ellipsis? Identifier?
        | Identifier? Assign theTypeId
    )
    ;

simpleTemplateId
    : templateName Less templateArgumentList? Greater
    ;

templateId
    : simpleTemplateId
    | (operatorFunctionId | literalOperatorId) Less templateArgumentList? Greater
    ;

templateName
    : Identifier
    ;

templateArgumentList
    : templateArgument Ellipsis? (Comma templateArgument Ellipsis?)*
    ;

templateArgument
    : theTypeId
    | constantExpression
    | idExpression
    ;

typeNameSpecifier
    : Typename_ nestedNameSpecifier (Identifier | Template? simpleTemplateId)
    ;

explicitInstantiation
    : Extern? Template declaration
    ;

explicitSpecialization
    : Template Less Greater declaration
    ;

/*Exception handling*/

tryBlock
    : Try compoundStatement handlerSeq
    ;

functionTryBlock
    : Try constructorInitializer? compoundStatement handlerSeq
    ;

handlerSeq
    : handler+
    ;

handler
    : Catch LeftParen exceptionDeclaration RightParen compoundStatement
    ;

exceptionDeclaration
    : attributeSpecifierSeq? typeSpecifierSeq (declarator | abstractDeclarator)?
    | Ellipsis
    ;

throwExpression
    : Throw assignmentExpression?
    ;

exceptionSpecification
    : dynamicExceptionSpecification
    | noeExceptSpecification
    ;

dynamicExceptionSpecification
    : Throw LeftParen typeIdList? RightParen
    ;

typeIdList
    : theTypeId Ellipsis? (Comma theTypeId Ellipsis?)*
    ;

noeExceptSpecification
    : Noexcept LeftParen constantExpression RightParen
    | Noexcept
    ;

/*Preprocessing directives*/

/*Lexer*/

theOperator
    : New (LeftBracket RightBracket)?
    | Delete (LeftBracket RightBracket)?
    | Plus
    | Minus
    | Star
    | Div
    | Mod
    | Caret
    | And
    | Or
    | Tilde
    | Not
    | Assign
    | Greater
    | Less
    | GreaterEqual
    | PlusAssign
    | MinusAssign
    | StarAssign
    | ModAssign
    | XorAssign
    | AndAssign
    | OrAssign
    | Less Less
    | Greater Greater
    | RightShiftAssign
    | LeftShiftAssign
    | Equal
    | NotEqual
    | LessEqual
    | AndAnd
    | OrOr
    | PlusPlus
    | MinusMinus
    | Comma
    | ArrowStar
    | Arrow
    | LeftParen RightParen
    | LeftBracket RightBracket
    ;

literal
    : IntegerLiteral
    | CharacterLiteral
    | FloatingLiteral
    | StringLiteral
    | BooleanLiteral
    | PointerLiteral
    | UserDefinedLiteral
    ;