"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Packaging = exports.DependenciesFile = void 0;
const fs = require("fs");
const path = require("path");
var DependenciesFile;
(function (DependenciesFile) {
    DependenciesFile["PIP"] = "requirements.txt";
    DependenciesFile["POETRY"] = "poetry.lock";
    DependenciesFile["PIPENV"] = "Pipfile.lock";
    DependenciesFile["NONE"] = "";
})(DependenciesFile || (exports.DependenciesFile = DependenciesFile = {}));
class Packaging {
    /**
     * Standard packaging with `pip`.
     */
    static withPip() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIP,
        });
    }
    /**
     * Packaging with `pipenv`.
     */
    static withPipenv() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIPENV,
            // By default, pipenv creates a virtualenv in `/.local`, so we force it to create one in the package directory.
            // At the end, we remove the virtualenv to avoid creating a duplicate copy in the Lambda package.
            exportCommand: `PIPENV_VENV_IN_PROJECT=1 pipenv requirements > ${DependenciesFile.PIP} && rm -rf .venv`,
        });
    }
    /**
     * Packaging with `poetry`.
     */
    static withPoetry(props) {
        return new Packaging({
            dependenciesFile: DependenciesFile.POETRY,
            // Export dependencies with credentials available in the bundling image.
            exportCommand: [
                'poetry', 'export',
                ...props?.poetryIncludeHashes ? [] : ['--without-hashes'],
                ...props?.poetryWithoutUrls ? ['--without-urls'] : [],
                '--with-credentials',
                '--format', DependenciesFile.PIP,
                '--output', DependenciesFile.PIP,
            ].join(' '),
        });
    }
    /**
     * No dependencies or packaging.
     */
    static withNoPackaging() {
        return new Packaging({ dependenciesFile: DependenciesFile.NONE });
    }
    static fromEntry(entry, poetryIncludeHashes, poetryWithoutUrls) {
        if (fs.existsSync(path.join(entry, DependenciesFile.PIPENV))) {
            return this.withPipenv();
        }
        if (fs.existsSync(path.join(entry, DependenciesFile.POETRY))) {
            return this.withPoetry({ poetryIncludeHashes, poetryWithoutUrls });
        }
        else if (fs.existsSync(path.join(entry, DependenciesFile.PIP))) {
            return this.withPip();
        }
        else {
            return this.withNoPackaging();
        }
    }
    constructor(props) {
        this.dependenciesFile = props.dependenciesFile;
        this.exportCommand = props.exportCommand;
    }
}
exports.Packaging = Packaging;
//# sourceMappingURL=data:application/json;base64,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