# commit_gen/cli.py

import argparse
import subprocess
import requests
import sys


def get_staged_diff():
    """
    Get the staged Git diff.
    """
    try:
        diff = subprocess.check_output(["git", "diff", "--cached"], stderr=subprocess.STDOUT)
        return diff.decode("utf-8")
    except subprocess.CalledProcessError:
        print("Error: Unable to fetch git diff (are you in a Git repository?)", file=sys.stderr)
        sys.exit(1)


def generate_commit_message(server_url, diff):
    """
    Send the diff to the commit-generation server and return the commit message.
    """
    payload = {"diff": diff}
    try:
        response = requests.post(server_url, json=payload)
        if response.status_code == 200:
            data = response.json()
            if "commit_message" in data:
                return data["commit_message"]
            else:
                print("Error: Unexpected server response structure.")
                sys.exit(1)
        else:
            print(f"Error: Server returned status code {response.status_code}")
            sys.exit(1)
    except Exception as e:
        print(f"Error: Could not connect to server. Details: {e}")
        sys.exit(1)


def main():
    parser = argparse.ArgumentParser(
        description="Generate a commit message from staged Git diff using a commit-generation server."
    )
    parser.add_argument(
        "--server",
        type=str,
        default="https://pabrh23o3e.execute-api.ap-south-1.amazonaws.com/Prod/generate-commit",
        help="Server URL for commit generation (default: https://pabrh23o3e.execute-api.ap-south-1.amazonaws.com/Prod/generate-commit)",
    )
    args = parser.parse_args()

    diff = get_staged_diff()
    if not diff.strip():
        print("No staged changes found. Please stage your changes (git add) before running this tool.")
        sys.exit(1)

    commit_message = generate_commit_message(args.server, diff)
    print("Generated Commit Message:\n")
    print(commit_message)


if __name__ == "__main__":
    main()