"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const ecs = require("aws-cdk-lib/aws-ecs");
const ec2 = require("aws-cdk-lib/aws-ec2");
const assertions_1 = require("aws-cdk-lib/assertions");
const clusterName = "custom-cluster-name";
const containerName = "custom-container-name";
const serviceName = "custom-service-name";
const familyName = "family-name";
const testCidr = "172.0.0.0/16";
test('Check for construct properties', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const construct = createFargateConstructWithNewResources(stack, publicApi);
    expect(construct.vpc).toBeDefined();
    expect(construct.service).toBeDefined();
    expect(construct.container).toBeDefined();
    expect(construct.stateMachine).toBeDefined();
    expect(construct.stateMachineLogGroup).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
});
test('Check for new service', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: serviceName,
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
});
test('Check for an existing service', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const existingVpc = defaults.getTestVpc(stack);
    const createFargateServiceResponse = defaults.CreateFargateService(stack, 'test', {
        constructVpc: existingVpc,
        clientClusterProps: {
            clusterName
        },
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clientFargateTaskDefinitionProps: {
            family: familyName
        },
        clientContainerDefinitionProps: {
            containerName
        },
        clientFargateServiceProps: {
            serviceName
        }
    });
    new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        existingFargateServiceObject: createFargateServiceResponse.service,
        existingContainerDefinitionObject: createFargateServiceResponse.containerDefinition,
        existingVpc,
        stateMachineProps: testStateMachineProps(stack)
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: serviceName,
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
});
test('Check for IAM startExecution policy', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "states:StartExecution",
                    Effect: "Allow",
                    Resource: {
                        Ref: "testconstructStateMachine3333AAA9"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testconstructtaskdefTaskRoleDefaultPolicyF34A1535",
        Roles: [
            {
                Ref: "testconstructtaskdefTaskRoleC60414C4"
            }
        ]
    });
});
test('Check for public/private VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    createFargateConstructWithNewResources(stack, publicApi);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: testCidr
    });
    template.hasResourceProperties('AWS::EC2::InternetGateway', {});
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::StepFunctions::StateMachine', 1);
    template.resourceCountIs('AWS::ECS::Service', 1);
});
test('Check for isolated VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: testCidr
    });
    defaults.expectNonexistence(stack, 'AWS::EC2::InternetGateway', {});
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::StepFunctions::StateMachine', 1);
    template.resourceCountIs('AWS::ECS::Service', 1);
});
test('Check for an existing VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    const existingVpc = defaults.getTestVpc(stack, publicApi);
    new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        existingVpc,
        stateMachineProps: testStateMachineProps(stack)
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: "172.168.0.0/16"
    });
    template.resourceCountIs("AWS::EC2::VPC", 1);
});
test('Check for custom ARN resource', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const customEnvName = "TEST_CUSTOM_ARN";
    new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
        stateMachineEnvironmentVariableName: customEnvName
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        Family: familyName,
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: customEnvName,
                        Value: {
                            Ref: "testconstructStateMachine3333AAA9"
                        }
                    }
                ],
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: containerName,
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
});
test('Check for no cloudwatch creation', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const construct = new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
        createCloudWatchAlarms: false
    });
    expect(construct.cloudwatchAlarms).not.toBeDefined();
    defaults.expectNonexistence(stack, "AWS::CloudWatch::Alarm", {
        ComparisonOperator: "GreaterThanOrEqualToThreshold",
        EvaluationPeriods: 1,
        AlarmDescription: "Alarm for the number of executions that aborted exceeded the threshold of 1. ",
        Dimensions: [
            {
                Name: "StateMachineArn",
                Value: {
                    Ref: "testconstructStateMachine3333AAA9"
                }
            }
        ],
        MetricName: "ExecutionsAborted",
        Namespace: "AWS/States",
        Period: 300,
        Statistic: "Maximum",
        Threshold: 1
    });
});
test('Check for custom log group props', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const logGroupName = "custom-log-group";
    new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
        logGroupProps: {
            logGroupName
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Logs::LogGroup", {
        LogGroupName: logGroupName
    });
});
test('Confirm that CheckVpcProps was called', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const props = {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
        existingVpc: defaults.getTestVpc(stack),
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
    };
    const app = () => {
        new lib_1.FargateToStepfunctions(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
function createFargateConstructWithNewResources(stack, publicApi) {
    return new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
    });
}
function testStateMachineProps(stack, userProps) {
    const defaultTestProp = {
        definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'farstp-test')
    };
    return defaults.consolidateProps(defaultTestProp, userProps);
}
test('Deploy 2 constructs', () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    new lib_1.FargateToStepfunctions(stack, 'test-new-fargate-stepfunctions', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingVpc,
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'test'),
        },
    });
    new lib_1.FargateToStepfunctions(stack, 'test-second-fargate-stepfunctions', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingVpc,
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'secondtest'),
        },
    });
    assertions_1.Template.fromStack(stack);
    // No checks, as our main concern is this has no collisions
});
//# sourceMappingURL=data:application/json;base64,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