import { Role } from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
import { TrackedConstruct } from '../common/tracked-construct';
import { EmrEksCluster } from '../emr-eks-platform';
import { NotebookUserOptions } from './notebook-user';
/**
 * The properties for NotebookPlatform Construct.
 */
export interface NotebookPlatformProps {
    /**
     * Required
     * the EmrEks infrastructure used for the deployment
     * */
    readonly emrEks: EmrEksCluster;
    /**
     * Required the name to be given to the Amazon EMR Studio
     * Must be unique across the AWS account
     * */
    readonly studioName: string;
    /**
     * Required the authentication mode of Amazon EMR Studio
     * Either 'SSO' or 'IAM' defined in the Enum {@link studioAuthMode}
     * */
    readonly studioAuthMode: StudioAuthMode;
    /**
     * the namespace where to deploy the EMR Virtual Cluster
     * @default - Use the {@link EmrVirtualClusterOptions} default namespace
     * */
    readonly eksNamespace?: string;
    /**
     * Used when IAM Authentication is selected with IAM federation with an external identity provider (IdP) for Amazon EMR Studio
     * This is the URL used to sign in the AWS console
     * */
    readonly idpAuthUrl?: string;
    /**
     * Used when IAM Authentication is selected with IAM federation with an external identity provider (IdP) for Amazon EMR Studio
     * Value can be set with {@link IdpRelayState} Enum or through a value provided by the user
     * */
    readonly idpRelayStateParameterName?: string;
    /**
     * Used when IAM Authentication is selected with IAM federation with an external identity provider (IdP) for Amazon EMR Studio
     * Value taken from the IAM console in the Identity providers console
     * */
    readonly idpArn?: string;
}
/**
 * Enum to define authentication mode for Amazon EMR Studio
 */
export declare enum StudioAuthMode {
    IAM = "IAM",
    SSO = "SSO"
}
/**
 * Enum to define the type of identity Type in EMR studio
 */
export declare enum SSOIdentityType {
    USER = "USER",
    GROUP = "GROUP"
}
/**
 * A CDK construct to create a notebook infrastructure based on Amazon EMR Studio and assign users to it
 *
 * This construct is initialized through a constructor that takes as argument an interface defined in {@link NotebookPlatformProps}
 * The construct has a method to add users {@link addUser} the method take as argument {@link NotebookUserOptions}
 *
 * Resources deployed:
 *
 * * An S3 Bucket used by EMR Studio to store the Jupyter notebooks
 * * A KMS encryption Key used to encrypt an S3 bucket used by EMR Studio to store jupyter notebooks
 * * An EMR Studio service Role as defined here, and allowed to access the S3 bucket and KMS key created above
 * * An EMR Studio User Role as defined here - The policy template which is leveraged is the Basic one from the Amazon EMR Studio documentation
 * * Multiple EMR on EKS Managed Endpoints, each for a user or a group of users
 * * An execution role to be passed to the Managed endpoint from a policy provided by the user
 * * Multiple Session Policies that are used to map an EMR Studio user or group to a set of resources they are allowed to access. These resources are:
 *   * EMR Virtual Cluster - created above
 *   * ManagedEndpoint
 *
 *
 * Usage example:
 *
 * ```typescript
 * const emrEks = EmrEksCluster.getOrCreate(stack, {
 *   eksAdminRoleArn: 'arn:aws:iam::012345678912:role/Admin-Admin',
 *   eksClusterName: 'cluster',
 * });
 *
 * const notebookPlatform = new NotebookPlatform(stack, 'platform-notebook', {
 *   emrEks: emrEks,
 *   eksNamespace: 'platformns',
 *   studioName: 'platform',
 *   studioAuthMode: StudioAuthMode.SSO,
 * });
 *
 * // If the S3 bucket is encrypted, add policy to the key for the role
 * const policy1 = new ManagedPolicy(stack, 'MyPolicy1', {
 *   statements: [
 *     new PolicyStatement({
 *       resources: <BUCKET ARN(s)>,
 *       actions: ['s3:*'],
 *     }),
 *     new PolicyStatement({
 *       resources: [
 *         stack.formatArn({
 *           account: Aws.ACCOUNT_ID,
 *           region: Aws.REGION,
 *           service: 'logs',
 *           resource: '*',
 *           arnFormat: ArnFormat.NO_RESOURCE_NAME,
 *         }),
 *       ],
 *       actions: [
 *         'logs:*',
 *       ],
 *     }),
 *   ],
 * });
 *
 * notebookPlatform.addUser([{
 *   identityName: 'user1',
 *   identityType: SSOIdentityType.USER,
 *   notebookManagedEndpoints: [{
 *     emrOnEksVersion: EmrVersion.V6_9,
 *     executionPolicy: policy1,
 *   }],
 * }]);
 *
 * ```
 */
export declare class NotebookPlatform extends TrackedConstruct {
    private static readonly DEFAULT_EMR_VERSION;
    private static readonly STUDIO_PRINCIPAL;
    private readonly studioId;
    private readonly workSpaceSecurityGroup;
    private readonly engineSecurityGroup;
    private readonly workspacesBucket;
    private readonly studioUserRole?;
    private readonly studioServicePolicy;
    private readonly studioUserPolicy;
    private readonly studioSubnetList;
    private readonly studioInstance;
    private readonly studioName;
    private readonly emrEks;
    private readonly emrVirtCluster;
    private readonly emrVirtualClusterName;
    private readonly notebookPlatformEncryptionKey;
    private readonly managedEndpointExecutionPolicyArnMapping;
    private readonly federatedIdPARN;
    private readonly authMode;
    private studioServiceRole;
    private vcNamespace;
    /**
     * @public
     * Constructs a new instance of the DataPlatform class
     * @param {Construct} scope the Scope of the AWS CDK Construct
     * @param {string} id the ID of the AWS CDK Construct
     * @param {NotebookPlatformProps} props the DataPlatformNotebooks [properties]{@link NotebookPlatformProps}
     */
    constructor(scope: Construct, id: string, props: NotebookPlatformProps);
    /**
     * @public
     * Method to add users, take a list of userDefinition and will create a managed endpoints for each user
     * and create an IAM Policy scoped to the list managed endpoints
     * @param {NotebookUserOptions} userList list of users
     */
    addUser(userList: NotebookUserOptions[]): Role[];
}
