"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LakeFormationS3Location = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const lakeformation = require("aws-cdk-lib/aws-lakeformation");
const constructs_1 = require("constructs");
/**
 * This CDK construct aims to register an S3 Location for Lakeformation with Read and Write access.
 * If the location is in a different account, cross account access should be granted via the [S3CrossAccount]{@link S3CrossAccount} construct.
 * If the S3 location is encrypted with KMS, the key must be explicitly passed to the construct because CDK cannot retrieve bucket encryption key from imported buckets.
 * Imported buckets are generally used in cross account setup like data mesh.
 *
 * This construct instantiate 2 objects:
 * * An IAM role with read/write permissions to the S3 location and encrypt/decrypt access to the KMS key used to encypt the bucket
 * * A CfnResource is based on an IAM role with 2 policy statement folowing the least privilege AWS best practices:
 *   * Statement 1 for S3 permissions
 *   * Statement 2 for KMS permissions if the bucket is encrypted
 *
 * The CDK construct instantiate the CfnResource in order to register the S3 location with Lakeformation using the IAM role defined above.
 *
 * Usage example:
 * ```typescript
 * import * as cdk from 'aws-cdk-lib';
 * import { LakeformationS3Location } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new cdk.App();
 * const stack = new cdk.Stack(exampleApp, 'LakeformationS3LocationStack');
 *
 * const myKey = new Key(stack, 'MyKey')
 * const myBucket = new Bucket(stack, 'MyBucket', {
 *   encryptionKey: myKey,
 * })
 *
 * new LakeFormationS3Location(stack, 'MyLakeformationS3Location', {
 *   s3Location: {
 *     bucketName: myBucket.bucketName,
 *     objectKey: 'my-prefix',
 *   },
 *   kmsKeyId: myBucket.encryptionKey.keyId,
 * });
 * ```
 */
class LakeFormationS3Location extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Create an Amazon IAM Role used by Lakeformation to register S3 location
        this.dataAccessRole = new aws_iam_1.Role(this, 'LFS3AccessRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lakeformation.amazonaws.com'),
        });
        // add policy to access S3 for Read and Write
        this.dataAccessRole.addToPolicy(new aws_iam_1.PolicyStatement({
            resources: [
                aws_cdk_lib_1.Arn.format({
                    account: '',
                    region: '',
                    resource: props.s3Location.bucketName,
                    service: 's3',
                    resourceName: props.s3Location.objectKey ? props.s3Location.objectKey + '/*' : '*',
                }, aws_cdk_lib_1.Stack.of(this)),
                aws_cdk_lib_1.Arn.format({
                    account: '',
                    region: '',
                    resource: props.s3Location.bucketName,
                    service: 's3',
                }, aws_cdk_lib_1.Stack.of(this)),
            ],
            actions: [
                "s3:GetObject*",
                "s3:GetBucket*",
                "s3:List*",
                "s3:DeleteObject*",
                "s3:PutObject",
                "s3:PutObjectLegalHold",
                "s3:PutObjectRetention",
                "s3:PutObjectTagging",
                "s3:PutObjectVersionTagging",
                "s3:Abort*",
            ],
        }));
        // add policy to access KMS key used for the bucket encryption
        if (props.kmsKeyId) {
            this.dataAccessRole.addToPolicy(new aws_iam_1.PolicyStatement({
                resources: [
                    aws_cdk_lib_1.Arn.format({
                        region: aws_cdk_lib_1.Aws.REGION,
                        resource: 'key',
                        service: 'kms',
                        account: props.accountId ? props.accountId : aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        resourceName: props.kmsKeyId,
                    }, aws_cdk_lib_1.Stack.of(this))
                ],
                actions: [
                    'kms:Encrypt*',
                    'kms:Decrypt*',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:Describe*',
                ],
            }));
        }
        new lakeformation.CfnResource(this, 'RegisterLocation', {
            resourceArn: aws_cdk_lib_1.Arn.format({
                account: '',
                region: '',
                resource: props.s3Location.bucketName,
                service: 's3',
                resourceName: props.s3Location.objectKey ? props.s3Location.objectKey + '/*' : '*',
            }, aws_cdk_lib_1.Stack.of(this)),
            useServiceLinkedRole: false,
            roleArn: this.dataAccessRole.roleArn,
        });
    }
}
exports.LakeFormationS3Location = LakeFormationS3Location;
_a = JSII_RTTI_SYMBOL_1;
LakeFormationS3Location[_a] = { fqn: "aws-analytics-reference-architecture.LakeFormationS3Location", version: "2.9.7" };
//# sourceMappingURL=data:application/json;base64,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